/*
 * Copyright (C) 2016, 2017 Mentor Graphics Development (Deutschland) GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "windowmanager.hpp"


//////////////////////////////////////////
// THIS IS STILL UNDER HEAVY DEVELOPMENT!
// DO NOT JUDGE THE SOURCE CODE :)
//////////////////////////////////////////

// three layers will be defined. The HomeScreen will be placed
// full screen in the background.
// On top all applications in one layer.
// On top of that, the popup layer.
#define WINDOWMANAGER_LAYER_POPUP 100
#define WINDOWMANAGER_LAYER_HOMESCREEN_OVERLAY 101
#define WINDOWMANAGER_LAYER_APPLICATIONS 102
#define WINDOWMANAGER_LAYER_HOMESCREEN 103

#define WINDOWMANAGER_LAYER_NUM 4

#define WINDOWMANAGER_SURFACE_ID_SHIFT 22

// the HomeScreen app has to have the surface id 4194304
#define WINDOWMANAGER_HOMESCREEN_MAIN_SURFACE_ID (1 << WINDOWMANAGER_SURFACE_ID_SHIFT)

// Quick hack for scaling layer to fit non-FHD(1920x1080) screen
//  * source rect of layer should be 1920x1080
//  * destination rect of layer should fit physical display resolution
//  * source rect of surface shoud be based on 1920x1080
//  * destination rect of surface should be based on 1920x1080
#define WINDOWMANAGER_HOMESCREEN_WIDTH  1080
#define WINDOWMANAGER_HOMESCREEN_HEIGHT 1920

void* WindowManager::myThis = 0;

WindowManager::WindowManager(int displayId, QObject *parent) :
    QObject(parent),
    m_layouts(),
    mp_layoutAreaToSurfaceIdAssignment(0),
    m_currentLayout(-1),
    m_screenId(displayId),
    m_screen()
#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    ,
    m_layerSrc(0, 0, WINDOWMANAGER_HOMESCREEN_WIDTH, WINDOWMANAGER_HOMESCREEN_HEIGHT),
    m_layerDest(),
    m_appSurfaces(),
    m_appLayers(),
#if !defined(NO_PROCESS_GROUP) || !defined(NO_PROCESS_SESSION)
    m_appPid2surfPid(),
#endif
    m_keepApps(),
    m_bgApps(),
    m_pending_to_show(-1)
#endif
{
#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    m_showLayers = new t_ilm_layer[WINDOWMANAGER_LAYER_NUM];

    m_showLayers[0] = 0; /* POPUP is not shown by default */
    m_showLayers[1] = 0; /* HOMESCREEN_OVERLAY is not shown by default */
    m_showLayers[2] = 0; /* APPLICATIONS is not shown by default */
    m_showLayers[3] = WINDOWMANAGER_LAYER_HOMESCREEN; /* HOMESCREEN is shwon by default */

    m_keepApps.append(QString("mediaplayer@"));
    m_keepApps.append(QString("navigation@"));
#endif
    qDebug("-=[WindowManager]=-");
}

void WindowManager::start()
{
    qDebug("-=[start]=-");
    mp_layoutAreaToSurfaceIdAssignment = new QMap<int, unsigned int>;
#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    ilmErrorTypes err;

    err = ilm_init();
    qDebug("ilm_init = %d", err);
    if(ILM_SUCCESS != err)
    {
        qDebug("failed! Exiting!");
        exit(-1);
    }

    myThis = this;

    t_ilm_uint width, height;
    ilm_getScreenResolution(m_screenId, &width, &height);
    m_screen.setWidth(width);
    m_screen.setHeight(height);

    const char *mode = secure_getenv("CONTENT_MODE");
    if (!mode) {
        qDebug("CONTENT_MODE not specified.  Falling back to aspect-fit");
        mode = "aspect-fit";
    }

    setRootGeometry(mode);

    createNewLayer(WINDOWMANAGER_LAYER_POPUP);
    createNewLayer(WINDOWMANAGER_LAYER_HOMESCREEN_OVERLAY);
    createNewLayer(WINDOWMANAGER_LAYER_HOMESCREEN);

    ilm_registerNotification(WindowManager::notificationFunc_static, this);
#endif

    QDBusConnection dbus = QDBusConnection::sessionBus();
    dbus.registerObject("/windowmanager", this);
    dbus.registerService("org.agl.windowmanager");

    // publish windowmanager interface
    mp_windowManagerAdaptor = new WindowmanagerAdaptor((QObject*)this);
}

WindowManager::~WindowManager()
{
    qDebug("-=[~WindowManager]=-");
    delete mp_windowManagerAdaptor;
#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    ilm_destroy();
#endif
    delete mp_layoutAreaToSurfaceIdAssignment;
}

#ifdef HAVE_IVI_LAYERMANAGEMENT_API
t_ilm_layer* WindowManager::getLayerRenderOrder(int& num_layers)
{
    int i, j, n_bg = m_bgApps.size();

    t_ilm_layer* id_array = new t_ilm_layer[n_bg + WINDOWMANAGER_LAYER_NUM];

    qDebug("Layer render order (ivi-layer-id), %d bgApps: ", n_bg);

    num_layers = 0;
    for (i = 0; i < n_bg; i++) {
        if (m_bgApps[i] != 0) {
            QMap<pid_t, t_ilm_layer>::const_iterator i_layers;
            QMap<pid_t, pid_t>::const_iterator i_surf_pids;

            i_surf_pids = m_appPid2surfPid.find(m_bgApps[i]);
            if (i_surf_pids != m_appPid2surfPid.end()) {
                i_layers = m_appLayers.find(i_surf_pids.value());
                /* m_showLayers[2] means layer for apps */
                if (i_layers != m_appLayers.end() && i_layers.value() != 0
                    && i_layers.value() != m_showLayers[2]) {
                    qDebug("  m_bgApps[%d]=%d", i, i_layers.value());
                    id_array[num_layers++] = i_layers.value();
                }
            }
        }
    }

    for (i = WINDOWMANAGER_LAYER_NUM - 1, j = 0; i >= 0; i--,j++) {
        if (m_showLayers[i] != 0) {
            qDebug("  m_showLayers[%d]=%d", i, m_showLayers[i]);
            id_array[num_layers++] = m_showLayers[i];
        }
    }

    return id_array;
}

void WindowManager::renderLayers()
{
    int num_layers;
    t_ilm_layer* renderOrder = getLayerRenderOrder(num_layers);

    qDebug("Screen render order %d, %d layers", m_screenId, num_layers);
    ilm_displaySetRenderOrder(m_screenId, renderOrder, num_layers);
    ilm_commitChanges();
}
#endif

void WindowManager::dumpScene()
{
    qDebug("\n");
    qDebug("current layout   : %d", m_currentLayout);
    qDebug("available layouts: %d", m_layouts.size());
    QList<Layout>::const_iterator i = m_layouts.begin();

    while (i != m_layouts.constEnd())
    {
        qDebug("--[id: %d]--[%s]--", i->id, i->name.toStdString().c_str());
        qDebug("  %d surface areas", i->layoutAreas.size());
        for (int j = 0; j < i->layoutAreas.size(); ++j)
        {
            qDebug("  -area %d", j);
            qDebug("    -x     : %d", i->layoutAreas.at(j).x);
            qDebug("    -y     : %d", i->layoutAreas.at(j).y);
            qDebug("    -width : %d", i->layoutAreas.at(j).width);
            qDebug("    -height: %d", i->layoutAreas.at(j).height);
        }

        ++i;
    }
}

#ifdef HAVE_IVI_LAYERMANAGEMENT_API

void WindowManager::createNewLayer(int layerId)
{
    qDebug("-=[createNewLayer]=-");
    qDebug("  layerId %d", layerId);

    t_ilm_layer newLayerId = layerId;

    // Currently the dimension of layer for apps inherits the root layer's.
    ilm_layerCreateWithDimension(&newLayerId,
                                 m_layerSrc.width(),
                                 m_layerSrc.height());
    ilm_commitChanges();
    ilm_layerSetSourceRectangle(newLayerId,
                                m_layerSrc.x(),
                                m_layerSrc.y(),
                                m_layerSrc.width(),
                                m_layerSrc.height());

    ilm_layerSetDestinationRectangle(newLayerId,
                                     m_layerDest.x(),
                                     m_layerDest.y(),
                                     m_layerDest.width(),
                                     m_layerDest.height());
    ilm_commitChanges();
    ilm_layerSetOpacity(newLayerId, 1.0);
    ilm_layerSetVisibility(newLayerId, ILM_TRUE);
    ilm_commitChanges();
}


void WindowManager::setRootGeometry(const char *mode)
{
    // To calculate layer management parameters

    // Calcurate parameters based on the content mode id given
    // May return false if parameter check, etc. fails (currently not done).

    if (!strcmp("scale-to-fill", mode)) {
        // scale, ignore aspect ratio
        m_layerDest.setRect(0, 0, m_screen.width(), m_screen.height());
    }
    else if (!strcmp("aspect-fit", mode) || !strcmp("aspect-fill", mode)) {
        // scale, keeping aspect ratio
        QSize size = m_layerSrc.size();

        if (!strcmp("aspect-fit", mode)) {
            size.scale(m_screen, Qt::KeepAspectRatio);
        } else {
            size.scale(m_screen, Qt::KeepAspectRatioByExpanding);
        }

        m_layerDest.setSize(size);
        m_layerDest.moveCenter(QPoint(m_screen.width() / 2, m_screen.height() / 2));
    }
    else {
        // other modes without scaling
        m_layerDest.setSize(m_layerSrc.size());

        // At first, move to center, then move to proper position
        m_layerDest.moveCenter(QPoint(m_screen.width() / 2, m_screen.height() / 2));

        if (!strcmp("center", mode)) {
            ;
        }
        else if (!strcmp("top", mode)) {
            m_layerDest.moveTop(0);
        }
        else if (!strcmp("bottom", mode)) {
            m_layerDest.moveBottom(m_screen.height());
        }
        else if (!strcmp("left", mode)) {
            m_layerDest.moveLeft(0);
        }
        else if (!strcmp("right", mode)) {
            m_layerDest.moveRight(m_screen.width());
        }
        else if (!strcmp("top-left", mode)) {
            m_layerDest.moveTo(0,0);
        }
        else if (!strcmp("top-right", mode)) {
            m_layerDest.moveTopRight(QPoint(m_screen.width(), 0));
        }
        else if (!strcmp("bottom-left", mode)) {
            m_layerDest.moveBottomLeft(QPoint(0, m_screen.height()));
        }
        else if (!strcmp("bottom-right", mode)) {
            m_layerDest.moveBottomRight(QPoint(m_screen.width(), m_screen.height()));
        }
        else {
            // fallback
            qDebug("Mode \"%s\" not known, fallingback to aspect-fit", mode);
            setRootGeometry("aspect-fit");
            return;
        }
    }

    qDebug("layer src=%dx%d dest=(%d,%d) %dx%d",
           m_layerSrc.width(), m_layerSrc.height(),
           m_layerDest.x(), m_layerDest.y(), m_layerDest.width(), m_layerDest.height());
}

t_ilm_layer WindowManager::getAppLayerID(pid_t pid)
{
    t_ilm_layer layer_id;

//    layer_id = pid + (WINDOWMANAGER_LAYER_APPLICATIONS << WINDOWMANAGER_LAYER_ID_SHIFT);
    layer_id = pid + (WINDOWMANAGER_LAYER_APPLICATIONS * 100000); /* for debug */

    return layer_id;
}


void WindowManager::addSurface(t_ilm_surface surfaceId)
{
    struct ilmSurfaceProperties surfaceProperties;
    pid_t surf_pid;

    ilm_getPropertiesOfSurface(surfaceId, &surfaceProperties);
    surf_pid = surfaceProperties.creatorPid;

    QMap<pid_t, t_ilm_surface>::const_iterator i = m_appSurfaces.find(surf_pid);
    if (i == m_appSurfaces.end() || i.value() == 0) {
        /* Only the 1st surface is handled by Window Manager */
        qDebug("This surface (%d) is 1st one for app (%d)", surfaceId, surf_pid);
        /* update surface id */
        m_appSurfaces.insert(surf_pid, surfaceId);

#if !defined(NO_PROCESS_GROUP) || !defined(NO_PROCESS_SESSION)
        registerAppPid2surfPid(surf_pid);
#endif
        /* this surface should be handled by WindowManager */
        ilm_surfaceAddNotification(surfaceId, surfaceCallbackFunction_static);
        ilm_commitChanges();
    }
}

t_ilm_layer WindowManager::addSurfaceToAppLayer(pid_t pid, int surfaceId)
{
    t_ilm_layer layer_id;

    qDebug("-=[addSurfaceToAppLayer]=-");
    qDebug("  surfaceId %d", surfaceId);

    if (pid < 0)
        return 0;

    QMap<pid_t, t_ilm_layer>::const_iterator i = m_appLayers.find(pid);
    if (i == m_appLayers.end()) {
        qDebug("No layer found, create new for app(pid=%d)", pid);

        /* not found, create new one */
        layer_id = getAppLayerID(pid);

        createNewLayer(layer_id);
        m_appLayers.insert(pid, layer_id);
    } else {
        layer_id = i.value();
    }

    return layer_id;
}

void WindowManager::addSurfaceToLayer(int surfaceId, int layerId)
{
    qDebug("-=[addSurfaceToLayer]=-");
    qDebug("  surfaceId %d", surfaceId);
    qDebug("  layerId %d", layerId);

    if (layerId == WINDOWMANAGER_LAYER_HOMESCREEN)
    {
      ilm_layerAddSurface(layerId, surfaceId);
    }
    else if (layerId == WINDOWMANAGER_LAYER_HOMESCREEN_OVERLAY)
    {
        struct ilmSurfaceProperties surfaceProperties;
        ilm_getPropertiesOfSurface(surfaceId, &surfaceProperties);

        //ilm_surfaceSetDestinationRectangle(surfaceId, 0, 0, surfaceProperties.origSourceWidth, surfaceProperties.origSourceHeight);
        //ilm_surfaceSetSourceRectangle(surfaceId, 0, 0, surfaceProperties.origSourceWidth, surfaceProperties.origSourceHeight);
        //ilm_surfaceSetOpacity(surfaceId, 0.5);
        //ilm_surfaceSetVisibility(surfaceId, ILM_TRUE);

        ilm_layerAddSurface(layerId, surfaceId);
    }
    else if (layerId == WINDOWMANAGER_LAYER_POPUP)
    {
        struct ilmSurfaceProperties surfaceProperties;
        ilm_getPropertiesOfSurface(surfaceId, &surfaceProperties);

        //ilm_surfaceSetDestinationRectangle(surfaceId, 0, 0, surfaceProperties.origSourceWidth, surfaceProperties.origSourceHeight);
        //ilm_surfaceSetSourceRectangle(surfaceId, 0, 0, surfaceProperties.origSourceWidth, surfaceProperties.origSourceHeight);
        //ilm_surfaceSetOpacity(surfaceId, 0.0);
        //ilm_surfaceSetVisibility(surfaceId, ILM_FALSE);

        ilm_layerAddSurface(layerId, surfaceId);
    } else {
        return;
    }

    ilm_commitChanges();
}

void WindowManager::configureHomeScreenMainSurface(t_ilm_surface surface, t_ilm_uint width, t_ilm_uint height)
{
    // homescreen app always fullscreen in the back
    ilm_surfaceSetDestinationRectangle(surface, 0, 0,
                                       WINDOWMANAGER_HOMESCREEN_WIDTH,
                                       WINDOWMANAGER_HOMESCREEN_HEIGHT);
    ilm_surfaceSetSourceRectangle(surface, 0, 0, width, height);
    ilm_surfaceSetOpacity(surface, 1.0);
    ilm_surfaceSetVisibility(surface, ILM_TRUE);

    ilm_commitChanges();
}

void WindowManager::configureAppSurface(t_ilm_surface surface, t_ilm_uint width, t_ilm_uint height)
{
    /* Dirty hack! cut & paste from HomeScreen/src/layouthandler.cpp */
    const int SCREEN_WIDTH = 1080;
    const int SCREEN_HEIGHT = 1920;

    const int TOPAREA_HEIGHT = 218;
    const int TOPAREA_WIDTH = SCREEN_WIDTH;
    const int TOPAREA_X = 0;
    const int TOPAREA_Y = 0;
    const int MEDIAAREA_HEIGHT = 215;
    const int MEDIAAREA_Y = SCREEN_HEIGHT - MEDIAAREA_HEIGHT;

    const t_ilm_int APPAREA_X = TOPAREA_X;
    const t_ilm_int APPAREA_Y = TOPAREA_Y + TOPAREA_HEIGHT;
    const t_ilm_uint APPAREA_WIDTH = TOPAREA_WIDTH;
    const t_ilm_uint APPAREA_HEIGHT = MEDIAAREA_Y - APPAREA_Y;

    ilm_surfaceSetDestinationRectangle(surface, APPAREA_X, APPAREA_Y,
                                       APPAREA_WIDTH, APPAREA_HEIGHT);
    ilm_surfaceSetSourceRectangle(surface, 0, 0, width, height);
    ilm_surfaceSetOpacity(surface, 1.0);
    ilm_surfaceSetVisibility(surface, ILM_TRUE); /* Hack to avoid blank screen when switch apps */

    ilm_commitChanges();
}
#endif

void WindowManager::updateScreen()
{
    qDebug("-=[updateScreen]=-");

#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    if (m_pending_to_show != -1) {
        qDebug("show pending app (%d)", m_pending_to_show);
        showAppLayer(m_pending_to_show);
    } else {
        // display layer render order
        renderLayers();
    }
#endif
}

#ifdef HAVE_IVI_LAYERMANAGEMENT_API
void WindowManager::notificationFunc_non_static(ilmObjectType object,
                                    t_ilm_uint id,
                                    t_ilm_bool created)
{
    qDebug("-=[notificationFunc_non_static]=-");
    qDebug("Notification from weston!");
    if (ILM_SURFACE == object)
    {
        if (created)
        {
            if (WINDOWMANAGER_HOMESCREEN_MAIN_SURFACE_ID == id)
            {
                ilm_surfaceAddNotification(id, surfaceCallbackFunction_static);
                ilm_commitChanges();
            }
            else
            {
                addSurface(id);
            }
        }
        else
        {
            qDebug("Surface destroyed, ID: %d", id);
#if 0
            m_appSurfaces.removeAt(m_appSurfaces.indexOf(id));
            ilm_surfaceRemoveNotification(id);

            ilm_commitChanges();
#endif
        }
    }
    if (ILM_LAYER == object)
    {
        //qDebug("Layer.. we don't care...");
    }
}

void WindowManager::notificationFunc_static(ilmObjectType object,
                                            t_ilm_uint id,
                                            t_ilm_bool created,
                                            void*)
{
    static_cast<WindowManager*>(WindowManager::myThis)->notificationFunc_non_static(object, id, created);
}

void WindowManager::surfaceCallbackFunction_non_static(t_ilm_surface surface,
                                    struct ilmSurfaceProperties* surfaceProperties,
                                    t_ilm_notification_mask mask)
{
    pid_t pid = surfaceProperties->creatorPid;

    qDebug("-=[surfaceCallbackFunction_non_static]=-");
    qDebug("surfaceCallbackFunction_non_static changes for surface %d", surface);
    if (ILM_NOTIFICATION_VISIBILITY & mask)
    {
        qDebug("ILM_NOTIFICATION_VISIBILITY");
        surfaceVisibilityChanged(surface, surfaceProperties->visibility);
        updateScreen();
    }
    if (ILM_NOTIFICATION_OPACITY & mask)
    {
        qDebug("ILM_NOTIFICATION_OPACITY");
    }
    if (ILM_NOTIFICATION_ORIENTATION & mask)
    {
        qDebug("ILM_NOTIFICATION_ORIENTATION");
    }
    if (ILM_NOTIFICATION_SOURCE_RECT & mask)
    {
        qDebug("ILM_NOTIFICATION_SOURCE_RECT");
    }
    if (ILM_NOTIFICATION_DEST_RECT & mask)
    {
        qDebug("ILM_NOTIFICATION_DEST_RECT");
    }
    if (ILM_NOTIFICATION_CONTENT_AVAILABLE & mask)
    {
        qDebug("ILM_NOTIFICATION_CONTENT_AVAILABLE");
    }
    if (ILM_NOTIFICATION_CONTENT_REMOVED & mask)
    {
        qDebug("ILM_NOTIFICATION_CONTENT_REMOVED");

        /* application being down */
        m_appLayers.remove(pid);
    }
    if (ILM_NOTIFICATION_CONFIGURED & mask)
    {
        qDebug("ILM_NOTIFICATION_CONFIGURED");
        qDebug("  surfaceProperties %d", surface);
        qDebug("    surfaceProperties.origSourceWidth: %d", surfaceProperties->origSourceWidth);
        qDebug("    surfaceProperties.origSourceHeight: %d", surfaceProperties->origSourceHeight);

        if (surface == WINDOWMANAGER_HOMESCREEN_MAIN_SURFACE_ID) {
            addSurfaceToLayer(surface, WINDOWMANAGER_LAYER_HOMESCREEN);
            configureHomeScreenMainSurface(surface, surfaceProperties->origSourceWidth, surfaceProperties->origSourceHeight);
        } else {
            ilmErrorTypes result;
            t_ilm_layer layer = addSurfaceToAppLayer(pid, surface);

            if (layer != 0) {
                configureAppSurface(surface,
                                    surfaceProperties->origSourceWidth,
                                    surfaceProperties->origSourceHeight);

                result = ilm_layerAddSurface(layer, surface);
                if (result != ILM_SUCCESS) {
                    qDebug("ilm_layerAddSurface(%d,%d) failed.", layer, surface);
                }
                ilm_commitChanges();
            }
        }
        updateScreen();
    }
}

void WindowManager::surfaceCallbackFunction_static(t_ilm_surface surface,
                                    struct ilmSurfaceProperties* surfaceProperties,
                                    t_ilm_notification_mask mask)

{
    static_cast<WindowManager*>(WindowManager::myThis)->surfaceCallbackFunction_non_static(surface, surfaceProperties, mask);
}
#endif

int WindowManager::layoutId() const
{
    return m_currentLayout;
}

QString WindowManager::layoutName() const
{
    QList<Layout>::const_iterator i = m_layouts.begin();

    QString result = "not found";
    while (i != m_layouts.constEnd())
    {
        if (i->id == m_currentLayout)
        {
            result = i->name;
        }

        ++i;
    }

    return result;
}


int WindowManager::addLayout(int layoutId, const QString &layoutName, const QList<LayoutArea> &surfaceAreas)
{
    qDebug("-=[addLayout]=-");
    m_layouts.append(Layout(layoutId, layoutName, surfaceAreas));

    qDebug("addLayout %d %s, size %d",
           layoutId,
           layoutName.toStdString().c_str(),
           surfaceAreas.size());

    dumpScene();

    return WINDOWMANAGER_NO_ERROR;
}

int WindowManager::deleteLayoutById(int layoutId)
{
    qDebug("-=[deleteLayoutById]=-");
    qDebug("layoutId: %d", layoutId);
    int result = WINDOWMANAGER_NO_ERROR;

    if (m_currentLayout == layoutId)
    {
        result = WINDOWMANAGER_ERROR_ID_IN_USE;
    }
    else
    {
        QList<Layout>::iterator i = m_layouts.begin();
        result = WINDOWMANAGER_ERROR_ID_IN_USE;
        while (i != m_layouts.constEnd())
        {
            if (i->id == layoutId)
            {
                m_layouts.erase(i);
                result = WINDOWMANAGER_NO_ERROR;
                break;
            }

            ++i;
        }
    }

    return result;
}


QList<Layout> WindowManager::getAllLayouts()
{
    qDebug("-=[getAllLayouts]=-");

    return m_layouts;
}

#if 0
QList<int> WindowManager::getAllSurfacesOfProcess(int pid)
{
    QList<int> result;
#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    struct ilmSurfaceProperties surfaceProperties;

    for (int i = 0; i < m_appSurfaces.size(); ++i)
    {
        ilm_getPropertiesOfSurface(m_appSurfaces.at(i), &surfaceProperties);
        if (pid == surfaceProperties.creatorPid)
        {
            result.append(m_appSurfaces.at(i));
        }
    }
#endif
    return result;
}
#endif

QList<int> WindowManager::getAvailableLayouts(int numberOfAppSurfaces)
{
    qDebug("-=[getAvailableLayouts]=-");
    QList<Layout>::const_iterator i = m_layouts.begin();

    QList<int> result;
    while (i != m_layouts.constEnd())
    {
        if (i->layoutAreas.size() == numberOfAppSurfaces)
        {
            result.append(i->id);
        }

        ++i;
    }

    return result;
}

#if 0
QList<int> WindowManager::getAvailableSurfaces()
{
    qDebug("-=[getAvailableSurfaces]=-");

    return m_appSurfaces;
}
#endif

QString WindowManager::getLayoutName(int layoutId)
{
    qDebug("-=[getLayoutName]=-");
    QList<Layout>::const_iterator i = m_layouts.begin();

    QString result = "not found";
    while (i != m_layouts.constEnd())
    {
        if (i->id == layoutId)
        {
            result = i->name;
        }

        ++i;
    }

    return result;
}

void WindowManager::hideLayer(int layer)
{
    qDebug("-=[hideLayer]=-");
    qDebug("layer %d", layer);

#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    // POPUP=0, HOMESCREEN_OVERLAY=1, APPS=2, HOMESCREEN=3
    if (layer >= 0 && layer < WINDOWMANAGER_LAYER_NUM) {
        /* hide target layer */
        m_showLayers[layer] = 0;
        if (layer == 2) {
            /* clear pending flag */
            m_pending_to_show = -1;
        }
        renderLayers();
    }
#endif
}

int WindowManager::setLayoutById(int layoutId)
{
    qDebug("-=[setLayoutById]=-");
    int result = WINDOWMANAGER_NO_ERROR;
    m_currentLayout = layoutId;

    mp_layoutAreaToSurfaceIdAssignment->clear();

    dumpScene();

    return result;
}

int WindowManager::setLayoutByName(const QString &layoutName)
{
    qDebug("-=[setLayoutByName]=-");
    int result = WINDOWMANAGER_NO_ERROR;

    QList<Layout>::const_iterator i = m_layouts.begin();

    while (i != m_layouts.constEnd())
    {
        if (i->name == layoutName)
        {
            m_currentLayout = i->id;

            mp_layoutAreaToSurfaceIdAssignment->clear();

            dumpScene();
        }

        ++i;
    }

    return result;
}

int WindowManager::setSurfaceToLayoutArea(int surfaceId, int layoutAreaId)
{
    qDebug("-=[setSurfaceToLayoutArea]=-");
    int result = WINDOWMANAGER_NO_ERROR;

    qDebug("surfaceId %d", surfaceId);
    qDebug("layoutAreaId %d", layoutAreaId);
    mp_layoutAreaToSurfaceIdAssignment->insert(layoutAreaId, surfaceId);

    updateScreen();

    dumpScene();

    return result;
}

void WindowManager::showLayer(int layer)
{
    qDebug("-=[showLayer]=-");
    qDebug("layer %d", layer);

#ifdef HAVE_IVI_LAYERMANAGEMENT_API
    // POPUP=0, HOMESCREEN_OVERLAY=1, APPS=2, HOMESCREEN=3
    if (layer >= 0 && layer < WINDOWMANAGER_LAYER_NUM) {
        static const int layer_id_array[] = {
            WINDOWMANAGER_LAYER_POPUP,
            WINDOWMANAGER_LAYER_HOMESCREEN_OVERLAY,
            WINDOWMANAGER_LAYER_APPLICATIONS,
            WINDOWMANAGER_LAYER_HOMESCREEN,
        };

        m_showLayers[layer] = layer_id_array[layer];

        renderLayers();
    }
#endif
}

void WindowManager::showAppLayer(int app_pid)
{
    qDebug("-=[showAppLayer]=-");
    qDebug("pid %d", app_pid);

    if (app_pid == -1) {
        /* nothing to show */
        return;
    }

#ifdef HAVE_IVI_LAYERMANAGEMENT_API
#if !defined(NO_PROCESS_GROUP) || !defined(NO_PROCESS_SESSION)
    pid_t surf_pid = appPid2surfPid(app_pid);
    if (surf_pid == -1) {
        qDebug("Error: app's pid(%d) failed to convert surface's pid.", app_pid);
        return;
    }
#else
    pid_t surf_pid = app_pid;
#endif

    /* clear pending flag */
    m_pending_to_show = -1;

    /* search layer id for application to show */
    QMap<pid_t, t_ilm_layer>::const_iterator i = m_appLayers.find(surf_pid);

    if (i != m_appLayers.end()) {
        qDebug("Found layer(%d) to show for app(pid=%d)", m_showLayers[2], surf_pid);
        m_showLayers[2] = i.value();
    } else {
        QMap<pid_t, t_ilm_surface>::const_iterator j = m_appSurfaces.find(surf_pid);
        /* check if this app is registered */
        if (j == m_appSurfaces.end()) {
            qDebug("New app %d", surf_pid);
            m_appSurfaces.insert(surf_pid, 0); /* register pid only so far */
        } /* check app is required keep running while background */

        /* Probably app layer hasn't been made yet */
        m_pending_to_show = app_pid;
        /* hide current app once, back to default screen */
        m_showLayers[2] = 0;
        qDebug("No layer to show for app(pid=%d)", surf_pid);
    }
    renderLayers();
#endif
}

void WindowManager::showAppLayer(const QString &app_id, int app_pid)
{
    qDebug("-=[showAppLayer]=-");
    qDebug() << "id=" << app_id << ", pid=" << app_pid;

    if (app_pid == -1) {
        /* nothing to show */
        return;
    }
#ifdef HAVE_IVI_LAYERMANAGEMENT_API
#if !defined(NO_PROCESS_GROUP) || !defined(NO_PROCESS_SESSION)
    pid_t surf_pid = appPid2surfPid(app_pid);
    if (surf_pid == -1) {
        qDebug("NOTE: surface for this app(%d) hasn't created yet.", app_pid);

        checkBackgroundApps(app_id, app_pid);
        m_pending_to_show = app_pid;
        return;
    }
#else
    pid_t surf_pid = app_pid;
#endif
    /* clear pending flag */
    m_pending_to_show = -1;

    /* search layer id for application to show */
    QMap<pid_t, t_ilm_layer>::const_iterator i = m_appLayers.find(surf_pid);

    if (i != m_appLayers.end()) {
        qDebug("Found layer(%d) to show for app(pid=%d)", m_showLayers[2], surf_pid);
        m_showLayers[2] = i.value();
    } else {
        QMap<pid_t, t_ilm_surface>::const_iterator j = m_appSurfaces.find(surf_pid);
        /* check if this app is registered */
        if (j == m_appSurfaces.end()) {
            qDebug("New app %d", surf_pid);
            m_appSurfaces.insert(surf_pid, 0); /* register pid only so far */
        }
        /* check app is required keep running while background */
        checkBackgroundApps(app_id, app_pid);

        /* Probably app layer hasn't been made yet */
        m_pending_to_show = app_pid;
        /* hide current app once, back to default screen */
        m_showLayers[2] = 0;
        qDebug("No layer to show for app(pid=%d)", surf_pid);
    }
    renderLayers();
#endif
}

#ifdef HAVE_IVI_LAYERMANAGEMENT_API
void WindowManager::checkBackgroundApps(const QString &app_id, int app_pid)
{
    /* check if rendering is always required */
    /* QML apps should be always rendered if */
    for (int k = 0; k != m_keepApps.size(); k++) {
        if (app_id.indexOf(m_keepApps[k]) == 0) {
          int idx = m_bgApps.indexOf(app_pid);
          if (idx == -1) {
              m_bgApps.append(app_pid);
          }
        }
    }
}

#if !defined(NO_PROCESS_GROUP) || !defined(NO_PROCESS_SESSION)
bool WindowManager::otherPids(pid_t pid, pid_t &gpid, pid_t &spid)
{
    char buffer[60];
    int rc;
    FILE *f;

    // get the filename
    rc = snprintf(buffer, sizeof buffer, "/proc/%d/stat", (int)pid);
    if (rc < 0 || rc >= (int)(sizeof buffer)) {
        qDebug("internal unexpected error %d", rc);
        return false;
    }
    // open the status file
    f = fopen(buffer, "r");
    if (f == NULL) {
        qDebug("failed to open %s: %m", buffer);
        return false;
    }
    // read and parse the status file
    rc = fscanf(f, "%*d %*s %*c %*d %d %d", &gpid, &spid);
    fclose(f);
    if (rc != 2) {
        qDebug("failed to parse %s: %d", buffer, rc);
        return false;
    }
    qDebug("retrieved for pid=%d gpid=%d spid=%d", (int)pid, (int)gpid, (int)spid);
    return true;
}

void WindowManager::registerAppPid2surfPid(pid_t surf_pid)
{
    /* Register app's and surface's pid to conversion table */
    pid_t gpid, spid;

    if (otherPids(surf_pid, gpid, spid)) {
        QMap<pid_t, pid_t>::const_iterator i;
#if !defined(NO_PROCESS_GROUP)
        i == m_appPid2surfPid.find(gpid);
        if (i != m_appPid2surfPid.end()) {
            m_appPid2surfPid.insert(gpid, surf_pid);
        }
#endif
#if !defined(NO_PROCESS_SESSION)
        i == m_appPid2surfPid.find(spid);
        if (i != m_appPid2surfPid.end()) {
            m_appPid2surfPid.insert(spid, surf_pid);
        }
#endif
    }
}

pid_t WindowManager::appPid2surfPid(pid_t app_pid)
{
    if (app_pid == 0) {
        /* pid = 0 means it is HomeScreen itself */
        return app_pid;
    }

    QMap<pid_t, pid_t>::const_iterator j;
    j = m_appPid2surfPid.find(app_pid);
    if (j != m_appPid2surfPid.end()) {
        /* app pid has already been registered. */
        return j.value();
    }

    return -1;
}
#endif

#endif  // HAVE_IVI_LAYMERMANAGEMENT_API
