/*
 * Copyright (C) 2015, 2016, 2017 IoT.bzh
 * Author "Fulup Ar Foll"
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define _GNU_SOURCE         /* See feature_test_macros(7) */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <json-c/json.h>

#define AFB_BINDING_VERSION 2
#include <afb/afb-binding.h>

#include "utils-jbus.h"

static const char _added_[]     = "added";
static const char _continue_[]  = "continue";
static const char _changed_[]   = "changed";
static const char _detail_[]    = "detail";
static const char _id_[]        = "id";
static const char _install_[]   = "install";
static const char _once_[]      = "once";
static const char _pause_[]     = "pause";
static const char _resume_[]    = "resume";
static const char _runid_[]     = "runid";
static const char _runnables_[] = "runnables";
static const char _runners_[]   = "runners";
static const char _start_[]     = "start";
static const char _state_[]     = "state";
static const char _stop_[]      = "stop";
static const char _terminate_[] = "terminate";
static const char _uninstall_[] = "uninstall";

static struct jbus *jbus;

/*
 * Structure for asynchronous call handling
 */
struct memo
{
	struct afb_req request;	/* the recorded request */
	const char *method;	/* the called method */
};

/*
 * Creates the memo for the 'request' and the 'method'.
 * Returns the memo in case of success.
 * In case of error, send a failure answer and returns NULL.
 */
static struct memo *memo_create(struct afb_req request, const char *method)
{
	struct memo *memo = malloc(sizeof *memo);
	if (memo == NULL)
		afb_req_fail(request, "failed", "out of memory");
	else {
		memo->request = request;
		memo->method = method;
		afb_req_addref(request);
	}
	return memo;
}

/*
 * Sends the asynchronous failed reply to the request recorded by 'memo'
 * Then fress the resources.
 */
static void memo_fail(struct memo *memo, const char *info)
{
	afb_req_fail(memo->request, "failed", info);
	afb_req_unref(memo->request);
	free(memo);
}

/*
 * Sends the asynchronous success reply to the request recorded by 'memo'
 * Then fress the resources.
 */
static void memo_success(struct memo *memo, struct json_object *obj, const char *info)
{
	afb_req_success(memo->request, obj, info);
	afb_req_unref(memo->request);
	free(memo);
}

/*
 * Broadcast the event "application-list-changed".
 * This event is sent was the event "changed" is received from dbus.
 */
static void application_list_changed(const char *data, void *closure)
{
	afb_daemon_broadcast_event("application-list-changed", NULL);
}

/*
 * Builds if possible the json object having one field of name 'tag'
 * whose value is 'obj' like here: { tag: obj } and returns it.
 * In case of error or when 'tag'==NULL or 'obj'==NULL, 'obj' is returned.
 * The reference count of 'obj' is not incremented.
 */
static struct json_object *embed(const char *tag, struct json_object *obj)
{
	struct json_object *result;

	if (obj == NULL || tag == NULL)
		result = obj;
	else {
		result = json_object_new_object();
		if (result == NULL) {
			/* can't embed */
			result = obj;
		}
		else {
			/* TODO why is json-c not returning a status? */
			json_object_object_add(result, tag, obj);
		}
	}
	return result;
}

/*
 * Callback for replies made by 'embed_call_void'.
 */
static void embed_call_void_callback(int iserror, struct json_object *obj, struct memo *memo)
{
	AFB_DEBUG("(afm-main-binding) %s(true) -> %s\n", memo->method,
			obj ? json_object_to_json_string(obj) : "NULL");

	if (iserror) {
		memo_fail(memo, obj ? json_object_get_string(obj) : "framework daemon failure");
	} else {
		memo_success(memo, embed(memo->method, json_object_get(obj)), NULL);
	}
}

/*
 * Calls with DBus the 'method' of the user daemon without arguments.
 */
static void embed_call_void(struct afb_req request, const char *method)
{
	struct memo *memo;

	/* store the request */
	memo = memo_create(request, method);
	if (memo == NULL)
		return;

	if (jbus_call_sj(jbus, method, "true", (void*)embed_call_void_callback, memo) < 0)
		memo_fail(memo, "dbus failure");
}

/*
 * Callback for replies made by 'call_appid' and 'call_runid'.
 */
static void call_xxxid_callback(int iserror, struct json_object *obj, struct memo *memo)
{
	AFB_DEBUG("(afm-main-binding) %s -> %s\n", memo->method, 
			obj ? json_object_to_json_string(obj) : "NULL");

	if (iserror) {
		memo_fail(memo, obj ? json_object_get_string(obj) : "framework daemon failure");
	} else {
		memo_success(memo, json_object_get(obj), NULL);
	}
}

/*
 * Calls with DBus the 'method' of the user daemon with the argument "id".
 */
static void call_appid(struct afb_req request, const char *method)
{
	struct memo *memo;
	char *sid;
	const char *id;

	id = afb_req_value(request, _id_);
	if (id == NULL) {
		afb_req_fail(request, "bad-request", "missing 'id'");
		return;
	}

	memo = memo_create(request, method);
	if (memo == NULL)
		return;

	if (asprintf(&sid, "\"%s\"", id) <= 0) {
		memo_fail(memo, "out of memory");
		return;
	}

	if (jbus_call_sj(jbus, method, sid, (void*)call_xxxid_callback, memo) < 0)
		memo_fail(memo, "dbus failure");

	free(sid);
}

static void call_runid(struct afb_req request, const char *method)
{
	struct memo *memo;
	const char *id;

	id = afb_req_value(request, _runid_);
	if (id == NULL) {
		afb_req_fail(request, "bad-request", "missing 'runid'");
		return;
	}

	memo = memo_create(request, method);
	if (memo == NULL)
		return;

	if (jbus_call_sj(jbus, method, id, (void*)call_xxxid_callback, memo) < 0)
		memo_fail(memo, "dbus failure");
}

/************************** entries ******************************/

static void runnables(struct afb_req request)
{
	embed_call_void(request, _runnables_);
}

static void detail(struct afb_req request)
{
	call_appid(request, _detail_);
}

static void start_callback(int iserror, struct json_object *obj, struct memo *memo)
{
	AFB_DEBUG("(afm-main-binding) %s -> %s\n", memo->method, 
			obj ? json_object_to_json_string(obj) : "NULL");

	if (iserror) {
		memo_fail(memo, obj ? json_object_get_string(obj) : "framework daemon failure");
	} else {
		obj = json_object_get(obj);
		if (json_object_get_type(obj) == json_type_int)
			obj = embed(_runid_, obj);
		memo_success(memo, obj, NULL);
	}
}

static void start(struct afb_req request)
{
	struct memo *memo;
	const char *id;
	char *query;
	int rc;

	/* get the id */
	id = afb_req_value(request, _id_);
	if (id == NULL) {
		afb_req_fail(request, "bad-request", "missing 'id'");
		return;
	}

	/* prepares asynchronous request */
	memo = memo_create(request, _start_);
	if (memo == NULL)
		return;

	/* create the query */
	rc = asprintf(&query, "{\"id\":\"%s\"}", id);
	if (rc < 0) {
		memo_fail(memo, "out of memory");
		return;
	}

	/* calls the service asynchronously */
	if (jbus_call_sj(jbus, _start_, query, (void*)start_callback, memo) < 0)
		memo_fail(memo, "dbus failure");
	free(query);
}

static void once(struct afb_req request)
{
	call_appid(request, _once_);
}

static void terminate(struct afb_req request)
{
	call_runid(request, _terminate_);
}

static void pause(struct afb_req request)
{
	call_runid(request, _pause_);
}

static void resume(struct afb_req request)
{
	call_runid(request, _resume_);
}

static void runners(struct afb_req request)
{
	embed_call_void(request, _runners_);
}

static void state(struct afb_req request)
{
	call_runid(request, _state_);
}

static void install_callback(int iserror, struct json_object *obj, struct memo *memo)
{
	struct json_object *added;

	if (iserror) {
		memo_fail(memo, obj ? json_object_get_string(obj) : "framework daemon failure");
	} else {
		if (json_object_object_get_ex(obj, _added_, &added))
			obj = added;
		obj = json_object_get(obj);
		obj = embed(_id_, obj);
		memo_success(memo, obj, NULL);
	}
}
static void install(struct afb_req request)
{
	struct memo *memo;
	char *query;
	const char *filename;
	struct afb_arg arg;

	/* get the argument */
	arg = afb_req_get(request, "widget");
	filename = arg.path;
	if (filename == NULL) {
		afb_req_fail(request, "bad-request", "missing 'widget' file");
		return;
	}

	/* prepares asynchronous request */
	memo = memo_create(request, _install_);
	if (memo == NULL)
		return;

	/* makes the query */
	if (0 >= asprintf(&query, "\"%s\"", filename)) {
		afb_req_fail(request, "server-error", "out of memory");
		return;
	}

	/* calls the service asynchronously */
	if (jbus_call_sj(jbus, _install_, query, (void*)install_callback, memo) < 0)
		memo_fail(memo, "dbus failure");
	free(query);
}

static void uninstall(struct afb_req request)
{
	call_appid(request, _uninstall_);
}

static int init()
{
	int rc;
	struct sd_bus *sbus;

	/* creates the jbus for accessing afm-user-daemon */
	sbus = afb_daemon_get_user_bus();
	if (sbus == NULL)
		return -1;
	jbus = create_jbus(sbus, "/org/AGL/afm/user");
        if (jbus == NULL)
		return -1;

	/* records the signal handler */
	rc = jbus_on_signal_s(jbus, _changed_, application_list_changed, NULL);
	if (rc < 0) {
		jbus_unref(jbus);
		return -1;
	}

	return 0;
}

static const struct afb_verb_v2 verbs[] =
{
	{_runnables_, runnables, NULL, "Get list of runnable applications",          AFB_SESSION_CHECK },
	{_detail_   , detail,    NULL, "Get the details for one application",        AFB_SESSION_CHECK },
	{_start_    , start,     NULL, "Start an application",                       AFB_SESSION_CHECK },
	{_once_     , once,      NULL, "Start once an application",                  AFB_SESSION_CHECK },
	{_terminate_, terminate, NULL, "Terminate a running application",            AFB_SESSION_CHECK },
	{_pause_    , pause,     NULL, "Pause a running application",                AFB_SESSION_CHECK },
	{_resume_   , resume,    NULL, "Resume a paused application",                AFB_SESSION_CHECK },
	{_stop_     , pause,     NULL, "Obsolete since 2016/11/08, use 'pause' instead", AFB_SESSION_CHECK },
	{_continue_ , resume,    NULL, "Obsolete since 2016/11/08, use 'resume' instead", AFB_SESSION_CHECK },
	{_runners_  , runners,   NULL, "Get the list of running applications",       AFB_SESSION_CHECK },
	{_state_    , state,     NULL, "Get the state of a running application",     AFB_SESSION_CHECK },
	{_install_  , install,   NULL, "Install an application using a widget file", AFB_SESSION_CHECK },
	{_uninstall_, uninstall, NULL, "Uninstall an application",                   AFB_SESSION_CHECK },
	{ NULL, NULL, NULL, NULL, 0 }
};

const struct afb_binding_v2 afbBindingV2 = {
	.api = "afm-main",
	.specification = NULL,
	.info = "Application Framework Master Service",
	.verbs = verbs,
	.preinit = NULL,
	.init = init,
	.onevent = NULL,
	.noconcurrency = 0
};

