# Usage

## Add libappcontroller as a static library to your binding

In your `config.cmake` file, add a dependency to the controller library, i.e:

```cmake
set(PKG_REQUIRED_LIST
	json-c
	afb-daemon
	appcontroller --> this is the controller library dependency name.
)
```

Or you can also use the [FIND_PACKAGE](https://cmake.org/cmake/help/v3.6/command/find_package.html?highlight=find_package)
CMake command to add it.

## Declare your controller config section in your binding

```C
// CtlSectionT syntax:
// key: "section name in config file"
// loadCB: callback to process section
// handle: a void* pass to callback when processing section
static CtlSectionT ctlSections[]= {
    {.key="plugins" , .loadCB= PluginConfig, .handle= &halCallbacks},
    {.key="onload"  , .loadCB= OnloadConfig},
    {.key="halmap"  , .loadCB= MapConfigLoad},
    {.key=NULL}
};

```

## Do the controller config parsing at binding pre-init

```C
   // check if config file exist
    const char *dirList= getenv("CTL_CONFIG_PATH");
    if (!dirList) dirList=CONTROL_CONFIG_PATH;

    const char *configPath = CtlConfigSearch(apiHandle, dirList, "prefix");
    if(!confiPath) return -1;

    ctlConfig = CtlConfigLoad(dirList, ctlSections);
    if (!ctlConfig) return -1;
```

## Execute the controller config during binding init

```C
  int err = CtlConfigExec (ctlConfig);
```

## (Optional) Migrate from the git submodule version

### Remove the git submodule version

If you already use the controller component but use the submodule version then
you have to get rid of it to be sure to link and use the library version. To do
so, you have to do the following:

* deinitialize the submodule using `git`

```bash
# This example assumes that the git submodule is named app-controller-submodule
# and is located at your root project repository.
git submodule deinit app-controller-submodule
```

* remove the relative submodule lines from the `.gitmodules` file

```bash
vim .gitmodules
```

* remove the `ctl-utilities` target link from any CMake target you specified.
 Those lines look like:

```bash
TARGET_LINK_LIBRARIES(${TARGET_NAME}
    ctl-utilities # REMOVE THIS LINE
    ${link_libraries}
    )
```

### Use the native af-binder functions

The controller redefined some binder's functions to add an abstraction between
several binding versions. But now, as the controller is binding v3 only, the
abstraction layer from the controller has been removed and you should switch
your functions from the old controller's definitions to binder's definitions.

You have to replace any `include` statements of `afb-definitions.h` by
`afb/afb-binding.h` if you included it somewhere. If you have only included
`ctl-config.h` file then you are fine.

```diff
- #include <afb-definitions.h>
+ #include <afb/afb-binding.h>
```

To help migrating gracefully the old controller's definitions, you could use the
sed script to automate the migration for you. From your project root directory,
executes the following commands:

```bash
wget -O controller-migration.sed https://iot.bzh/download/public/tools/controller-migration.sed
for f in $(find . -name *.c -o -name *.h)
do
sed -i -rf controller-migration.sed  ${f}
done
```

> **NOTE**: `AFB_ServiceCall` and `AFB_ServiceSync` has been migrated to their
> homologue `afb_api_call_legacy` and `afb_api_call_sync_legacy` respectively
> but you have to be aware that they are *legacy* functions and you should use
> the news call functions `afb_api_call` and `afb_api_call_sync` instead.
> Cf [Binder API functions reference](../../../apis_services/reference/af-binder/reference-v3/func-api.html#calls-and-job-functions)
> for more details on these functions.

As a reminder, here are the old controller's functions definitions that you
should migrate:

```c
    #define AFB_ReqNone NULL
    typedef afb_req_t   AFB_ReqT;
    typedef afb_api_t   AFB_ApiT;
    typedef afb_event_t AFB_EventT;

    #define AFB_EventIsValid(eventid) eventid
    #define AFB_EventPush afb_event_push
    #define AFB_ReqSubscribe afb_req_subscribe
    #define AFB_EventMake(api, name) afb_api_make_event(api, name)

    #define AFB_ReqJson(request) afb_req_json(request)

    #define AFB_ReqSuccess  afb_req_success
    #define AFB_ReqSuccessF afb_req_success_f
    #define AFB_ReqFail     afb_req_fail
    #define AFB_ReqFailF    afb_req_fail_f

    #define AFB_ReqNotice(request, ...)   AFB_REQ_NOTICE (request, __VA_ARGS__)
    #define AFB_ReqWarning(request, ...)  AFB_REQ_WARNING (request, __VA_ARGS__)
    #define AFB_ReqDebug(request, ...)    AFB_REQ_DEBUG (request, __VA_ARGS__)
    #define AFB_ReqError(request, ...)    AFB_REQ_ERROR (request, __VA_ARGS__)
    #define AFB_ReqInfo(request, ...)     AFB_REQ_INFO (request, __VA_ARGS__)

    #define AFB_ApiVerbose(api, level, ...) afb_api_verbose(api, level, __VA_ARGS__)
    #define AFB_ApiNotice(api, ...)   AFB_API_NOTICE (api, __VA_ARGS__)
    #define AFB_ApiWarning(api, ...)  AFB_API_WARNING (api, __VA_ARGS__)
    #define AFB_ApiDebug(api, ...)    AFB_API_DEBUG (api, __VA_ARGS__)
    #define AFB_ApiError(api, ...)    AFB_API_ERROR (api, __VA_ARGS__)
    #define AFB_ApiInfo(api, ...)     AFB_API_INFO (api, __VA_ARGS__)

    #define AFB_GetApiSettings afb_api_settings

    #define AFB_ReqIsValid(request)   request
    #define AFB_EvtIsValid(evtHandle) evtHandle

    #define AFB_ServiceCall(api, ...) afb_api_call_legacy(api, __VA_ARGS__)
    #define AFB_ServiceSync(api, ...) afb_api_call_sync_legacy(api, __VA_ARGS__)

    #define AFB_ApiCall(api, ...) afb_api_call(api, __VA_ARGS__)
    #define AFB_ApiSync(api, ...) afb_api_call_sync(api, __VA_ARGS__)

    #define AFB_ReqVCBData           afb_req_get_vcbdata
    #define AFB_ReqGetApi            afb_req_get_api
    #define AFB_GetEventLoop(api)    afb_api_get_event_loop(api)
    #define AFB_RootDirGetFD(api)    afb_api_rootdir_get_fd(api)
    #define AFB_RequireApi(api, ...) afb_api_require_api(api, __VA_ARGS__)

    #define AFB_ClientCtxSet(request, replace, createCB, freeCB, handle) afb_req_context(request, replace, createCB, freeCB, handle)
    #define AFB_ClientCtxClear(request) afb_req_context_clear(request)

    #define AFB_ReqSetLOA(request, level) afb_req_session_set_LOA(request, level)

    #define AFB_NewApi afb_api_new_api

    #define AFB_ApiAddVerb afb_api_add_verb

    #define AFB_ApiSetUserData afb_api_set_userdata
    #define AFB_ApiGetUserData afb_api_get_userdata

    #define AFB_ApiOnEvent afb_api_on_event
    #define AFB_ApiOnInit  afb_api_on_init
    #define AFB_ApiSeal    afb_api_seal
```