/*
 * Copyright (C) 2018-2020 Konsulko Group
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <QDebug>

#include "callmessage.h"
#include "eventmessage.h"
#include "responsemessage.h"
#include "messagefactory.h"
#include "messageengine.h"
#include "messageenginefactory.h"
#include "bluetooth.h"
#include "bluetoothmodel.h"


Bluetooth::Bluetooth (QUrl &url, QQmlContext *context, QObject * parent) :
    QObject(parent),
    m_context(context)
{
    m_mloop = MessageEngineFactory::getInstance().getMessageEngine(url);
    m_bluetooth = new BluetoothModel();
    QObject::connect(m_mloop.get(), &MessageEngine::connected, this, &Bluetooth::onConnected);
    QObject::connect(m_mloop.get(), &MessageEngine::disconnected, this, &Bluetooth::onDisconnected);
    QObject::connect(m_mloop.get(), &MessageEngine::messageReceived, this, &Bluetooth::onMessageReceived);

    BluetoothModelFilter *m_model = new BluetoothModelFilter();
    m_model->setSourceModel(m_bluetooth);
    m_model->setFilterFixedString("true");
    context->setContextProperty("BluetoothPairedModel", m_model);

    m_model = new BluetoothModelFilter();
    m_model->setSourceModel(m_bluetooth);
    m_model->setFilterFixedString("false");
    context->setContextProperty("BluetoothDiscoveryModel", m_model);

    uuids.insert("a2dp", "0000110a-0000-1000-8000-00805f9b34fb");
    uuids.insert("avrcp", "0000110e-0000-1000-8000-00805f9b34fb");
    uuids.insert("hfp", "0000111f-0000-1000-8000-00805f9b34fb");
}

Bluetooth::~Bluetooth()
{
}

void Bluetooth::send_command(QString verb, QJsonObject parameter)
{
    std::unique_ptr<Message> msg = MessageFactory::getInstance().createOutboundMessage(MessageId::Call);
    if (!msg)
        return;

    CallMessage* btmsg = static_cast<CallMessage*>(msg.get());
    btmsg->createRequest("Bluetooth-Manager", verb, parameter);
    m_mloop->sendMessage(std::move(msg));
}

void Bluetooth::setPower(bool state)
{
    QJsonObject parameter;
    parameter.insert("powered", state ? "true" : "false");
    send_command("adapter_state", parameter);
}

void Bluetooth::setDiscoverable(bool state)
{
    QJsonObject parameter;
    parameter.insert("discoverable", state ? "true" : "false");
    send_command("adapter_state", parameter);

    m_discoverable = state;

    emit discoverableChanged();
}

void Bluetooth::discovery_command(bool state)
{
    QJsonObject parameter;
    parameter.insert("discovery", state ? "true" : "false");

    set_discovery_filter();

    send_command("adapter_state", parameter);
}

void Bluetooth::start_discovery()
{
    discovery_command(true);

    // temp workaround to list already discovered devices
    send_command("managed_objects", QJsonObject());
}

void Bluetooth::stop_discovery()
{
    discovery_command(false);
}

void Bluetooth::remove_device(QString device)
{
    QJsonObject parameter;
    parameter.insert("device", device);

    send_command("remove_device", parameter);
}

void Bluetooth::pair(QString device)
{
    QJsonObject parameter;
    parameter.insert("device", device);

    send_command("pair", parameter);
}

void Bluetooth::cancel_pair(QString device)
{
    send_command("cancel_pairing", QJsonObject());
}

void Bluetooth::connect(QString device, QString uuid)
{
    QJsonObject parameter;
    uuid = process_uuid(uuid);
    parameter.insert("device", device);
    parameter.insert("uuid", uuid);
    send_command("connect", parameter);
}

void Bluetooth::connect(QString device)
{
    QJsonObject parameter;
    parameter.insert("device", device);
    send_command("connect", parameter);
}

void Bluetooth::disconnect(QString device, QString uuid)
{
    QJsonObject parameter;
    uuid = process_uuid(uuid);
    parameter.insert("device", device);
    parameter.insert("uuid", uuid);
    send_command("disconnect", parameter);
}

void Bluetooth::disconnect(QString device)
{
    QJsonObject parameter;
    parameter.insert("device", device);
    send_command("disconnect", parameter);
}

void Bluetooth::send_confirmation(int pincode)
{
    QJsonObject parameter;
    parameter.insert("pincode", pincode);
    send_command("confirm_pairing", parameter);
}


void Bluetooth::set_discovery_filter()
{
    QStringListIterator eventIterator(uuids.values());
    QJsonObject parameter;
    QJsonArray array;

    while (eventIterator.hasNext())
        array.push_back(eventIterator.next());

    // send inital adapter state + discovery filter
    parameter.insert("filter", array);
    parameter.insert("transport", "bredr");
    send_command("adapter_state", parameter);
}

void Bluetooth::onConnected()
{
    QStringListIterator eventIterator(events);

    while (eventIterator.hasNext()) {
        QJsonObject parameter;
        parameter.insert("value", eventIterator.next());
        send_command("subscribe", parameter);
    }

    // send initial list
    send_command("managed_objects", QJsonObject());

    // get initial power state
    send_command("adapter_state", QJsonObject());
}

void Bluetooth::onDisconnected()
{
    QStringListIterator eventIterator(events);

    while (eventIterator.hasNext()) {
        QJsonObject parameter;
        parameter.insert("value", eventIterator.next());
        send_command("unsubscribe", parameter);
    }
}

void Bluetooth::populateDeviceList(QJsonObject data)
{
    QJsonArray devices = data.value("devices").toArray();

    m_bluetooth->removeAllDevices();

    for (auto value : devices) {
        BluetoothDevice *device = m_bluetooth->updateDeviceProperties(nullptr, value.toObject());
        m_bluetooth->addDevice(device);
    }
}

void Bluetooth::processDeviceChangesEvent(QJsonObject data)
{
    QString action = data.value("action").toString();
    QString id = data.value("device").toString();

    if (id.isEmpty())
        return;

    BluetoothDevice *device = m_bluetooth->getDevice(id);
    if (action == "removed") {
        if (device != nullptr)
            m_bluetooth->removeDevice(device);
	return;
    }

    BluetoothDevice *ndevice = m_bluetooth->updateDeviceProperties(device, data);
    if (ndevice == nullptr) {
        qDebug() << "bt - failed to create device object with id: " << id;
        return;
    }
    if (device == nullptr)  //device not previously in model
        m_bluetooth->addDevice(ndevice);
}

void Bluetooth::processAdapterChangesEvent(QJsonObject data)
{
    QString action = data.value("action").toString();
    if (action != "changed")
        return;

    QJsonObject properties = data.value("properties").toObject();
    if (!properties.contains("powered"))
        return;

    bool powered = properties.find("powered").value().toBool();
    if (!powered)
        m_bluetooth->removeAllDevices();

    if (m_power != powered) {
        m_power = powered;
        emit powerChanged(powered);
    }

    if (!m_power)
        m_discoverable = false;
}

void Bluetooth::onMessageReceived(std::shared_ptr<Message> msg)
{
    if (!msg)
        return;

    if (msg->isEvent()) {
        std::shared_ptr<EventMessage> emsg = std::static_pointer_cast<EventMessage>(msg);
        QString ename = emsg->eventName();
        QString eapi = emsg->eventApi();
        QJsonObject data = emsg->eventData();
        if (eapi != "Bluetooth-Manager")
            return;
        if (ename == "device_changes") {
            processDeviceChangesEvent(data);
        } else if (ename == "adapter_changes") {
            processAdapterChangesEvent(data);
        } else if (ename == "agent") {
            emit requestConfirmationEvent(data);
        }
    } else if (msg->isReply()) {
        std::shared_ptr<ResponseMessage> rmsg = std::static_pointer_cast<ResponseMessage>(msg);
        QString verb = rmsg->requestVerb();
        QJsonObject data = rmsg->replyData();
        if (rmsg->requestApi() != "Bluetooth-Manager")
            return;

        if (rmsg->replyStatus() == "failed") {
            qDebug() << "failed bt verb:" << verb;
            if (rmsg->replyInfo().contains("No adapter")) {
                m_power = false;
                emit powerChanged(m_power);
            }
        }
        else if (verb == "managed_objects") {
            populateDeviceList(data);
        } else if (verb == "adapter_state") {
            bool powered = data.value("powered").toBool();
            if (m_power != powered) {
                m_power = powered;
                emit powerChanged(m_power);
            }
        }
    }
}
