#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>
#include <asm/errno.h>

//#define DEBUG

static void pabort(const char *s)
{
	perror(s);
	abort();
}

static uint8_t mode = SPI_MODE_2;
static uint8_t bits = 8;
static uint32_t speed = 25000000;

void write_16(char* ptr, uint8_t val)
{
	
}

int spi_init(char *devname)
{
	int fd;
	int ret = 0;

	fd = open(devname, O_RDWR);
	if (fd < 0)
		pabort("can't open device");

	/* spi mode */
	ret = ioctl(fd, SPI_IOC_WR_MODE, &mode);
	if (ret == -1)
		pabort("can't set spi mode");

	ret = ioctl(fd, SPI_IOC_RD_MODE, &mode);
	if (ret == -1)
		pabort("can't get spi mode");

	/* bits per word */
	ret = ioctl(fd, SPI_IOC_WR_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't set bits per word");

	ret = ioctl(fd, SPI_IOC_RD_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't get bits per word");

	/* max speed hz */
	ret = ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't set max speed hz");

	ret = ioctl(fd, SPI_IOC_RD_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't get max speed hz");

#ifdef DEBUG
	printf("spi mode: %d\n", mode);
	printf("bits per word: %d\n", bits);
	printf("max speed: %d Hz (%d KHz)\n", speed, speed/1000);
#endif

	return fd;
}

static int
spi_write(int fd, uint8_t *buf, int len)
{
	int i;
	int ret;
	struct spi_ioc_transfer tr = {
		.tx_buf = (unsigned long)buf,
		.rx_buf = (unsigned long)NULL,
		.len = len,
		.delay_usecs = 0,
		.speed_hz = 0,
		.bits_per_word = bits,
	};

#ifdef DEBUG
	printf("->");
	for (i = 0; i < len; i++)
		printf(" %02x", buf[i]);
	printf("\n");
#endif

	ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);
	if (ret < 1) {
		fprintf(stderr, "can't send spi message");
		return -EIO;
	}
	return 0;
}

static int
spi_read(int fd, uint8_t *buf, int len)
{
	int i;
	int ret;
	struct spi_ioc_transfer tr = {
		.tx_buf = (unsigned long)NULL,
		.rx_buf = (unsigned long)buf,
		.len = len,
		.delay_usecs = 0,
		.speed_hz = 0,
		.bits_per_word = bits,
	};

	ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);
	if (ret < 1) {
		fprintf(stderr, "can't send spi message");
		return -EIO;
	}
#ifdef DEBUG
	printf("<-");
	for (i = 0; i < len; i++)
		printf(" %02x", buf[i]);
	printf("\n");
#endif

	return 0;
}

static int
spi_rw(int fd, uint8_t *wbuf, uint8_t *rbuf, int len)
{
	int i;
	int ret;
	struct spi_ioc_transfer tr = {
		.tx_buf = (unsigned long)wbuf,
		.rx_buf = (unsigned long)rbuf,
		.len = len,
		.delay_usecs = 0,
		.speed_hz = 0,
		.bits_per_word = bits,
	};

#ifdef DEBUG
	printf("->");
	for (i = 0; i < len; i++)
		printf(" %02x", wbuf[i]);
	printf("\n");
#endif

	ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);
	if (ret < 1) {
		fprintf(stderr, "can't send spi message");
		return -EIO;
	}

#ifdef DEBUG
	printf("<-");
	for (i = 0; i < len; i++)
		printf(" %02x", rbuf[i]);
	printf("\n");
#endif

	return 0;
}

int spi_rreg(int fd, int addr, uint64_t *value, int len)
{
	int i;
	int ret = 0;
	uint64_t _value = 0;
	uint8_t data[2 + 8]; //max

	data[0] = 0x60; //normal read
	data[1] = addr; //spi status reg
	data[2] = 0x00;

	ret = spi_rw(fd, data, data, 2 + len);
	if (ret)
		return ret;

	for (i = 0; i < len; i++)
		_value |= ((uint64_t)data[2 + i] << (8 * i));
	*value = _value;

	return 0;
}

int spi_wreg(int fd, int addr, uint64_t value, int len)
{
	int i;
	int ret = 0;
	uint8_t data[2 + 8]; //max

	if ((len != 1) && (len != 2) && (len != 4) && (len != 8))
		return -EINVAL;

	data[0] = 0x61; //normal write
	data[1] = addr; //spi status reg

	for (i = 0; i < len; i++)
		data[2 + i] = (value >> (8 * i)) & 0xff;

	ret = spi_rw(fd, data, data, 2 + len);

	return ret;
}

int spi_get_stat(int fd)
{
	int ret;
	uint64_t stat;
	ret = spi_rreg(fd, 0xfe, &stat, 1);
	if (ret < 0)
		return ret;
	return stat;
}

int spi_wait_busy(int fd, int retry)
{
	int tmp;
	uint64_t reg;

	while (retry--) {
		tmp = spi_rreg(fd, 0xfe, &reg, 1);

		/* exit in case of read error */
		if (tmp < 0)
			return tmp;

		/* return if not busy */
		if (reg & (1 << 5))
			return 0;
	}

	if (retry == 0) {
		printf("SPI status reg 0x%02x\n", tmp);
		return -EBUSY;
	}

	return 0;
}

int sw_rreg(int fd, int page, int addr, uint64_t *value, int len)
{
	int tmp;
	uint64_t reg;

	/* do not allow to write shared registers */
	if (addr >= 0xF0)
		return -EINVAL;
	/* 1. Issue a Normal Read Command (opcode = 0x60)
	 * to poll the SPIF bit in the SPI Status register (0xFE) to 
	 * determine the operation can start.
	 */
	tmp = spi_get_stat(fd);
	if (tmp < 0)
		return tmp;
	if (tmp & (1 << 7)) {
		printf("SPI status reg 0x%02x\n", tmp);
		return -EBUSY;
	}

	/* 2. Issue a Normal Write command (opcode = 0x61)
	 * to write the register page value into the SPI Page
	 * register 0xFF
	 */
	tmp = spi_wreg(fd, 0xff, page, 1);
	if (tmp < 0)
		return tmp;

	/* 3. Issue a Normal Read command (opcode = 0x60) to
	 * set up the required RoboSwitch register address.
	 */
	tmp = spi_rreg(fd, addr, &reg, len);
	if (tmp < 0)
		return tmp;

	/* 4. Issue a Normal Read command (opcode = 0x60) to poll
	 * the RACK bit in the SPI status register(0xFE) to
	 * determine the completion of read (register content
	 * gets loaded in SPI Data I/O register).
	 */
	tmp = spi_wait_busy(fd, 100);
	if (tmp < 0)
		return tmp;

	/* 5. Issue a Normal Read command (opcode = 0x60) to read
	 * the specific registers' content placed in the SPI Data
	 * I/O register (0xF0). */
	tmp = spi_rreg(fd, 0xf0, value, len);

	return tmp;
}

int sw_wreg(int fd, int page, int addr, uint64_t value, int len)
{
	int tmp;

	/* do not allow to read shared registers */
	if (addr >= 0xF0)
		return -EINVAL;
	/* 1. Issue a Normal Read Command (opcode = 0x60)
	 * to poll the SPIF bit in the SPI Status register
	 * (0xFE) to determine the operation can start.
	 */
	tmp = spi_get_stat(fd);
	if (tmp < 0)
		return tmp;
	if (tmp & (1 << 7)) {
		printf("SPI status reg 0x%02x\n", tmp);
		return -EBUSY;
	}

	/* 2. Issue a Normal Write command (opcode = 0x61)
	 * to set up the accessed register page value into
	 * the page register (0xFF).
	 */
	tmp = spi_wreg(fd, 0xff, page, 1);
	if (tmp < 0)
		return tmp;

	/* 3. Issue a Normal Write command (opcode = 0x61)
	 * to set up the accessed register address value,
	 * followed by the write content starting from a
	 * lower byte.
	 */
	tmp = spi_wreg(fd, addr, value, len);

	return tmp;
}

int main(int argc, char *argv[])
{
	int i;
	int ret = 0;
	int fd;

	if(argc < 4)
		goto usage;

	fd = spi_init(argv[1]);
	if (fd < 0) {
		fprintf(stderr, "Failed to open %s\n", argv[1]);
		exit(1);
	}

	/* dump */
	if (argv[2][0] == 'd') {
		int page = strtol(argv[3], NULL, 16);

		printf("Dumping page %d (0x00-0xf0):\n", page);
		for (i = 0; i < 0xF0; i++) {
			uint64_t reg;
			ret = sw_rreg(fd, page, i, &reg, 1);
			if ((i % 16) == 0)
				printf("0x%02x: ", i);
			if (ret < 0)
				printf("-- ");
			else
				printf("%02x ", reg);
			if ((i % 16) == 15)
				printf("\n");
		}
	}
	/* write */
	else if (argv[2][0] == 'w') {
		int page, addr;
		uint64_t value;
		int len = 1;
		if (argc < 6)
			goto usage;
		if ((argv[2][1] == '1') && (argv[2][2] == '6'))
			len = 2;
		if ((argv[2][1] == '3') && (argv[2][2] == '2'))
			len = 4;
		if ((argv[2][1] == '6') && (argv[2][2] == '4'))
			len = 8;
		page = strtoul(argv[3], NULL, 16);
		addr = strtoul(argv[4], NULL, 16);
		value = strtoull(argv[5], NULL, 16);

		printf("Writing 0x%02x:0x%02x value%d 0x%0*llx", page, addr, len * 8, len * 2, value);
		ret = sw_wreg(fd, page, addr, value, len);
		if (ret < 0)
			printf(" failed %d\n", value);
		else
			printf(" ok\n");
	}
	/* read */
	else if (argv[2][0] == 'r') {
		int page, addr;
		uint64_t value;
		int len = 1;
		if (argc < 5)
			goto usage;
		if ((argv[2][1] == '1') && (argv[2][2] == '6'))
			len = 2;
		if ((argv[2][1] == '3') && (argv[2][2] == '2'))
			len = 4;
		if ((argv[2][1] == '6') && (argv[2][2] == '4'))
			len = 8;
		page = strtoul(argv[3], NULL, 16);
		addr = strtoul(argv[4], NULL, 16);

		printf("Reading%d 0x%02x:0x%02x ", len * 8, page, addr);

		ret = sw_rreg(fd, page, addr, &value, len);
		if (ret < 0)
			printf("failed %d\n", value);
		else
			printf(" 0x%0*llx\n", 2 * len, value);
	}

	close(fd);

	return ret;

usage:
	if (fd)
		close(fd);

	fprintf(stderr, "Usage:\n");
	fprintf(stderr, "%s <spidev> d 0xpage               - dump page\n", argv[0]);
	fprintf(stderr, "%s <spidev> w 0xpage 0xreg 0xvalue - write register\n", argv[0]);
	fprintf(stderr, "%s <spidev> r 0xpage 0xreg         - read register\n", argv[0]);
	fprintf(stderr, "w/r - write/read 8-bit register\n");
	fprintf(stderr, "w16/r16 - write/read 16-bit register\n");
	fprintf(stderr, "w32/r32 - write/read 32-bit register\n");
	fprintf(stderr, "w64/r64 - write/read 64-bit register\n");
	exit(1);
}