/*
 * plugin.h
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <stdbool.h>
#include "cve-check-tool.h"
#include "common.h"

#pragma once

typedef enum {
        PLUGIN_TYPE_MIN         = 1 << 0,
        PLUGIN_TYPE_PACKAGE     = 1 << 1,       /**<Handles source packages */
        PLUGIN_TYPE_REPORT      = 1 << 2,       /**<Handles output reports */
        PLUGIN_TYPE_MAX         = 1 << 3
} CvePluginType;

struct CvePlugin;


#define _module_export_ __attribute__ ((visibility("default")))

/** Report function */
typedef bool (*cve_plugin_report_func) (CveCheckTool *db);

/** Initialisation */
typedef bool (*cve_plugin_init)         (struct CvePlugin *self);

/** Plugin teardown **/
typedef void (*cve_plugin_destroy)      (struct CvePlugin *self);

/** PACKAGING */
typedef bool (*cve_plugin_is_patched)                (struct source_package_t* pkg, char* id);
typedef bool (*cve_plugin_is_ignored)                 (struct source_package_t* pkg, char* id);
typedef bool (*cve_plugin_is_package)                  (const char *path);
typedef SourcePackage *(*cve_plugin_scan_package)    (const char *path);
typedef SourcePackage *(*cve_plugin_scan_package_archive)    (const char *dir, const char *name, const char *version, const char *release);


typedef void (*cve_free_cb) (struct source_package_t*);

/** Encapsulate a plugin */
typedef struct CvePlugin {
        /* Plugin description */
        int flags;                              /**<Bitwise flags for this plugins support */
        const char *name;                      /**<Name of the plugin */
        void *handle;                           /**<dlopen handle */

        cve_plugin_destroy destroy;            /**<Destroy this plugin */

        /* Functions */
        cve_plugin_report_func report;         /**<Used by plugins supporting PLUGIN_OP_REPORT */

        cve_plugin_is_patched is_patched;      /**<Is this package patched? */
        cve_plugin_is_package is_package;      /**<Is this a supported package? */
        cve_plugin_scan_package scan_package;  /**<Scan a package, make it usable */
        cve_plugin_scan_package_archive scan_archive;  /**<Scan a package, make it usable */
        cve_plugin_is_ignored is_ignored;      /**<Is the given CVE ignored? */
        cve_free_cb free_package;
} CvePlugin;

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
