/*
 * cli.c - cve-check-tool helpers
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <errno.h>

#include "config.h"
#include "util.h"
#include "cve-check-tool.h"
#include "cli.h"
#include "plugin.h"

static bool cli_write_report(CveCheckTool *self)
{
        GHashTableIter iter;
        gchar *key = NULL;
        struct source_package_t *v = NULL;
        GList *c = NULL, *t = NULL;
        struct cve_entry_t *entry = NULL;
        FILE *fd = NULL;
        bool ret = false;

        if (self->output_file) {
                fd = fopen(self->output_file, "w");
                if (!fd) {
                        fprintf(stderr, "Unable to open %s for writing: %s\n", self->output_file, strerror(errno));
                        return false;
                }
        } else {
                fd = stdout;
        }

        g_hash_table_iter_init(&iter, self->db);
        while (g_hash_table_iter_next(&iter, (void**)&key, (void**)&v)) {
                if (!v->issues && !v->patched && !self->show_unaffected) {
                        continue;
                }
                if (!v->issues && self->hide_patched) {
                        continue;
                }
                if (fprintf(fd, "%s %s (%u patched, %u issues)\n"C_WHITE"------------"C_RESET"\n",
                        key, (char*)v->version, g_list_length(v->patched), g_list_length(v->issues)) < 0) {
                        goto io_error;
                }
                for (c = v->issues; c; c = c->next) {
                        entry = cve_db_get_cve(self->cve_db, (gchar*)c->data);
                        if (self->modified > 0 && entry->modified > self->modified) {
                                cve_free(entry);
                                continue;
                        }
                        if (fprintf(fd, " * "C_RED"%s"C_RESET" : %s\n\n", (char*)c->data, entry->summary) < 0) {
                                goto io_error;
                        }
                        /* Print links.. */
                        bool p = false;
                        for (t = entry->uris; t; t = t->next) {
                                if (fprintf(fd, "     - %s\n", (char*)t->data) < 0) {
                                        goto io_error;
                                }
                                p = true;
                        }
                        if (p) {
                                if (fprintf(fd, "\n") < 0) {
                                        goto io_error;
                                }
                        }
                        cve_free(entry);
                }
                if (!self->hide_patched) {
                        for (c = v->patched; c; c = c->next) {
                                entry = cve_db_get_cve(self->cve_db, (gchar*)c->data);
                                if (self->modified > 0 && entry->modified > self->modified) {
                                        cve_free(entry);
                                        continue;
                                }
                                if (fprintf(fd, " * "C_BLUE"%s [PATCHED]"C_RESET" : %s\n\n", (char*)c->data, entry->summary) < 0) {
                                        goto io_error;
                                }
                                cve_free(entry);
                        }
                }
                fputc('\n', fd);
                ret = true;
                goto success;
        }
        /* Nothing to write */
        ret = true;
        goto success;

io_error:
        fprintf(stderr, "Error writing to file: %s\n", strerror(errno));
success:
        if (fd != stdout && self->output_file) {
                fclose(fd);
        }

        return ret;
}


_module_export_ bool cve_plugin_module_init(CvePlugin *self)
{
        self->report = cli_write_report;
        self->flags = PLUGIN_TYPE_REPORT;
        self->name = "cli";
        return true;
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
