/*
 * template.h - cve-check-tool
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#pragma once

#include <glib.h>
#include <stdbool.h>

#include "util.h"

struct TemplateContext;

typedef struct TemplateContext TemplateContext;

/**
 * Short hand method to construct a TemplateContext from a GHashTable,
 * run all input through template_context_process_line, and return the
 * allocated string.
 *
 * @param original The string to perform replacements on
 * @param keys A string->string kv GHashTable
 * @return A newly allocated string with all replacements performed
 */
cve_string *template_string(const char *original, GHashTable *keys);

/**
 * Replace all {{style}} variables in a string with the string values
 * in the given TemplateContext, and child contexts
 * This method only supports string->string right now, however in future
 * support for lists will added.
 *
 * @note Section rendering is done on a second pass once the section has
 * been correctly built.
 *
 * @param original The string to perform replacements on
 * @param keys An initialised TemplateContext
 * @param skip Skip section expansion/rendering, default should be false.
 * @return A newly allocated string with all replacements performed
 */
cve_string *template_context_process_line(TemplateContext *context, const char *original, bool skip);

/**
 * Create a new TemplateContext
 */
TemplateContext *template_context_new(void);

/**
 * Free any resources used by a TemplateContext
 *
 * @param context Valid TemplateContext instance
 */
void template_context_free(TemplateContext *context);

/**
 * Add a key->value string pair to this context
 *
 * @param key Identifying string
 * @param val Value to pair with key
 *
 * return True if the operation succeeded
 */
bool template_context_add_string(TemplateContext *context, const char *key, char *val);

/**
 * Add a key->value string pair to this context
 *
 * @param key Identifying string
 * @param val Boolean value
 *
 * return True if the operation succeeded
 */
bool template_context_add_bool(TemplateContext *context, const char *key, bool value);

/**
 * Append to a named list
 *
 * @param key Identifying string
 * @param val Seeded TemplateContext
 *
 * return True if the operation succeeded
 */
bool template_context_add_list(TemplateContext *context, const char *key, TemplateContext *value);

/**
 * Add a named subcontext to this TemplateContext, enabling conditional
 * sections.
 *
 * @param ctx Root TemplateContext instance
 * @param key Unique name for the section and subcontext
 * @param child Child to add as a new subcontext
 */
void template_context_add_subcontext(TemplateContext *context, const char *key, TemplateContext *child);

DEF_AUTOFREE(TemplateContext, template_context_free)

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
