/*
 * pkgbuild.c - Arch Linux specific
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#define _GNU_SOURCE
#include <glib.h>
#include <libgen.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdlib.h>

#include "pkgbuild.h"
#include "util.h"
#include "plugin.h"

struct source_package_t *pkgbuild_inspect_spec(const char *filename)
{
        struct source_package_t *t = NULL;
        autofree(GFile) *fi = g_file_new_for_path(filename);
        autofree(GError) *error = NULL;
        if (!fi) {
                return NULL;
        }
        autofree(GFileInputStream) *fis = g_file_read(fi, NULL, &error);
        if (error) {
                g_printerr("Unable to read: %s\n", error->message);
                return NULL;
        }

        autofree(GDataInputStream) *dis = g_data_input_stream_new(G_INPUT_STREAM(fis));
        char *read = NULL;
        char *fpath = NULL;
        autofree(gchar) *name = NULL;
        autofree(gchar) *version = NULL;
        autofree(gchar) *release = NULL;

        while ((read = g_data_input_stream_read_line(dis, NULL, NULL, NULL)) != NULL) {
                autofree(gstrv) *strv = NULL;
                const gchar *key = NULL;
                autofree(gchar) *value = NULL;

                read = g_strstrip(read);

                if (!strchr(read, '=')) {
                        goto clean;
                }

                strv = g_strsplit(read, "=", -1);
                if (g_strv_length(strv) < 2) {
                        goto clean;
                }
                key = g_strstrip(strv[0]);
                value = g_strjoinv("=", strv+1);
                value = g_strstrip(value);

                if (g_str_equal(key, "pkgname")) {
                        name = g_strdup(value);
                } else if (g_str_equal(key, "pkgver")) {
                        version = g_strdup(value);
                } else if (g_str_equal(key, "pkgrel")) {
                        release = g_strdup(value);
                }

                if (name && version && release) {
                        g_free(read);
                        break;
                }
clean:
                g_free(read);
        }

        if (!name || !version || !release) {
                return NULL;
        }

        fpath = cve_get_file_parent(filename);
        if (!fpath) {
                return NULL;
        }

        t = calloc(1, sizeof(struct source_package_t));
        if (!t) {
                free(fpath);
                return NULL;
        }
        t->name = g_strdup(name);
        t->version = g_strdup(version);
        t->release = atoi(release);
        t->path = fpath;
        t->type = PACKAGE_TYPE_PKGBUILD;
        return t;
}

bool pkgbuild_is_patched(struct source_package_t *pkg, char *id)
{
        /* Determine if its patched. */
        autofree(gchar) *pnamet = g_ascii_strdown((gchar*)id, -1);
        autofree(gchar) *pnamel = g_strdup_printf("%s/%s.patch", pkg->path, pnamet);
        autofree(gchar) *pname = g_strdup_printf("%s/%s.patch", pkg->path, id);

        if (g_file_test(pname, G_FILE_TEST_EXISTS) || g_file_test(pnamel, G_FILE_TEST_EXISTS)) {
                return true;
        }

        return false;
}

bool pkgbuild_is_package(const char *filename)
{
        return g_str_has_suffix((const gchar*)filename, "PKGBUILD");
}

void pkgbuild_locate_sources(const char *directory, bool recurse, cve_add_callback cb)
{
        find_sources(directory, &pkgbuild_is_package, recurse, cb);
}

_module_export_ bool cve_plugin_module_init(CvePlugin *self)
{
        self->flags = PLUGIN_TYPE_PACKAGE;
        self->name = "pkgbuild";
        self->is_ignored = NULL;
        self->is_patched = pkgbuild_is_patched;
        self->is_package = pkgbuild_is_package;
        self->scan_package = pkgbuild_inspect_spec;
        return true;
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
