/*
 * This file is part of cve-check-tool
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#define _GNU_SOURCE
#include <check.h>
#include <stdlib.h>
#include <stdio.h>

#include "cve-string.c"
#include "util.h"
#include "util.c"
#include "hashmap.c"

#define TEST_SUITE_BUILD 1

#include "plugin-manager.c"

#include "config.h"

const char *nvd_file = "nvd.db";


static int add_count = 0;

void cve_add_package(__attribute__((unused)) const char *path)
{
        add_count++;
}

static inline void package_free(void *p, CvePlugin *pkg_plugin)
{
        if (!p) {
                return;
        }
        struct source_package_t *t = p;

        if (t->extra && pkg_plugin && pkg_plugin->free_package) {
                pkg_plugin->free_package(t);
                t->extra = NULL;
        }
        
        if (t->issues) { /* bless you */
                g_list_free_full(t->issues, xmlFree);
        }
        if (t->patched) {
                g_list_free_full(t->patched, xmlFree);
        }
        if (t->path) {
                free(t->path);
        }
        if (t->xml) {
                xmlFree((xmlChar*)t->name);
                xmlFree((xmlChar*)t->version);
        } else {
                g_free((gchar*)t->name);
                g_free((gchar*)t->version);
        }

        free(t);
}

/**
 * General RPM test
 */
START_TEST(cve_rpm_test)
{
        CvePlugin *plugin = cve_plugin_get_by_name("rpm");
        struct source_package_t *pkg = NULL;

        fail_if(!plugin, "rpm plugin not found!");

        pkg = plugin->scan_package(TOP_DIR "/tests/dummy_data/rpm/package_does_not_exist.spec");
        fail_if(pkg, "Incorrectly succeeded on non-existing spec!");

        pkg = plugin->scan_package(TOP_DIR "/tests/dummy_data/rpm/package2.spec");
        fail_if(!pkg, "Failed to inspect RPM spec!");
        package_free(pkg, plugin);
        pkg = NULL;

        pkg = plugin->scan_package(TOP_DIR "/tests/dummy_data/rpm/package.spec");
        fail_if(!pkg, "Failed to inspect RPM spec!");

        fail_if(!g_str_equal(pkg->name, "test-package"),
                "Invalid RPM package name");
        fail_if(!g_str_equal(pkg->version, "1.1.0"),
                "Invalid RPM package version");
        fail_if(pkg->release != 5, "Invalid RPM package release");

        fail_if(!plugin->is_patched(pkg, "CVE-2014-5461"),
                "Failed to detect RPM CVE patch");

        fail_if(!plugin->is_ignored(pkg, "CVE-2013-4459"),
                "Failed to detect ignored CVE");
        fail_if(plugin->is_ignored(pkg, "CVE-2013-0012"),
                "Incorrectly detected non-ignored CVE");

        package_free(pkg, plugin);
}
END_TEST

START_TEST(cve_srpm_test)
{
        gchar *c = NULL;
        CvePlugin *plugin = NULL;
        struct source_package_t *pkg = NULL;

        if (!(c = g_find_program_in_path("srpm"))) {
                fprintf(stderr, "Unable to perform SRPM tests!");
                return;
        }
        g_free(c);

        plugin = cve_plugin_get_by_name("srpm");

        fail_if(!plugin, "srpm plugin not found!");

        pkg = plugin->scan_archive(TOP_DIR "/tests/dummy_data/rpm/", "package", "1.", "5" );
        fail_if(!pkg, "Failed to inspect source RPM!");
        fail_if(!plugin->is_patched(pkg, "CVE-2014-5461"), "SRPM patch test failed");
        fail_if(!g_str_equal(pkg->name, "test-package"),
                "Invalid SRPM package name");
        fail_if(!g_str_equal(pkg->version, "1."),
                "Invalid SRPM package version");

        fail_if(plugin->is_ignored(pkg, "CVE-2013-0012"),
                "Incorrectly detected non-ignored CVE");

        package_free(pkg, plugin);

        pkg = plugin->scan_archive(TOP_DIR "/tests/dummy_data/rpm/", "invalid_package", "1.", "5" );
        fail_if(pkg, "Incorrectly succeeded at missing source RPM!");
}
END_TEST


/**
 * Solus eopkg test
 */
START_TEST(cve_eopkg_test)
{
        CvePlugin *plugin = cve_plugin_get_by_name("eopkg");
        struct source_package_t *pkg = NULL;

        fail_if(!plugin, "eopkg plugin not found!");

        pkg = plugin->scan_package(TOP_DIR "/tests/dummy_data/eopkg/not-exist");
        fail_if(pkg, "Non-existent eopkg should be NULL");

        pkg = plugin->scan_package(TOP_DIR "/tests/dummy_data/eopkg/pspec.xml");
        fail_if(!pkg, "Failed to inspect eopkg spec!");

        fail_if(!g_str_equal(pkg->name, "budgie-desktop"),
                "Invalid eopkg package name");
        fail_if(!g_str_equal(pkg->version, "8.1"),
                "Invalid eopkg package version");
        fail_if(pkg->release != 41, "Invalid eopkg package release");

        fail_if(!plugin->is_patched(pkg, "CVE-2014-5461"),
                "Failed to detect eopkg CVE patch");

        fail_if(!plugin->is_ignored(pkg, "CVE-2013-4459"),
                "Failed to detect ignored CVE");
        fail_if(plugin->is_ignored(pkg, "CVE-2013-0012"),
                "Incorrectly detected non-ignored CVE");

        package_free(pkg, plugin);
}
END_TEST

/**
 * Arch Linux pkgbuild test
 */
START_TEST(cve_pkgbuild_test)
{
        CvePlugin *plugin = cve_plugin_get_by_name("pkgbuild");
        struct source_package_t *pkg = NULL;

        fail_if(!plugin, "pkgbuild plugin not found!");

        pkg = plugin->scan_package(TOP_DIR "/tests/dummy_data/pkgbuild/not-exist");
        fail_if(pkg, "Non-existent PKGBUILD should be NULL");

        pkg = plugin->scan_package(TOP_DIR "/tests/dummy_data/pkgbuild/PKGBUILD");
        fail_if(!pkg, "Failed to inspect PKGBUILD spec!");

        fail_if(!g_str_equal(pkg->name, "my-test-package"),
                "Invalid PKGBUILD package name");
        fail_if(!g_str_equal(pkg->version, "1.0.3"),
                "Invalid PKGBUILD package version");
        fail_if(pkg->release != 10, "Invalid PKGBUILD package release");

        fail_if(!plugin->is_patched(pkg, "CVE-2014-5461"),
                "Failed to detect pkgbuild CVE patch");

        package_free(pkg, plugin);
}
END_TEST

static Suite *core_suite(void)
{
        Suite *s = NULL;
        TCase *tc = NULL;

        s = suite_create("cve_packaging");
        tc = tcase_create("cve_packaging_functions");
        tcase_add_test(tc, cve_rpm_test);
        tcase_add_test(tc, cve_srpm_test);
        tcase_add_test(tc, cve_eopkg_test);
        tcase_add_test(tc, cve_pkgbuild_test);
        suite_add_tcase(s, tc);

        return s;
}

int main(void)
{
        Suite *s;
        SRunner *sr;
        int fail;

        cve_plugin_manager_init();
        s = core_suite();
        sr = srunner_create(s);
        srunner_run_all(sr, CK_VERBOSE);
        fail = srunner_ntests_failed(sr);
        srunner_free(sr);

        cve_plugin_manager_destroy();

        if (fail > 0) {
                return EXIT_FAILURE;
        }

        return EXIT_SUCCESS;
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
