/*
 * core.h - cve-check-tool
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#pragma once

#include <libxml/xmlreader.h>
#include "common.h"

extern const char *nvd_file;        /* nvd.db */
extern const char *nvd_dir;         /* NVDS */

/**
 * A CVE Database, using the National Vulnerability Database as its source
 */
typedef struct CveDB CveDB;

typedef enum {
        CVE_DB_MIN = 1 << 0,
        CVE_DB_ID = 1 << 1,
        CVE_DB_PRODUCT = 1 << 2,
        CVE_DB_VENDOR = 1 << 3,
        CVE_DB_MODIFIED = 1 << 4,
        CVE_DB_LIKE = 1 << 5,
        CVE_DB_MAX = 1 << 6
} SearchType;

/**
 * Construct a new CveDB
 */
CveDB *cve_db_new(const char *path);

/**
 * Free an allocated CveDB
 *
 * @param db A valid CveDB instance
 */
void cve_db_free(CveDB *db);

/**
 * Finalize the DB initialisation
 */
bool cve_db_finalize(CveDB *db);

/**
 * Begin the DB initialisation
 */
bool cve_db_begin(CveDB *db);


/**
 * Load the NVD XML file into this DB
 *
 * @param db A valid CveDB instance
 * @param file Path to the NVD XML data
 *
 * @return a boolean value, indicating success
 */
bool cve_db_load(CveDB *db, const char *file);

/**
 * Get CVE data for the given ID
 *
 * @param db A valid CveDB instance
 * @param id The CVE-XXXX-XXXX ID
 *
 * @return A newly allocated data structure, or NULL if the CVE is not found
 */
struct cve_entry_t *cve_db_get_cve(CveDB *db, char *id);

/**
 * Retrieve a list of CVEs for the given parameters
 *
 * @param db A valid CveDB instance
 * @param product The CPE product name
 * @param version A version string to match against
 *
 * @return A newly allocated list of strings if found, otherwise NULL
 */
GList *cve_db_get_issues(CveDB *db, char *product, char *version);

/**
 * Util to free a struct cve_entry_t
 */
static inline void cve_free(void *v)
{
        if (!v) {
                return;
        }
        struct cve_entry_t *t = v;
        if (t->uris) {
                g_list_free_full(t->uris, xmlFree);
        }
        if (t->score) {
                g_free(t->score);
        }
        if (t->vector) {
                g_free(t->vector);
        }
        g_free(t->id);
        g_free(t->summary);
        g_free(t);
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
