/*
 * Driver for Renesas R-Car VIN
 *
 * Copyright (C) 2016-2017 Renesas Electronics Corp.
 * Copyright (C) 2011-2013 Renesas Solutions Corp.
 * Copyright (C) 2013 Cogent Embedded, Inc., <source@cogentembedded.com>
 * Copyright (C) 2008 Magnus Damm
 *
 * Based on the soc-camera rcar_vin driver
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */

#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_graph.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/sys_soc.h>
#include <linux/workqueue.h>

#include <media/v4l2-of.h>

#include "rcar-vin.h"

static const struct soc_device_attribute r8a7795es1[] = {
	{ .soc_id = "r8a7795", .revision = "ES1.*" },
	{ /* sentinel */ }
};

/* -----------------------------------------------------------------------------
 * Media Controller link notification
 */

static unsigned int rvin_group_csi_pad_to_chan(unsigned int pad)
{
	/*
	 * The CSI2 driver is rcar-csi2 and we know it's pad layout are
	 * 0: Source 1-4: Sinks so if we remove one from the pad we
	 * get the rcar-vin internal CSI2 channel number
	 */
	return pad - 1;
}

/* group lock should be held when calling this function */
static int rvin_group_entity_to_vin_num(struct rvin_group *group,
					struct media_entity *entity)
{
	struct video_device *vdev;
	int i;

	if (!is_media_entity_v4l2_video_device(entity))
		return -ENODEV;

	vdev = media_entity_to_video_device(entity);

	for (i = 0; i < RCAR_VIN_NUM; i++) {
		if (!group->vin[i])
			continue;

		if (&group->vin[i]->vdev == vdev)
			return i;
	}

	return -ENODEV;
}

/* group lock should be held when calling this function */
static int rvin_group_entity_to_csi_num(struct rvin_group *group,
					struct media_entity *entity)
{
	struct v4l2_subdev *sd;
	int i;

	if (!is_media_entity_v4l2_subdev(entity))
		return -ENODEV;

	sd = media_entity_to_v4l2_subdev(entity);

	for (i = 0; i < RVIN_CSI_MAX; i++)
		if (group->bridge[i].subdev == sd)
			return i;

	return -ENODEV;
}

/* group lock should be held when calling this function */
static void __rvin_group_build_link_list(struct rvin_group *group,
					 struct rvin_group_chsel *map,
					 int start, int len)
{
	struct media_pad *vin_pad, *remote_pad;
	unsigned int n;

	for (n = 0; n < len; n++) {
		map[n].csi = -1;
		map[n].chan = -1;

		if (!group->vin[start + n])
			continue;

		vin_pad = &group->vin[start + n]->vdev.entity.pads[RVIN_SINK];

		remote_pad = media_entity_remote_pad(vin_pad);
		if (!remote_pad)
			continue;

		map[n].csi =
			rvin_group_entity_to_csi_num(group, remote_pad->entity);
		map[n].chan = rvin_group_csi_pad_to_chan(remote_pad->index);
	}
}

/* group lock should be held when calling this function */
static int __rvin_group_try_get_chsel(struct rvin_group *group,
				      struct rvin_group_chsel *map,
				      int start, int len)
{
	const struct rvin_group_chsel *sel;
	unsigned int i, n;
	int chsel;

	for (i = 0; i < group->vin[start]->info->num_chsels; i++) {
		chsel = i;
		for (n = 0; n < len; n++) {

			/* If the link is not active it's OK */
			if (map[n].csi == -1)
				continue;

			/* Check if chsel match requested link */
			sel = &group->vin[start]->info->chsels[start + n][i];
			if (map[n].csi != sel->csi ||
			    map[n].chan != sel->chan) {
				chsel = -1;
				break;
			}
		}

		/* A chsel which satisfy the links have been found */
		if (chsel != -1)
			return chsel;
	}

	/* No chsel can satisfy the requested links */
	return -1;
}

/* group lock should be held when calling this function */
static bool rvin_group_in_use(struct rvin_group *group)
{
	struct media_entity *entity;

	media_device_for_each_entity(entity, &group->mdev)
		if (entity->use_count)
			return true;

	return false;
}

static int rvin_group_link_notify(struct media_link *link, u32 flags,
				  unsigned int notification)
{
	struct rvin_group *group = container_of(link->graph_obj.mdev,
						struct rvin_group, mdev);
	struct rvin_group_chsel chsel_map[4];
	int vin_num, vin_master, csi_num, csi_chan;
	unsigned int chsel;

	mutex_lock(&group->lock);

	vin_num = rvin_group_entity_to_vin_num(group, link->sink->entity);
	csi_num = rvin_group_entity_to_csi_num(group, link->source->entity);
	csi_chan = rvin_group_csi_pad_to_chan(link->source->index);

	/*
	 * Figure out which VIN node is the subgroup master.
	 *
	 * VIN0-3 are controlled by VIN0
	 * VIN4-7 are controlled by VIN4
	 */
	vin_master = vin_num < 4 ? 0 : 4;

	/* If not all devices exists something is horribly wrong */
	if (vin_num < 0 || csi_num < 0 || !group->vin[vin_master])
		goto error;

	/* Special checking only needed for links which are to be enabled */
	if (notification != MEDIA_DEV_NOTIFY_PRE_LINK_CH ||
	    !(flags & MEDIA_LNK_FL_ENABLED))
		goto out;

	/* If any link in the group are in use, no new link can be enabled */
	if (rvin_group_in_use(group))
		goto error;

	/* If the VIN already have a active link it's busy */
	if (media_entity_remote_pad(&link->sink->entity->pads[RVIN_SINK]))
		goto error;

	/* Build list of active links */
	__rvin_group_build_link_list(group, chsel_map, vin_master, 4);

	/* Add the new proposed link */
	chsel_map[vin_num - vin_master].csi = csi_num;
	chsel_map[vin_num - vin_master].chan = csi_chan;

	/* See if there is a chsel value which match our link selection */
	chsel = __rvin_group_try_get_chsel(group, chsel_map, vin_master, 4);

	/* No chsel can provide the request links */
	if (chsel == -1)
		goto error;

	/* Update chsel value at group master */
	if (rvin_set_chsel(group->vin[vin_master], chsel))
		goto error;

out:
	mutex_unlock(&group->lock);

	return v4l2_pipeline_link_notify(link, flags, notification);
error:
	mutex_unlock(&group->lock);

	return -EBUSY;
}


static const struct media_device_ops rvin_media_ops = {
	.link_notify = rvin_group_link_notify,
};

/* -----------------------------------------------------------------------------
 * Gen3 CSI2 Group Allocator
 */

static DEFINE_MUTEX(rvin_group_lock);
static struct rvin_group *rvin_group_data;

static void rvin_group_release(struct kref *kref)
{
	struct rvin_group *group =
		container_of(kref, struct rvin_group, refcount);

	mutex_lock(&rvin_group_lock);

	media_device_unregister(&group->mdev);
	media_device_cleanup(&group->mdev);

	rvin_group_data = NULL;

	mutex_unlock(&rvin_group_lock);

	kfree(group);
}

static struct rvin_group *__rvin_group_allocate(struct rvin_dev *vin)
{
	struct rvin_group *group;

	if (rvin_group_data) {
		group = rvin_group_data;
		kref_get(&group->refcount);
		vin_dbg(vin, "%s: get group=%p\n", __func__, group);
		return group;
	}

	group = kzalloc(sizeof(*group), GFP_KERNEL);
	if (!group)
		return NULL;

	kref_init(&group->refcount);
	rvin_group_data = group;

	vin_dbg(vin, "%s: alloc group=%p\n", __func__, group);
	return group;
}

static struct rvin_group *rvin_group_allocate(struct rvin_dev *vin)
{
	struct rvin_group *group;
	struct media_device *mdev;
	int ret;

	mutex_lock(&rvin_group_lock);

	group = __rvin_group_allocate(vin);
	if (!group) {
		mutex_unlock(&rvin_group_lock);
		return ERR_PTR(-ENOMEM);
	}

	/* Init group data if its not already initialized */
	mdev = &group->mdev;
	if (!mdev->dev) {
		mutex_init(&group->lock);
		mdev->dev = vin->dev;

		strlcpy(mdev->driver_name, "Renesas VIN",
			sizeof(mdev->driver_name));
		strlcpy(mdev->model, vin->dev->of_node->name,
			sizeof(mdev->model));
		strlcpy(mdev->bus_info, of_node_full_name(vin->dev->of_node),
			sizeof(mdev->bus_info));
		mdev->driver_version = LINUX_VERSION_CODE;
		media_device_init(mdev);

		mdev->ops = &rvin_media_ops;

		ret = media_device_register(mdev);
		if (ret) {
			vin_err(vin, "Failed to register media device\n");
			mutex_unlock(&rvin_group_lock);
			return ERR_PTR(ret);
		}
	}

	vin->v4l2_dev.mdev = mdev;

	mutex_unlock(&rvin_group_lock);

	return group;
}

static void rvin_group_delete(struct rvin_dev *vin)
{
	vin_dbg(vin, "%s: group=%p\n", __func__, &vin->group);
	kref_put(&vin->group->refcount, rvin_group_release);
}

/* -----------------------------------------------------------------------------
 * Subdevice helpers
 */

static int rvin_group_vin_to_csi(struct rvin_dev *vin)
{
	int i, vin_num, vin_master, chsel, csi;

	/* Only valid on Gen3 */
	if (vin->info->chip != RCAR_GEN3)
		return -1;

	/*
	 * Only try to translate to a CSI2 number if there is a enabled
	 * link from the VIN sink pad. However if there are no links at
	 * all we are at probe time so ignore the need for enabled links
	 * to be able to make a better guess of initial format
	 */
	if (vin->pads[RVIN_SINK].entity->num_links &&
	    !media_entity_remote_pad(&vin->pads[RVIN_SINK]))
		return -1;

	/* Find which VIN we are */
	vin_num = -1;
	for (i = 0; i < RCAR_VIN_NUM; i++)
		if (vin == vin->group->vin[i])
			vin_num = i;

	if (vin_num == -1)
		return -1;

	vin_master = vin_num < 4 ? 0 : 4;
	if (!vin->group->vin[vin_master])
		return -1;

	chsel = rvin_get_chsel(vin->group->vin[vin_master]);

	csi = vin->info->chsels[vin_num][chsel].csi;
	if (csi >= RVIN_CSI_MAX)
		return -1;

	if (!vin->group->source[csi].subdev || !vin->group->bridge[csi].subdev)
		return -1;

	return csi;
}

bool vin_have_bridge(struct rvin_dev *vin)
{
	return vin->digital.subdev == NULL;
}

struct rvin_graph_entity *vin_to_entity(struct rvin_dev *vin)
{
	int csi;

	/* If there is a digital subdev use it */
	if (vin->digital.subdev)
		return &vin->digital;

	csi = rvin_group_vin_to_csi(vin);
	if (csi < 0)
		return NULL;

	return &vin->group->source[csi];
}

struct v4l2_subdev *vin_to_source(struct rvin_dev *vin)
{
	int csi;

	/* If there is a digital subdev use it */
	if (vin->digital.subdev)
		return vin->digital.subdev;

	csi = rvin_group_vin_to_csi(vin);
	if (csi < 0)
		return NULL;

	return vin->group->source[csi].subdev;
}

struct v4l2_subdev *vin_to_bridge(struct rvin_dev *vin)
{
	int csi;

	if (vin->digital.subdev)
		return NULL;

	csi = rvin_group_vin_to_csi(vin);
	if (csi < 0)
		return NULL;

	return vin->group->bridge[csi].subdev;
}

/* -----------------------------------------------------------------------------
 * Async notifier helpers
 */

#define notifier_to_vin(n) container_of(n, struct rvin_dev, notifier)

static bool rvin_mbus_supported(struct rvin_graph_entity *entity)
{
	struct v4l2_subdev *sd = entity->subdev;
	struct v4l2_subdev_mbus_code_enum code = {
		.which = V4L2_SUBDEV_FORMAT_ACTIVE,
	};

	code.index = 0;
	code.pad = entity->source_pad_idx;
	while (!v4l2_subdev_call(sd, pad, enum_mbus_code, NULL, &code)) {
		code.index++;
		switch (code.code) {
		case MEDIA_BUS_FMT_YUYV8_1X16:
		case MEDIA_BUS_FMT_UYVY8_2X8:
		case MEDIA_BUS_FMT_UYVY10_2X10:
		case MEDIA_BUS_FMT_RGB888_1X24:
			entity->code = code.code;
			return true;
		default:
			break;
		}
	}

	/*
	 * Older versions where looking for the wrong media bus format.
	 * It where looking for a YUVY format but then treated it as a
	 * UYVY format. This was not noticed since atlest one subdevice
	 * used for testing (adv7180) reported a YUVY media bus format
	 * but provided UYVY data. There might be other unknown subdevices
	 * which also do this, to not break compatibility try to use them
	 * in legacy mode.
	 */
	code.index = 0;
	while (!v4l2_subdev_call(sd, pad, enum_mbus_code, NULL, &code)) {
		code.index++;
		switch (code.code) {
		case MEDIA_BUS_FMT_YUYV8_2X8:
			entity->code = MEDIA_BUS_FMT_UYVY8_2X8;
			return true;
		case MEDIA_BUS_FMT_YUYV10_2X10:
			entity->code = MEDIA_BUS_FMT_UYVY10_2X10;
			return true;
		default:
			break;
		}
	}

	return false;
}

static unsigned int rvin_pad_idx(struct v4l2_subdev *sd, int direction)
{
	unsigned int pad_idx;

	for (pad_idx = 0; pad_idx < sd->entity.num_pads; pad_idx++)
		if (sd->entity.pads[pad_idx].flags == direction)
			return pad_idx;

	return 0;
}

/* -----------------------------------------------------------------------------
 * Digital async notifier
 */

static int rvin_digital_notify_complete(struct v4l2_async_notifier *notifier)
{
	struct rvin_dev *vin = notifier_to_vin(notifier);
	struct v4l2_subdev *sd = vin->digital.subdev;
	int ret;

	/* Verify subdevices mbus format */
	if (!rvin_mbus_supported(&vin->digital)) {
		vin_err(vin, "Unsupported media bus format for %s\n",
			vin->digital.subdev->name);
		return -EINVAL;
	}

	vin_dbg(vin, "Found media bus format for %s: %d\n",
		vin->digital.subdev->name, vin->digital.code);

	/* Figure out source and sink pad ids */
	vin->digital.source_pad_idx = rvin_pad_idx(sd, MEDIA_PAD_FL_SOURCE);
	vin->digital.sink_pad_idx = rvin_pad_idx(sd, MEDIA_PAD_FL_SINK);

	vin_dbg(vin, "Found media pads for %s source: %d sink %d\n",
		vin->digital.subdev->name, vin->digital.source_pad_idx,
		vin->digital.sink_pad_idx);

	ret = v4l2_device_register_subdev_nodes(&vin->v4l2_dev);
	if (ret < 0) {
		vin_err(vin, "Failed to register subdev nodes\n");
		return ret;
	}

	return 0;
}

static void rvin_digital_notify_unbind(struct v4l2_async_notifier *notifier,
				       struct v4l2_subdev *subdev,
				       struct v4l2_async_subdev *asd)
{
	struct rvin_dev *vin = notifier_to_vin(notifier);

	if (vin->digital.subdev == subdev) {
		vin_dbg(vin, "unbind digital subdev %s\n", subdev->name);
		vin->digital.subdev = NULL;
		return;
	}

	vin_err(vin, "no entity for subdev %s to unbind\n", subdev->name);
}

static int rvin_digital_notify_bound(struct v4l2_async_notifier *notifier,
				     struct v4l2_subdev *subdev,
				     struct v4l2_async_subdev *asd)
{
	struct rvin_dev *vin = notifier_to_vin(notifier);

	v4l2_set_subdev_hostdata(subdev, vin);

	if (vin->digital.asd.match.of.node == subdev->dev->of_node) {
		vin_dbg(vin, "bound digital subdev %s\n", subdev->name);
		vin->digital.subdev = subdev;
		return 0;
	}

	vin_err(vin, "no entity for subdev %s to bind\n", subdev->name);
	return -EINVAL;
}

static int rvin_digitial_parse_v4l2(struct rvin_dev *vin,
				    struct device_node *ep,
				    struct v4l2_mbus_config *mbus_cfg)
{
	struct v4l2_of_endpoint v4l2_ep;
	int ret;

	ret = v4l2_of_parse_endpoint(ep, &v4l2_ep);
	if (ret) {
		vin_err(vin, "Could not parse v4l2 endpoint\n");
		return -EINVAL;
	}

	mbus_cfg->type = v4l2_ep.bus_type;

	switch (mbus_cfg->type) {
	case V4L2_MBUS_PARALLEL:
		vin_dbg(vin, "Found PARALLEL media bus\n");
		mbus_cfg->flags = v4l2_ep.bus.parallel.flags;
		break;
	case V4L2_MBUS_BT656:
		vin_dbg(vin, "Found BT656 media bus\n");
		mbus_cfg->flags = 0;
		break;
	default:
		vin_err(vin, "Unknown media bus type\n");
		return -EINVAL;
	}

	return 0;
}

static int rvin_digital_graph_parse(struct rvin_dev *vin)
{
	struct device_node *ep, *np;
	int ret;

	vin->digital.asd.match.of.node = NULL;
	vin->digital.subdev = NULL;

	/*
	 * Port 0 id 0 is local digital input, try to get it.
	 * Not all instances can or will have this, that is OK
	 */
	ep = of_graph_get_endpoint_by_regs(vin->dev->of_node, 0, 0);
	if (!ep)
		return 0;

	np = of_graph_get_remote_port_parent(ep);
	if (!np) {
		vin_err(vin, "No remote parent for digital input\n");
		of_node_put(ep);
		return -EINVAL;
	}
	of_node_put(np);

	ret = rvin_digitial_parse_v4l2(vin, ep, &vin->digital.mbus_cfg);
	of_node_put(ep);
	if (ret)
		return ret;

	vin->digital.asd.match.of.node = np;
	vin->digital.asd.match_type = V4L2_ASYNC_MATCH_OF;

	return 0;
}

static int rvin_digital_graph_init(struct rvin_dev *vin)
{
	struct v4l2_async_subdev **subdevs = NULL;
	int ret;

	ret = rvin_digital_graph_parse(vin);
	if (ret)
		return ret;

	if (!vin->digital.asd.match.of.node) {
		vin_dbg(vin, "No digital subdevice found\n");
		return -ENODEV;
	}

	/* Register the subdevices notifier. */
	subdevs = devm_kzalloc(vin->dev, sizeof(*subdevs), GFP_KERNEL);
	if (subdevs == NULL)
		return -ENOMEM;

	subdevs[0] = &vin->digital.asd;

	vin_dbg(vin, "Found digital subdevice %s\n",
		of_node_full_name(subdevs[0]->match.of.node));

	vin->notifier.num_subdevs = 1;
	vin->notifier.subdevs = subdevs;
	vin->notifier.bound = rvin_digital_notify_bound;
	vin->notifier.unbind = rvin_digital_notify_unbind;
	vin->notifier.complete = rvin_digital_notify_complete;

	ret = v4l2_async_notifier_register(&vin->v4l2_dev, &vin->notifier);
	if (ret < 0) {
		vin_err(vin, "Digital notifier registration failed\n");
		return ret;
	}

	return 0;
}

/* -----------------------------------------------------------------------------
 * CSI async notifier
 */

/* group lock should be held when calling this function */
static void rvin_group_update_pads(struct rvin_graph_entity *entity)
{
	struct media_entity *ent = &entity->subdev->entity;
	unsigned int i;

	/* Make sure source pad idx are sane */
	if (entity->source_pad_idx >= ent->num_pads ||
	    ent->pads[entity->source_pad_idx].flags != MEDIA_PAD_FL_SOURCE) {
		entity->source_pad_idx =
			rvin_pad_idx(entity->subdev, MEDIA_PAD_FL_SOURCE);
	}

	/* Try to find sink for source, fall back 0 which always is sink */
	entity->sink_pad_idx = 0;
	for (i = 0; i < ent->num_pads; ++i) {
		struct media_pad *sink = &ent->pads[i];

		if (!(sink->flags & MEDIA_PAD_FL_SINK))
			continue;

		if (sink->index == entity->source_pad_idx)
			continue;

		if (media_entity_has_route(ent, sink->index,
					   entity->source_pad_idx))
			entity->sink_pad_idx = sink->index;
	}
}

/* group lock should be held when calling this function */
static int rvin_group_add_link(struct rvin_dev *vin,
			       struct media_entity *source,
			       unsigned int source_pad_idx,
			       struct media_entity *sink,
			       unsigned int sink_idx,
			       u32 flags)
{
	struct media_pad *source_pad, *sink_pad;
	int ret = 0;

	source_pad = &source->pads[source_pad_idx];
	sink_pad = &sink->pads[sink_idx];

	if (!media_entity_find_link(source_pad, sink_pad))
		ret = media_create_pad_link(source, source_pad_idx,
					    sink, sink_idx, flags);

	if (ret)
		vin_err(vin, "Error adding link from %s to %s\n",
			source->name, sink->name);

	return ret;
}

static int rvin_group_update_links(struct rvin_dev *vin)
{
	struct media_entity *source, *sink;
	struct rvin_dev *master;
	unsigned int i, n, idx, chsel, csi;
	u32 flags;
	int ret;

	mutex_lock(&vin->group->lock);

	/* Update Source -> Bridge */
	for (i = 0; i < RVIN_CSI_MAX; i++) {
		if (!vin->group->source[i].subdev)
			continue;

		if (!vin->group->bridge[i].subdev)
			continue;

		source = &vin->group->source[i].subdev->entity;
		sink = &vin->group->bridge[i].subdev->entity;
		idx = vin->group->source[i].source_pad_idx;
		flags = MEDIA_LNK_FL_ENABLED | MEDIA_LNK_FL_IMMUTABLE;

		ret = rvin_group_add_link(vin, source, idx, sink, 0, flags);
		if (ret)
			goto out;
	}

	/* Update Bridge -> VIN */
	for (n = 0; n < RCAR_VIN_NUM; n++) {

		/* Check that VIN is part of the group */
		if (!vin->group->vin[n])
			continue;

		/* Check that subgroup master is part of the group */
		master = vin->group->vin[n < 4 ? 0 : 4];
		if (!master)
			continue;

		chsel = rvin_get_chsel(master);

		for (i = 0; i < vin->info->num_chsels; i++) {
			csi = vin->info->chsels[n][i].csi;

			/* If the CSI is out of bounds it's a no operate skip */
			if (csi >= RVIN_CSI_MAX)
				continue;

			/* Check that bridge are part of the group */
			if (!vin->group->bridge[csi].subdev)
				continue;

			source = &vin->group->bridge[csi].subdev->entity;
			sink = &vin->group->vin[n]->vdev.entity;
			idx = vin->info->chsels[n][i].chan + 1;
			flags = i == chsel ? MEDIA_LNK_FL_ENABLED : 0;

			ret = rvin_group_add_link(vin, source, idx, sink, 0,
						  flags);
			if (ret)
				goto out;
		}
	}
out:
	mutex_unlock(&vin->group->lock);

	return ret;
}

static int rvin_group_notify_complete(struct v4l2_async_notifier *notifier)
{
	struct rvin_dev *vin = notifier_to_vin(notifier);
	unsigned int i;
	int ret;

	mutex_lock(&vin->group->lock);
	for (i = 0; i < RVIN_CSI_MAX; i++) {
		if (!vin->group->source[i].subdev)
			continue;

		rvin_group_update_pads(&vin->group->source[i]);

		if (!rvin_mbus_supported(&vin->group->source[i])) {
			vin_err(vin, "Unsupported media bus format for %s\n",
				vin->group->source[i].subdev->name);
			mutex_unlock(&vin->group->lock);
			return -EINVAL;
		}
	}
	mutex_unlock(&vin->group->lock);

	ret = v4l2_device_register_subdev_nodes(&vin->v4l2_dev);
	if (ret) {
		vin_err(vin, "Failed to register subdev nodes\n");
		return ret;
	}

	return rvin_group_update_links(vin);
}

static void rvin_group_notify_unbind(struct v4l2_async_notifier *notifier,
				     struct v4l2_subdev *subdev,
				     struct v4l2_async_subdev *asd)
{
	struct rvin_dev *vin = notifier_to_vin(notifier);
	unsigned int i;

	if (!subdev->dev)
		return;

	mutex_lock(&vin->group->lock);
	for (i = 0; i < RVIN_CSI_MAX; i++) {
		struct device_node *del = subdev->dev->of_node;

		if (vin->group->bridge[i].asd.match.of.node == del) {
			vin_dbg(vin, "Unbind bridge %s\n", subdev->name);
			vin->group->bridge[i].subdev = NULL;
			mutex_unlock(&vin->group->lock);
			return;
		}

		if (vin->group->source[i].asd.match.of.node == del) {
			vin_dbg(vin, "Unbind source %s\n", subdev->name);
			vin->group->source[i].subdev = NULL;
			mutex_unlock(&vin->group->lock);
			return;
		}
	}
	mutex_unlock(&vin->group->lock);

	vin_err(vin, "No entity for subdev %s to unbind\n", subdev->name);
}

static int rvin_group_notify_bound(struct v4l2_async_notifier *notifier,
				   struct v4l2_subdev *subdev,
				   struct v4l2_async_subdev *asd)
{
	struct rvin_dev *vin = notifier_to_vin(notifier);
	unsigned int i;

	v4l2_set_subdev_hostdata(subdev, vin);

	mutex_lock(&vin->group->lock);
	for (i = 0; i < RVIN_CSI_MAX; i++) {
		struct device_node *new = subdev->dev->of_node;

		if (vin->group->bridge[i].asd.match.of.node == new) {
			vin_dbg(vin, "Bound bridge %s\n", subdev->name);
			vin->group->bridge[i].subdev = subdev;
			mutex_unlock(&vin->group->lock);
			return 0;
		}

		if (vin->group->source[i].asd.match.of.node == new) {
			vin_dbg(vin, "Bound source %s\n", subdev->name);
			vin->group->source[i].subdev = subdev;
			mutex_unlock(&vin->group->lock);
			return 0;
		}
	}
	mutex_unlock(&vin->group->lock);

	vin_err(vin, "No entity for subdev %s to bind\n", subdev->name);
	return -EINVAL;
}

static int rvin_group_parse_v4l2(struct rvin_dev *vin,
				 struct device_node *ep,
				 struct v4l2_mbus_config *mbus_cfg)
{
	struct v4l2_of_endpoint v4l2_ep;
	int ret;

	ret = v4l2_of_parse_endpoint(ep, &v4l2_ep);
	if (ret) {
		vin_err(vin, "Could not parse v4l2 endpoint\n");
		return -EINVAL;
	}

	if (v4l2_ep.bus_type != V4L2_MBUS_CSI2) {
		vin_err(vin, "Unsupported media bus type for %s\n",
			of_node_full_name(ep));
		return -EINVAL;
	}

	mbus_cfg->type = v4l2_ep.bus_type;
	mbus_cfg->flags = v4l2_ep.bus.mipi_csi2.flags;

	return 0;
}

static int rvin_group_vin_num_from_bridge(struct rvin_dev *vin,
					  struct device_node *node,
					  int test)
{
	struct device_node *remote;
	struct of_endpoint endpoint;
	int num;

	remote = of_parse_phandle(node, "remote-endpoint", 0);
	if (!remote)
		return -EINVAL;

	of_graph_parse_endpoint(remote, &endpoint);
	of_node_put(remote);

	num = endpoint.id;

	if (test != -1 && num != test) {
		vin_err(vin, "VIN numbering error at %s, was %d now %d\n",
			of_node_full_name(node), test, num);
		return -EINVAL;
	}

	return num;
}

static struct device_node *rvin_group_get_bridge(struct rvin_dev *vin,
						 struct device_node *node)
{
	struct device_node *bridge;

	bridge = of_graph_get_remote_port_parent(node);
	if (!bridge) {
		vin_err(vin, "No bridge found %s\n", of_node_full_name(node));
		return ERR_PTR(-EINVAL);
	}

	/* Not all bridges are available, this is OK */
	if (!of_device_is_available(bridge)) {
		vin_dbg(vin, "Bridge %s not available\n",
			of_node_full_name(bridge));
		of_node_put(bridge);
		return NULL;
	}

	return bridge;
}

static struct device_node *
rvin_group_get_source(struct rvin_dev *vin,
		      struct device_node *bridge,
		      struct v4l2_mbus_config *mbus_cfg,
		      unsigned int *remote_pad)
{
	struct device_node *source, *ep, *rp;
	struct of_endpoint endpoint;
	int ret;

	ep = of_graph_get_endpoint_by_regs(bridge, 0, 0);
	if (!ep) {
		vin_dbg(vin, "Endpoint %s not connected to source\n",
			of_node_full_name(ep));
		return ERR_PTR(-EINVAL);
	}

	/* Check that source uses a supported media bus */
	ret = rvin_group_parse_v4l2(vin, ep, mbus_cfg);
	if (ret) {
		of_node_put(ep);
		return ERR_PTR(ret);
	}

	rp = of_graph_get_remote_port(ep);
	of_graph_parse_endpoint(rp, &endpoint);
	of_node_put(rp);
	*remote_pad = endpoint.id;

	source = of_graph_get_remote_port_parent(ep);
	of_node_put(ep);
	if (!source) {
		vin_err(vin, "No source found for endpoint '%s'\n",
			of_node_full_name(ep));
		return ERR_PTR(-EINVAL);
	}

	return source;
}

/* group lock should be held when calling this function */
static int rvin_group_graph_parse(struct rvin_dev *vin, unsigned long *bitmap)
{
	struct device_node *ep, *bridge, *source;
	unsigned int i, remote_pad;
	int vin_num = -1;

	*bitmap = 0;

	for (i = 0; i < RVIN_CSI_MAX; i++) {

		/* Check if instance is connected to the bridge */
		ep = of_graph_get_endpoint_by_regs(vin->dev->of_node, 1, i);
		if (!ep) {
			vin_dbg(vin, "Bridge: %d not connected\n", i);
			continue;
		}

		vin_num = rvin_group_vin_num_from_bridge(vin, ep, vin_num);
		if (vin_num < 0) {
			of_node_put(ep);
			return vin_num;
		}

		if (vin->group->bridge[i].asd.match.of.node) {
			of_node_put(ep);
			vin_dbg(vin, "Bridge: %d handled by other device\n", i);
			continue;
		}

		bridge = rvin_group_get_bridge(vin, ep);
		of_node_put(ep);
		if (IS_ERR(bridge))
			return PTR_ERR(bridge);
		if (bridge == NULL)
			continue;

		source = rvin_group_get_source(vin, bridge,
					       &vin->group->source[i].mbus_cfg,
					       &remote_pad);
		of_node_put(bridge);
		if (IS_ERR(source))
			return PTR_ERR(source);
		if (source == NULL)
			continue;

		of_node_put(source);

		vin->group->bridge[i].asd.match.of.node = bridge;
		vin->group->bridge[i].asd.match_type = V4L2_ASYNC_MATCH_OF;
		vin->group->source[i].asd.match.of.node = source;
		vin->group->source[i].asd.match_type = V4L2_ASYNC_MATCH_OF;
		vin->group->source[i].source_pad_idx = remote_pad;

		*bitmap |= BIT(i);

		vin_dbg(vin, "Handle bridge %s and source %s pad %d\n",
			of_node_full_name(bridge), of_node_full_name(source),
			remote_pad);
	}

	/* Insert ourself in the group */
	vin_dbg(vin, "I'm VIN number %d", vin_num);
	if (vin->group->vin[vin_num] != NULL) {
		vin_err(vin, "VIN number %d already occupied\n", vin_num);
		return -EINVAL;
	}
	vin->group->vin[vin_num] = vin;
	vin->index = vin_num;

	return 0;
}

/* group lock should be held when calling this function */
static void rvin_group_graph_revert(struct rvin_dev *vin, unsigned long bitmap)
{
	int bit;

	for_each_set_bit(bit, &bitmap, RVIN_CSI_MAX) {
		vin_dbg(vin, "Reverting graph for %s\n",
			of_node_full_name(vin->dev->of_node));
		vin->group->bridge[bit].asd.match.of.node = NULL;
		vin->group->bridge[bit].asd.match_type = 0;
		vin->group->source[bit].asd.match.of.node = NULL;
		vin->group->source[bit].asd.match_type = 0;
	}
}

static int rvin_group_graph_init(struct rvin_dev *vin)
{
	struct v4l2_async_subdev **subdevs = NULL;
	unsigned long bitmap;
	int i, bit, count, ret;

	mutex_lock(&vin->group->lock);

	ret = rvin_group_graph_parse(vin, &bitmap);
	if (ret) {
		rvin_group_graph_revert(vin, bitmap);
		mutex_unlock(&vin->group->lock);
		return ret;
	}

	/* Check if instance need to handle subdevices on behalf of the group */
	count = hweight_long(bitmap) * 2;
	if (!count) {
		mutex_unlock(&vin->group->lock);
		return 0;
	}

	subdevs = devm_kzalloc(vin->dev, sizeof(*subdevs) * count, GFP_KERNEL);
	if (subdevs == NULL) {
		rvin_group_graph_revert(vin, bitmap);
		mutex_unlock(&vin->group->lock);
		return -ENOMEM;
	}

	i = 0;
	for_each_set_bit(bit, &bitmap, RVIN_CSI_MAX) {
		subdevs[i++] = &vin->group->bridge[bit].asd;
		subdevs[i++] = &vin->group->source[bit].asd;
	}

	vin_dbg(vin, "Claimed %d subdevices for group\n", count);

	vin->notifier.num_subdevs = count;
	vin->notifier.subdevs = subdevs;
	vin->notifier.bound = rvin_group_notify_bound;
	vin->notifier.unbind = rvin_group_notify_unbind;
	vin->notifier.complete = rvin_group_notify_complete;

	mutex_unlock(&vin->group->lock);

	ret = v4l2_async_notifier_register(&vin->v4l2_dev, &vin->notifier);
	if (ret < 0) {
		vin_err(vin, "Group notifier registration failed\n");
		return ret;
	}

	return 0;
}

/* -----------------------------------------------------------------------------
 * Platform Device Driver
 */

static const struct rvin_info rcar_info_h1 = {
	.chip = RCAR_H1,
	.max_width = 2048,
	.max_height = 2048,
};

static const struct rvin_info rcar_info_m1 = {
	.chip = RCAR_M1,
	.max_width = 2048,
	.max_height = 2048,
};

static const struct rvin_info rcar_info_r8a7795_es1x = {
	.chip = RCAR_GEN3,
	.max_width = 4096,
	.max_height = 4096,

	.num_chsels = 6,
	.chsels = {
		{
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI21, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI21, .chan = 0 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI21, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI21, .chan = 1 },
		}, {
			{ .csi = RVIN_CSI21, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 2 },
			{ .csi = RVIN_CSI20, .chan = 2 },
			{ .csi = RVIN_CSI21, .chan = 2 },
		}, {
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI21, .chan = 1 },
			{ .csi = RVIN_CSI40, .chan = 3 },
			{ .csi = RVIN_CSI20, .chan = 3 },
			{ .csi = RVIN_CSI21, .chan = 3 },
		}, {
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI21, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI21, .chan = 0 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI21, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI21, .chan = 1 },
		}, {
			{ .csi = RVIN_CSI21, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 2 },
			{ .csi = RVIN_CSI20, .chan = 2 },
			{ .csi = RVIN_CSI21, .chan = 2 },
		}, {
			{ .csi = RVIN_CSI41, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI21, .chan = 1 },
			{ .csi = RVIN_CSI41, .chan = 3 },
			{ .csi = RVIN_CSI20, .chan = 3 },
			{ .csi = RVIN_CSI21, .chan = 3 },
		},
	},
};

static const struct rvin_info rcar_info_r8a7795 = {
	.chip = RCAR_GEN3,
	.max_width = 4096,
	.max_height = 4096,

	.num_chsels = 5,
	.chsels = {
		{
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 2 },
			{ .csi = RVIN_CSI20, .chan = 2 },
		}, {
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI40, .chan = 3 },
			{ .csi = RVIN_CSI20, .chan = 3 },
		}, {
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 1 },
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 1 },
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI41, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI41, .chan = 2 },
			{ .csi = RVIN_CSI20, .chan = 2 },
		}, {
			{ .csi = RVIN_CSI41, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_CSI41, .chan = 3 },
			{ .csi = RVIN_CSI20, .chan = 3 },
		},
	},
};

static const struct rvin_info rcar_info_r8a7796 = {
	.chip = RCAR_GEN3,
	.max_width = 4096,
	.max_height = 4096,

	.num_chsels = 5,
	.chsels = {
		{
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_NOOPE, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_NOOPE, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
		}, {
			{ .csi = RVIN_NOOPE, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 2 },
			{ .csi = RVIN_CSI20, .chan = 2 },
		}, {
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_NOOPE, .chan = 1 },
			{ .csi = RVIN_CSI40, .chan = 3 },
			{ .csi = RVIN_CSI20, .chan = 3 },
		}, {
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_NOOPE, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
		}, {
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_NOOPE, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
		}, {
			{ .csi = RVIN_NOOPE, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 0 },
			{ .csi = RVIN_CSI20, .chan = 0 },
			{ .csi = RVIN_CSI40, .chan = 2 },
			{ .csi = RVIN_CSI20, .chan = 2 },
		}, {
			{ .csi = RVIN_CSI40, .chan = 1 },
			{ .csi = RVIN_CSI20, .chan = 1 },
			{ .csi = RVIN_NOOPE, .chan = 1 },
			{ .csi = RVIN_CSI40, .chan = 3 },
			{ .csi = RVIN_CSI20, .chan = 3 },
		},
	},
};

static const struct rvin_info rcar_info_gen2 = {
	.chip = RCAR_GEN2,
	.max_width = 2048,
	.max_height = 2048,
};

static const struct of_device_id rvin_of_id_table[] = {
	{
		.compatible = "renesas,vin-r8a7795",
		.data = &rcar_info_r8a7795,
	},
	{
		.compatible = "renesas,vin-r8a7796",
		.data = &rcar_info_r8a7796,
	},
	{
		.compatible = "renesas,vin-r8a7794",
		.data = &rcar_info_gen2,
	},
	{
		.compatible = "renesas,vin-r8a7793",
		.data = &rcar_info_gen2,
	},
	{
		.compatible = "renesas,vin-r8a7791",
		.data = &rcar_info_gen2,
	},
	{
		.compatible = "renesas,vin-r8a7790",
		.data = &rcar_info_gen2,
	},
	{
		.compatible = "renesas,vin-r8a7779",
		.data = &rcar_info_h1,
	},
	{
		.compatible = "renesas,vin-r8a7778",
		.data = &rcar_info_m1,
	},
	{
		.compatible = "renesas,rcar-gen2-vin",
		.data = &rcar_info_gen2,
	},
	{ },
};
MODULE_DEVICE_TABLE(of, rvin_of_id_table);

static int rvin_graph_init(struct rvin_dev *vin)
{
	int ret;

	/* Try to get digital video pipe */
	ret = rvin_digital_graph_init(vin);

	/* No digital pipe and we are on Gen3 try to joint CSI2 group */
	if (ret == -ENODEV && vin->info->chip == RCAR_GEN3) {

		vin->pads[RVIN_SINK].flags = MEDIA_PAD_FL_SINK;
		ret = media_entity_pads_init(&vin->vdev.entity, 1, vin->pads);
		if (ret)
			return ret;

		vin->group = rvin_group_allocate(vin);
		if (IS_ERR(vin->group))
			return PTR_ERR(vin->group);

		ret = rvin_group_graph_init(vin);
		if (ret)
			return ret;

		ret = rvin_group_update_links(vin);
		if (ret)
			return ret;
	}

	return ret;
}

static int rcar_vin_probe(struct platform_device *pdev)
{
	const struct of_device_id *match;
	struct rvin_dev *vin;
	struct resource *mem;
	int irq, ret;

	vin = devm_kzalloc(&pdev->dev, sizeof(*vin), GFP_KERNEL);
	if (!vin)
		return -ENOMEM;

	match = of_match_device(of_match_ptr(rvin_of_id_table), &pdev->dev);
	if (!match)
		return -ENODEV;

	vin->dev = &pdev->dev;

	vin->info = match->data;
	vin->last_input = NULL;

	if (soc_device_match(r8a7795es1))
		vin->info = &rcar_info_r8a7795_es1x;

	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (mem == NULL)
		return -EINVAL;

	vin->base = devm_ioremap_resource(vin->dev, mem);
	if (IS_ERR(vin->base))
		return PTR_ERR(vin->base);

	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return irq;

	ret = rvin_dma_probe(vin, irq);
	if (ret)
		return ret;

	ret = rvin_graph_init(vin);
	if (ret < 0)
		goto error;

	ret = rvin_v4l2_probe(vin);
	if (ret)
		goto error;

	pm_suspend_ignore_children(&pdev->dev, true);
	pm_runtime_enable(&pdev->dev);

	vin->work_queue = create_singlethread_workqueue(dev_name(vin->dev));
	if (!vin->work_queue) {
		ret = -ENOMEM;
		goto error;
	}
	INIT_DELAYED_WORK(&vin->rvin_resume, rvin_resume_start_streaming);

	platform_set_drvdata(pdev, vin);

	return 0;
error:
	rvin_dma_remove(vin);

	return ret;
}

static int rcar_vin_remove(struct platform_device *pdev)
{
	struct rvin_dev *vin = platform_get_drvdata(pdev);

	pm_runtime_disable(&pdev->dev);
	cancel_delayed_work_sync(&vin->rvin_resume);
	destroy_workqueue(vin->work_queue);

	rvin_v4l2_remove(vin);

	v4l2_async_notifier_unregister(&vin->notifier);

	if (vin->group)
		rvin_group_delete(vin);

	rvin_dma_remove(vin);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int rcar_vin_suspend(struct device *dev)
{
	struct rvin_dev *vin = dev_get_drvdata(dev);
	int ret;

	if ((vin->info->chip == RCAR_GEN3) &&
		((vin->index == 0) || (vin->index == 4)))
		vin->chsel = rvin_get_chsel(vin);

	if (vin->state != STALLED)
		return 0;

	ret = rvin_suspend_stop_streaming(vin);

	pm_runtime_put(vin->dev);

	return ret;
}

static int rcar_vin_resume(struct device *dev)
{
	struct rvin_dev *vin = dev_get_drvdata(dev);
	int ret;

	if ((vin->info->chip == RCAR_GEN3) &&
		((vin->index == 0) || (vin->index == 4)))
		rvin_set_chsel(vin, vin->chsel);

	if (vin->state != STALLED)
		return 0;

	pm_runtime_get_sync(vin->dev);
	queue_delayed_work(vin->work_queue, &vin->rvin_resume,
				msecs_to_jiffies(CONNECTION_TIME));

	return ret;
}

static SIMPLE_DEV_PM_OPS(rcar_vin_pm_ops,
			rcar_vin_suspend, rcar_vin_resume);
#define DEV_PM_OPS (&rcar_vin_pm_ops)
#else
#define DEV_PM_OPS NULL
#endif /* CONFIG_PM_SLEEP */

static struct platform_driver rcar_vin_driver = {
	.driver = {
		.name = "rcar-vin",
		.pm		= DEV_PM_OPS,
		.of_match_table = rvin_of_id_table,
	},
	.probe = rcar_vin_probe,
	.remove = rcar_vin_remove,
};

module_platform_driver(rcar_vin_driver);

MODULE_AUTHOR("Niklas Söderlund <niklas.soderlund@ragnatech.se>");
MODULE_DESCRIPTION("Renesas R-Car VIN camera host driver");
MODULE_LICENSE("GPL v2");
