// SPDX-License-Identifier: GPL-2.0
/*
 * AVIRT - ALSA Virtual Soundcard
 *
 * Copyright (c) 2010-2018 Fiberdyne Systems Pty Ltd
 * 
 * avirt.h - AVIRT system-level header
 */

#ifndef __SOUND_AVIRT_H
#define __SOUND_AVIRT_H

#include <sound/core.h>
#include <sound/pcm.h>
#include <linux/configfs.h>

#define MAX_STREAMS 16
#define MAX_NAME_LEN 80

#define DINFO(logname, fmt, args...)                                           \
	snd_printk(KERN_INFO "AVIRT: %s: " fmt "\n", logname, ##args)

#define DERROR(logname, fmt, args...)                                          \
	snd_printk(KERN_ERR "AVIRT: %s: " fmt "\n", logname, ##args)

#define DDEBUG(logname, fmt, args...)                                          \
	snd_printk(KERN_DEBUG "AVIRT: %s: " fmt "\n", logname, ##args)

struct snd_avirt_stream_array; /* Forward declaration */

/**
 * AVIRT Audio Path configure function type
 * Each Audio Path registers this at snd_avirt_audiopath_register time.
 * It is then called by the core once AVIRT has been configured
 */
typedef int (*snd_avirt_audiopath_configure)(
	struct snd_card *card, struct snd_avirt_stream_array *stream_array);

/**
 * AVIRT Audio Path info
 */
struct snd_avirt_audiopath {
	const char *uid; /* Unique identifier */
	const char *name; /* Pretty name */
	unsigned int version[3]; /* Version - Major.Minor.Ext */
	const struct snd_pcm_hardware *hw; /* ALSA PCM HW conf */
	const struct snd_pcm_ops *pcm_playback_ops; /* ALSA PCM playback ops */
	const struct snd_pcm_ops *pcm_capture_ops; /* ALSA PCM capture ops */
	snd_avirt_audiopath_configure configure; /* Config callback function */

	void *context;
};

/**
 * Audio stream configuration
 */
struct snd_avirt_stream {
	char name[MAX_NAME_LEN]; /* Stream name */
	char map[MAX_NAME_LEN]; /* Stream Audio Path mapping */
	unsigned int channels; /* Stream channel count */
	unsigned int device; /* Stream PCM device no. */
	unsigned int direction; /* Stream direction */
	struct snd_pcm *pcm; /* ALSA PCM  */
	struct config_item item; /* configfs item reference */
};

/**
 * Private Data Expansion
 */
struct snd_avirt_private_data {
	struct snd_avirt_audiopath *audiopath;
	void *ap_private_data;

	void (*ap_private_free)(struct snd_pcm *pcm);
};

/**
 * Audio stream group
 */
struct snd_avirt_stream_array {
	struct snd_avirt_stream *streams[MAX_STREAMS];
	int count;
};

/**
 * snd_avirt_audiopath_register - register Audio Path with AVIRT
 * @audiopath: Audio Path to be registered
 * @return: 0 on success or error code otherwise
 */
int snd_avirt_audiopath_register(struct snd_avirt_audiopath *audiopath);

/**
 * snd_avirt_audiopath_deregister - deregister Audio Path with AVIRT
 * @audiopath: Audio Path to be deregistered
 * @return: 0 on success or error code otherwise
 */
int snd_avirt_audiopath_deregister(struct snd_avirt_audiopath *audiopath);

/**
 * snd_avirt_pcm_period_elapsed - PCM buffer complete callback
 * @substream: pointer to ALSA PCM substream
 *
 * This should be called from a child Audio Path once it has finished processing
 * the PCM buffer
 */
void snd_avirt_pcm_period_elapsed(struct snd_pcm_substream *substream);

#endif // __SOUND_AVIRT_H
