/*
 * Copyright (c) 2018 TOYOTA MOTOR CORPORATION
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <algorithm>
#include "hs-clientmanager.h"
#include "hmi-debug.h"

static const char _homescreen[] = "homescreen";

HS_ClientManager* HS_ClientManager::me = nullptr;

static void cbRemoveClientCtxt(void *data)
{
    HS_ClientManager::instance()->removeClientCtxt(data);
}

/**
 * HS_ClientManager construction function
 *
 * #### Parameters
 *  - Nothing
 *
 * #### Return
 * None
 *
 */
HS_ClientManager::HS_ClientManager()
{
}

/**
 * get instance
 *
 * #### Parameters
 *  - Nothing
 *
 * #### Return
 * HS_ClientManager instance pointer
 *
 */
HS_ClientManager* HS_ClientManager::instance(void)
{
    if(me == nullptr)
        me = new HS_ClientManager();

    return me;
}

/**
 * HS_ClientManager init function
 *
 * #### Parameters
 *  - Nothing
 *
 * #### Return
 * init result
 *
 */
int HS_ClientManager::init(void)
{
    HMI_NOTICE("homescreen-service","called.");
}

/**
 * create client's afb_req_context
 *
 * #### Parameters
 *  - appid: app's id
 *
 * #### Return
 * HS_ClientCtxt pointer
 *
 */
HS_ClientCtxt* HS_ClientManager::createClientCtxt(afb_req_t req, std::string appid)
{
    HS_ClientCtxt *ctxt = (HS_ClientCtxt *)afb_req_context_get(req);
    if (!ctxt)
    {
        HMI_NOTICE("homescreen-service", "create new session for %s", appid.c_str());
        HS_ClientCtxt *ctxt = new HS_ClientCtxt(appid.c_str());
        afb_req_session_set_LOA(req, 1);
        afb_req_context_set(req, ctxt, cbRemoveClientCtxt);
    }
    return ctxt;
}

/**
 * add Client
 *
 * #### Parameters
 *  - ctxt: app's id
 *
 * #### Return
 * HS_Client pointer
 *
 */
HS_Client* HS_ClientManager::addClient(afb_req_t req, std::string appid)
{
    return (client_list[appid] = new HS_Client(req, appid));
}

/**
 * remove Client
 *
 * #### Parameters
 *  - appid: app's id
 *
 * #### Return
 * None
 *
 */
void HS_ClientManager::removeClient(std::string appid)
{
    delete client_list[appid];
    client_list.erase(appid);
}

/**
 * remove Client from list
 *
 * #### Parameters
 *  - data: HS_ClientCtxt pointer
 *
 * #### Return
 * None
 *
 */
void HS_ClientManager::removeClientCtxt(void *data)
{
    HS_ClientCtxt *ctxt = (HS_ClientCtxt *)data;
    if(ctxt == nullptr)
    {
        HMI_ERROR("homescreen-service", "data is nullptr");
        return;
    }

    HMI_NOTICE("homescreen-service", "remove app %s", ctxt->id.c_str());
    std::lock_guard<std::mutex> lock(this->mtx);
    removeClient(ctxt->id);
    delete appid2ctxt[ctxt->id];
    appid2ctxt.erase(ctxt->id);
}

/**
 * tap_shortcut
 *
 * #### Parameters
 *  - request: the request to bindings
 *
 * #### Return
 * result
 *
 */
int HS_ClientManager::tap_shortcut(afb_req_t request)
{
    int ret = 0;
    const char* value = afb_req_value(request, _application_id);
    if (value) {
        HMI_NOTICE("homescreen-service","request params = %s.", value);
        std::lock_guard<std::mutex> lock(this->mtx);
        auto ip = client_list.find(std::string(value));
        if(ip != client_list.end()) {
            ip->second->tap_shortcut(value);
        }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input application_id");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * on_screen_message
 *
 * #### Parameters
 *  - request: the request to bindings
 *
 * #### Return
 * result
 *
 */
int HS_ClientManager::on_screen_message(afb_req_t request)
{
    int ret = 0;
    const char* value = afb_req_value(request, _display_message);
    if (value) {
        HMI_NOTICE("homescreen-service","request params = %s.", value);
        std::lock_guard<std::mutex> lock(this->mtx);
        for(auto m : client_list) {
            m.second->on_screen_message(request, value);
        }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input display_message");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * on_screen_reply
 *
 * #### Parameters
 *  - request: the request to bindings
 *
 * #### Return
 * result
 *
 */
int HS_ClientManager::on_screen_reply(afb_req_t request)
{
    int ret = 0;
    const char* value = afb_req_value(request, _reply_message);
    if (value) {
      HMI_NOTICE("homescreen-service","request params = %s.", value);
      std::lock_guard<std::mutex> lock(this->mtx);
      for(auto m : client_list) {
        m.second->on_screen_reply(request, value);
      }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input reply_message");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * subscribe
 *
 * #### Parameters
 *  - request: the request to bindings
 *
 * #### Return
 * result
 *
 */
int HS_ClientManager::subscribe(afb_req_t request)
{
    int ret = 0;
    const char *value = afb_req_value(request, "event");
    HMI_NOTICE("homescreen-service","value is %s", value);
    if(value) {
        std::string appid =std::move(get_application_id(request));
        if(appid.empty()) {
            HMI_NOTICE("homescreen-service","can't get application identifier");
            return AFB_REQ_GETAPPLICATIONID_ERROR;
        }

        std::lock_guard<std::mutex> lock(this->mtx);
        HS_Client* client = nullptr;
        auto ip = client_list.find(appid);
        if(ip != client_list.end()) {
            client = client_list[appid];
        }
        else {
            appid2ctxt[appid] = createClientCtxt(request, appid);
            client = addClient(request, appid);
        }

        if(client->subscribe(request, value) != 0) {
            HMI_NOTICE("homescreen-service","subscribe failed");
            ret = AFB_REQ_SUBSCRIBE_ERROR;
        }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input event name");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * unsubscribe
 *
 * #### Parameters
 *  - request: the request to bindings
 *
 * #### Return
 * result
 *
 */
int HS_ClientManager::unsubscribe(afb_req_t request)
{
    const char *value = afb_req_value(request, "event");
    HMI_NOTICE("homescreen-service","value is %s", value);
    int ret = 0;
    if(value) {
        std::string appid = std::move(get_application_id(request));
        if(appid.empty()) {
            HMI_NOTICE("homescreen-service","can't get application identifier");
            return AFB_REQ_GETAPPLICATIONID_ERROR;
        }

        std::lock_guard<std::mutex> lock(this->mtx);
        auto ip = client_list.find(appid);
        if(ip != client_list.end()
        && ip->second->unsubscribe(request, value) != 0) {
            HMI_NOTICE("homescreen-service","unsubscribe failed");
            ret = AFB_REQ_UNSUBSCRIBE_ERROR;
        }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input event name");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * showWindow event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_ClientManager::showWindow(afb_req_t request)
{
    int ret = 0;
    const char* value = afb_req_value(request, _application_id);
    if (value) {
        HMI_NOTICE("homescreen-service","request params = %s.", value);
        std::lock_guard<std::mutex> lock(this->mtx);
        auto ip = client_list.find(std::string(value));
        if(ip != client_list.end()) {
            ret = ip->second->showWindow(request, value);
        }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input application_id");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * hideWindow event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_ClientManager::hideWindow(afb_req_t request)
{
    int ret = 0;
    const char* value = afb_req_value(request, _application_id);
    if (value) {
        HMI_NOTICE("homescreen-service","request params = %s.", value);
        std::lock_guard<std::mutex> lock(this->mtx);
        auto ip = client_list.find(std::string(value));
        if(ip != client_list.end()) {
            ret = ip->second->hideWindow(request);
        }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input application_id");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * replyShowWindow event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_ClientManager::replyShowWindow(afb_req_t request)
{
    int ret = 0;
    const char* value = afb_req_value(request, _application_id);
    if (value) {
        HMI_NOTICE("homescreen-service","request params = %s.", value);
        std::lock_guard<std::mutex> lock(this->mtx);
        auto ip = client_list.find(std::string(value));
        if(ip != client_list.end()) {
            ret = ip->second->replyShowWindow(request, value);
        }
    }
    else {
        HMI_NOTICE("homescreen-service","Please input application_id");
        ret = AFB_EVENT_BAD_REQUEST;
    }
    return ret;
}

/**
 * showNotification event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_ClientManager::showNotification(afb_req_t request)
{
    int ret = 0;
    std::lock_guard<std::mutex> lock(this->mtx);
    auto ip = client_list.find(_homescreen);
    if(ip != client_list.end()) {
        ret = ip->second->showNotification(request);
    }
    else {
        HMI_NOTICE("homescreen-service","not exist sessiion with homescreen");
        ret = AFB_REQ_SHOWNOTIFICATION_ERROR;
    }

    return ret;
}

/**
 * showInformation event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_ClientManager::showInformation(afb_req_t request)
{
    int ret = 0;
    std::lock_guard<std::mutex> lock(this->mtx);
    auto ip = client_list.find(_homescreen);
    if(ip != client_list.end()) {
        ret = ip->second->showInformation(request);
    }
    else {
        HMI_NOTICE("homescreen-service","not exist sessiion with homescreen");
        ret = AFB_REQ_SHOWINFORMATION_ERROR;
    }

    return ret;
}
