/*
 * Copyright (C) 2015, 2016, 2017 "IoT.bzh"
 * Author: José Bollo <jose.bollo@iot.bzh>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#define _GNU_SOURCE

#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <json-c/json.h>

#define AFB_BINDING_VERSION 3
#include <afb/afb-binding.h>

#include "agl-forgerock.h"

static afb_event_t event;
static json_object *current_identity;

/* default vin corresponds to Toyota Camry 2016, 429 */
static const char default_vin[] = "4T1BF1FK5GU260429";
static char *vin;

/***** configuration ********************************************/

static struct json_object *readjson(int fd)
{
	char *buffer;
	struct stat s;
	json_object *result = NULL;
	int rc;

	rc = fstat(fd, &s);
	if (rc == 0 && S_ISREG(s.st_mode)) {
		buffer = alloca((size_t)(s.st_size)+1);
		if (read(fd, buffer, (size_t)s.st_size) == (ssize_t)s.st_size) {
			buffer[s.st_size] = 0;
			//AFB_API_NOTICE(afbBindingV3root, "Config file: %s", buffer);
			result = json_tokener_parse(buffer);
			//if (!result)
			//{
			//	AFB_API_ERROR(afbBindingV3root, "Config file is not a valid JSON: %s", json_tokener_error_desc(json_tokener_get_error(NULL)));
			//}
		}
	}
	close(fd);

	return result;
}

static struct json_object *get_global_config(const char *name, const char *locale)
{
	int fd = afb_daemon_rootdir_open_locale(name, O_RDONLY, locale);
	if (fd < 0) AFB_API_ERROR(afbBindingV3root, "Config file not found: %s", name);
	return fd < 0 ? NULL : readjson(fd);
}

static struct json_object *get_local_config(const char *name)
{
	int fd = openat(AT_FDCWD, name, O_RDONLY, 0);
	return fd < 0 ? NULL : readjson(fd);
}

static void confsetstr(struct json_object *conf, const char *name,
		       char **value, const char *def)
{
	struct json_object *v;
	const char *s;
	char *p;

	s = conf && json_object_object_get_ex(conf, name, &v) ? json_object_get_string(v) : def;
	p = *value;
	if (s && p != s) {
		*value = strdup(s);
		free(p);
	}
}

static void setconfig(struct json_object *conf)
{
	if (conf) {
		confsetstr(conf, "vin", &vin, vin ? : default_vin);
		agl_forgerock_setconfig(conf);
	}
}

static void readconfig()
{
	setconfig(get_global_config("etc/config.json", NULL));
	setconfig(get_local_config("/etc/agl/identity-agent-config.json"));
	setconfig(get_local_config("config.json"));
}

/****************************************************************/

static struct json_object *make_event_object(const char *name, const char *id, const char *nick)
{
	struct json_object *object = json_object_new_object();

	/* TODO: errors */
	json_object_object_add(object, "eventName", json_object_new_string(name));
	json_object_object_add(object, "accountid", json_object_new_string(id));
	if (nick)
		json_object_object_add(object, "nickname", json_object_new_string(nick));
	return object;
}

static int send_event_object(const char *name, const char *id, const char *nick)
{
	return afb_event_push(event, make_event_object(name, id, nick));
}

static void do_login(struct json_object *desc)
{
	if (current_identity == NULL && desc == NULL) return; // Switching from NULL to NULL -> do nothing
	if (current_identity && desc)
	{
		const char* a = json_object_to_json_string(current_identity);
		const char* b = json_object_to_json_string(desc);
		if (strcmp(a, b) == 0)
		{
			AFB_API_NOTICE(afbBindingV3root, "Reloging of the same user.");
			return; // Switching from one user to the same user -> do nothing
		}
	}

	struct json_object *object;

	/* switching the user */
	AFB_API_INFO(afbBindingV3root, "Switching to user %s",
		     desc ? json_object_to_json_string(desc) : "null");
	object = current_identity;
	current_identity = json_object_get(desc);
	json_object_put(object);

	if (!json_object_object_get_ex(desc, "name", &object))
		object = 0;
	send_event_object("login", !object ? "null" : json_object_get_string(object)? : "?", 0);
}

static void do_logout()
{
	struct json_object *object;

	AFB_API_INFO(afbBindingV3root, "Switching to no user");
	object = current_identity;
	current_identity = 0;
	json_object_put(object);

	send_event_object("logout", "null", 0);
}

static void on_forgerock_data(struct json_object *data, const char *error)
{
	if (error) {
		AFB_API_ERROR(afbBindingV3root, "Can't get data: %s", error);
	} else {
		do_login(data);
	}
}

/****************************************************************/

static void subscribe (afb_req_t request)
{
	if (afb_req_subscribe(request, event)) {
		AFB_REQ_ERROR(request, "subscribe error");
		afb_req_reply(request, NULL,  "failed", "subscribtion failed");
	}
	else
		afb_req_reply(request, NULL, NULL, NULL);
}

static void unsubscribe (afb_req_t request)
{
	afb_req_unsubscribe(request, event);
	afb_req_reply(request, NULL, NULL, NULL);
}

static void logout (afb_req_t request)
{
	do_logout();
	afb_req_reply(request, NULL, NULL, NULL);
}

static void fake_login (afb_req_t request)
{
	struct json_object *desc = afb_req_json(request);
	do_logout();
	if (desc)
		do_login(desc);
	afb_req_reply(request, NULL, NULL, NULL);
}

static void get (afb_req_t request)
{
	afb_req_reply(request, json_object_get(current_identity), NULL, NULL);
}

static void on_nfc_subscribed(void *closure, struct json_object *result,
			      const char *error, const char *info,
			      afb_api_t api)
{
	if (error)
		AFB_API_ERROR(api, "Failed to subscribe to nfc events.");
	else
		AFB_API_DEBUG(api, "Subscribed to nfc events.");
}

static int service_init(afb_api_t api)
{
	struct json_object *jrequest = NULL;

	agl_forgerock_setcb(on_forgerock_data);

	event = afb_api_make_event(api, "event");
	if (!afb_event_is_valid(event)) {
		AFB_API_ERROR(api, "Failed to create event");
		return -1;
	}
	
	readconfig();

	if (afb_api_require_api(api, "nfc", 1))
		return -1;

	if (afb_api_require_api(api, "persistence", 1))
		return -1;

	jrequest = json_object_new_object();
	json_object_object_add(jrequest, "value", json_object_new_string("presence"));

	afb_api_call(api, "nfc", "subscribe", jrequest, on_nfc_subscribed, NULL);

	return 0;
}

static void on_nfc_target_add(struct json_object *object)
{
	struct json_object * json_uid;
	const char *uid;

	if (json_object_object_get_ex(object, "uid", &json_uid))
	{
		uid = json_object_get_string(json_uid);
		AFB_API_NOTICE(afbBindingV3root,
			       "nfc tag detected, call forgerock with vincode=%s and key=%s",
			       vin ? vin : default_vin, uid);
		send_event_object("incoming", uid, uid);
		agl_forgerock_download_request(vin ? vin : default_vin, "nfc", uid);
	}
	else AFB_API_ERROR(afbBindingV3root,
			   "nfc target add event is received but no UID found: %s",
			   json_object_to_json_string(object));
}

static void onevent(afb_api_t api, const char *event, struct json_object *object)
{
	AFB_API_NOTICE(api, "Received event: %s", event);
	if (!strcmp("nfc/presence", event))
	{
		on_nfc_target_add(object);
		return;
	}
	AFB_API_WARNING(api, "Unhandled event: %s", event);
}

static void fake_auth(afb_req_t req)
{
	struct json_object* req_object;
	struct json_object* kind_object;
	struct json_object* key_object;

	req_object = afb_req_json(req);

	if ((!json_object_object_get_ex(req_object, "kind", &kind_object)) ||
	    (!json_object_object_get_ex(req_object, "key", &key_object)))
	{
		AFB_REQ_ERROR(req, "bad request: %s",
			      json_object_get_string(req_object));
		afb_req_reply(req, NULL, "Missing arg", NULL);
	}
	else {
		const char* kind = json_object_get_string(kind_object);
		const char* key = json_object_get_string(key_object);

		AFB_REQ_NOTICE(req, "kind: %s, key: %s", kind, key);
		
		send_event_object("incoming", key, key);
		agl_forgerock_download_request(vin ? vin : default_vin, kind, key);

		afb_req_reply(req, NULL, NULL, "fake auth success!");
	}
}

/*
 * NOTE: this sample does not use session to keep the test as basic as possible
 *       in real application most APIs should be protected with AFB_SESSION_CHECK
 */
const afb_verb_t verbs[]=
{
	{.verb = "subscribe"  , .callback = subscribe    , .auth = NULL,
	 .info = "subscribe to events"     , .session = AFB_SESSION_NONE },
	{.verb = "unsubscribe", .callback = unsubscribe  , .auth = NULL,
	 .info = "unsubscribe to events"   , .session = AFB_SESSION_NONE },
	{.verb = "fake-login" , .callback = fake_login   , .auth = NULL,
	 .info = "fake a login"            , .session = AFB_SESSION_NONE },
	{.verb = "logout"     , .callback = logout       , .auth = NULL,
	 .info = "log the current user out", .session = AFB_SESSION_NONE },
	{.verb = "get"        , .callback = get          , .auth = NULL,
	 .info = "get data"                , .session = AFB_SESSION_NONE },
	{.verb = "fake-auth"  , .callback = fake_auth    , .auth = NULL,
	 .info = "fake an authentication"  , .session = AFB_SESSION_NONE },
	{NULL}
};

const afb_binding_t afbBindingExport =
{
	.api = "identity",
	.specification = NULL,
	.info = "AGL identity service",
	.verbs = verbs,
	.preinit = NULL,
	.init = service_init,
	.onevent = onevent,
	.noconcurrency = 0
};

/* vim: set colorcolumn=80: */
