/*
 * Copyright 2019-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *     http://aws.amazon.com/apache2.0/
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package com.amazon.aace.apl.config;

import com.amazon.aace.core.config.EngineConfiguration;

/**
 * A factory interface for creating APL configuration objects.
 */
public class APLConfiguration {
    private static final String TAG = "APLConfiguration";

    public enum AlexaPresentationTimeoutType {
        /**
         *  RenderDocument timeout in ms for display card timeout.
         *  @hideinitializer
         */
        DISPLAY_DOCUMENT_INTERACTION_IDLE_TIMEOUT(
                "DISPLAY_DOCUMENT_INTERACTION_IDLE_TIMEOUT", "displayCardAudioPlaybackStoppedPausedTimeout");

        /**
         * @internal
         */
        private String mName;

        /**
         * @internal
         */
        private String mKey;

        /**
         * Type used to identify a AlexaPresentation configuration type and value pair.
         */
        AlexaPresentationTimeoutType(String name, String key) {
            mName = name;
            mKey = key;
        }

        /**
         * @internal
         */
        public String toString() {
            return mName;
        }

        /**
         * @internal
         */
        public String getKey() {
            return mKey;
        }
    }

    public static class AlexaPresentationTimeout {
        private AlexaPresentationTimeoutType mType;
        private int mValue;

        public AlexaPresentationTimeout(AlexaPresentationTimeoutType type, int value) {
            mType = type;
            mValue = value;
        }

        public AlexaPresentationTimeoutType getType() {
            return mType;
        }
        public int getValue() {
            return mValue;
        }
    }

    /**
     * Factory method used to programmatically generate Alexa Presentation configuration data.
     * This is an optional configuration. Following are the accepted keys and their description.
     * - displayDocumentInteractionIdleTimeout If present, the timeout in ms for display card timeout.
     * The data generated by this method is equivalent to providing the following JSON
     * values in a configuration file:
     *
     * @code{.json}
     * {
     *   "alexaPresentationCapabilityAgent": {
     *      "displayDocumentInteractionIdleTimeout": <TIMEOUT_IN_MS>
     *   }
     * }
     * @endcode
     *
     * @param timeoutList A list of @c AlexaPresentationTimeout type and value pairs.
     *
     */
    public static EngineConfiguration createAlexaPresentationTimeoutConfig(
            final AlexaPresentationTimeout[] timeoutList) {
        return new EngineConfiguration() {
            @Override
            protected long createNativeRef() {
                return createAlexaPresentationTimeoutConfigBinder(timeoutList);
            }
        };
    }

    // Native Engine JNI methods
    static private native long createAlexaPresentationTimeoutConfigBinder(AlexaPresentationTimeout[] timeoutList);
}
