# Copyright (C) 2020 Konsulko Group
# Author: Edi Feschiyan
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import asyncio
import os
import pytest

from pyagl.services.base import AFBResponse, AFBT
from pyagl.services.bluetooth import BluetoothService as BTS
import logging

logger = logging.getLogger(f'pytest-{BTS.service}')
logger.setLevel(logging.DEBUG)
pytestmark = [pytest.mark.asyncio, pytest.mark.bluetooth]


@pytest.fixture(scope='module')
def event_loop():
    loop = asyncio.get_event_loop()
    yield loop


@pytest.fixture(scope='module')
async def service():
    address = os.environ.get('AGL_TGT_IP', 'localhost')
    port = os.environ.get('AGL_TGT_PORT', None)

    bts = await BTS(ip=address, port=port)
    yield bts
    await bts.websocket.close()


@pytest.mark.xfail
@pytest.fixture(scope='module')
def btaddr():
    bthtestaddr = os.environ.get('AGL_BT_TEST_ADDR', None)
    if not bthtestaddr:
        pytest.xfail('No test bluetooth test address set in environment variables; please export AGL_BT_TEST_ADDR in '
                     'DBus address style as AGL_BT_TEST_ADDR=dev_A1_02_B3_04_C5_00')

    return bthtestaddr


@pytest.mark.dependency
async def test_default_adapter(event_loop, service: BTS):
    msgid = await service.default_adapter('hci0')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info
    assert 'adapter' in resp.data.keys()
    assert resp.data['adapter'] == 'hci0'


async def test_subscribe_device_changes(event_loop, service: BTS):
    msgid = await service.subscribe('device_changes')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info


async def test_unsubscribe_device_changes(event_loop, service: BTS):
    msgid = await service.unsubscribe('device_changes')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info
    

async def test_subscribe_adapter_changes(event_loop, service: BTS):
    msgid = await service.subscribe('adapter_changes')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info


async def test_unsubscribe_adapter_changes(event_loop, service: BTS):
    msgid = await service.unsubscribe('adapter_changes')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info


async def test_subscribe_media(event_loop, service: BTS):
    msgid = await service.subscribe('media')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info


async def test_unsubscribe_media(event_loop, service: BTS):
    msgid = await service.unsubscribe('media')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info


async def test_subscribe_agent(event_loop, service: BTS):
    msgid = await service.subscribe('agent')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info


async def test_unsubscribe_agent(event_loop, service: BTS):
    msgid = await service.unsubscribe('agent')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info

@pytest.mark.dependency(depends=['test_default_adapter'])
async def test_managed_objects(event_loop, service: BTS):
    msgid = await service.managed_objects()
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info

@pytest.mark.hwrequired
async def test_has_single_adapter(event_loop, service: BTS):
    msgid = await service.managed_objects()
    resp = await service.afbresponse()
    assert len(resp.data['adapters']) == 1, \
        f'Detected {len(resp.data["adapters"])} adapters. Multiple adapters may also affect testing'


@pytest.mark.hwrequired
@pytest.mark.dependency(depends=['test_default_adapter'])
async def test_adapter_state(event_loop, service: BTS):
    msgid = await service.adapter_state('hci0')
    resp = await service.afbresponse()
    assert resp.status == 'success', 'adapter state verb failed'


@pytest.mark.hwrequired
async def test_pairing_verb(event_loop, service: BTS, btaddr):
    msgid = await service.pair(btaddr)
    resp = await service.afbresponse()
    assert msgid == resp.msgid
    assert resp.status == 'success', f'pair verb failed - {resp.info}'


@pytest.mark.hwrequired
@pytest.mark.dependency
async def test_connect_verb(event_loop, service: BTS, btaddr):
    msgid = await service.connect(btaddr)
    resp = await service.afbresponse()
    assert msgid == resp.msgid
    assert resp.status == 'success', f'connect verb failed - {resp.info}'


@pytest.mark.hwrequired
@pytest.mark.xfail(reason='This is expected to fail because there has to be an ongoing pairing attempt')
async def test_confirm_pairing_verb(event_loop, service: BTS, btaddr):
    msgid = await service.confirm_pairing(pincode='123456')
    resp = await service.afbresponse()
    assert msgid == resp.msgid
    assert resp.status == 'success', f'confirm_pairing verb failed - {resp.info}'


@pytest.mark.hwrequired
@pytest.mark.dependency(depends=['test_connect_verb'])
async def test_avrcp_controls(event_loop, service: BTS):
    avrcp_actions = ['Play', 'Pause', 'Stop', 'Next', 'Previous']
    for a in avrcp_actions:
        msgid = await service.avrcp_controls(a)
        resp = await service.afbresponse()
        assert msgid == resp.msgid
        assert resp.status == 'success', f'avrcp_control verb failed with action {a} - {resp.info}'


@pytest.mark.hwrequired
async def test_disconnect_verb(event_loop, service: BTS, btaddr):
    msgid = await service.disconnect(btaddr)
    resp = await service.afbresponse()
    assert msgid == resp.msgid
    assert resp.status == 'success', f'disconnect verb failed - {resp.info}'


@pytest.mark.hwrequired
async def test_remove_pairing_verb(event_loop, service: BTS, btaddr):
    msgid = await service.remove_device(btaddr)
    resp = await service.afbresponse()
    assert msgid == resp.msgid
    assert resp.status == 'success'
