"use strict";
/**
 * Copyright 2017 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const EventEmitter = require("events");
const mime = require("mime");
const Events_1 = require("./Events");
const Connection_1 = require("./Connection");
const Dialog_1 = require("./Dialog");
const EmulationManager_1 = require("./EmulationManager");
const FrameManager_1 = require("./FrameManager");
const Input_1 = require("./Input");
const Tracing_1 = require("./Tracing");
const helper_1 = require("./helper");
const Coverage_1 = require("./Coverage");
const Worker_1 = require("./Worker");
const JSHandle_1 = require("./JSHandle");
const Accessibility_1 = require("./Accessibility");
const TimeoutSettings_1 = require("./TimeoutSettings");
const writeFileAsync = helper_1.helper.promisify(fs.writeFile);
const paperFormats = {
    letter: { width: 8.5, height: 11 },
    legal: { width: 8.5, height: 14 },
    tabloid: { width: 11, height: 17 },
    ledger: { width: 17, height: 11 },
    a0: { width: 33.1, height: 46.8 },
    a1: { width: 23.4, height: 33.1 },
    a2: { width: 16.54, height: 23.4 },
    a3: { width: 11.7, height: 16.54 },
    a4: { width: 8.27, height: 11.7 },
    a5: { width: 5.83, height: 8.27 },
    a6: { width: 4.13, height: 5.83 },
};
class Page extends EventEmitter {
    constructor(client, target, ignoreHTTPSErrors, screenshotTaskQueue) {
        super();
        this._closed = false;
        this._timeoutSettings = new TimeoutSettings_1.TimeoutSettings();
        this._pageBindings = new Map();
        this._javascriptEnabled = true;
        this._workers = new Map();
        // TODO: improve this typedef - it's a function that takes a file chooser or something?
        this._fileChooserInterceptors = new Set();
        this._client = client;
        this._target = target;
        this._keyboard = new Input_1.Keyboard(client);
        this._mouse = new Input_1.Mouse(client, this._keyboard);
        this._touchscreen = new Input_1.Touchscreen(client, this._keyboard);
        this._accessibility = new Accessibility_1.Accessibility(client);
        this._frameManager = new FrameManager_1.FrameManager(client, this, ignoreHTTPSErrors, this._timeoutSettings);
        this._emulationManager = new EmulationManager_1.EmulationManager(client);
        this._tracing = new Tracing_1.Tracing(client);
        this._coverage = new Coverage_1.Coverage(client);
        this._screenshotTaskQueue = screenshotTaskQueue;
        this._viewport = null;
        client.on('Target.attachedToTarget', event => {
            if (event.targetInfo.type !== 'worker') {
                // If we don't detach from service workers, they will never die.
                client.send('Target.detachFromTarget', {
                    sessionId: event.sessionId
                }).catch(helper_1.debugError);
                return;
            }
            const session = Connection_1.Connection.fromSession(client).session(event.sessionId);
            const worker = new Worker_1.Worker(session, event.targetInfo.url, this._addConsoleMessage.bind(this), this._handleException.bind(this));
            this._workers.set(event.sessionId, worker);
            this.emit(Events_1.Events.Page.WorkerCreated, worker);
        });
        client.on('Target.detachedFromTarget', event => {
            const worker = this._workers.get(event.sessionId);
            if (!worker)
                return;
            this.emit(Events_1.Events.Page.WorkerDestroyed, worker);
            this._workers.delete(event.sessionId);
        });
        this._frameManager.on(Events_1.Events.FrameManager.FrameAttached, event => this.emit(Events_1.Events.Page.FrameAttached, event));
        this._frameManager.on(Events_1.Events.FrameManager.FrameDetached, event => this.emit(Events_1.Events.Page.FrameDetached, event));
        this._frameManager.on(Events_1.Events.FrameManager.FrameNavigated, event => this.emit(Events_1.Events.Page.FrameNavigated, event));
        const networkManager = this._frameManager.networkManager();
        networkManager.on(Events_1.Events.NetworkManager.Request, event => this.emit(Events_1.Events.Page.Request, event));
        networkManager.on(Events_1.Events.NetworkManager.Response, event => this.emit(Events_1.Events.Page.Response, event));
        networkManager.on(Events_1.Events.NetworkManager.RequestFailed, event => this.emit(Events_1.Events.Page.RequestFailed, event));
        networkManager.on(Events_1.Events.NetworkManager.RequestFinished, event => this.emit(Events_1.Events.Page.RequestFinished, event));
        this._fileChooserInterceptors = new Set();
        client.on('Page.domContentEventFired', () => this.emit(Events_1.Events.Page.DOMContentLoaded));
        client.on('Page.loadEventFired', () => this.emit(Events_1.Events.Page.Load));
        client.on('Runtime.consoleAPICalled', event => this._onConsoleAPI(event));
        client.on('Runtime.bindingCalled', event => this._onBindingCalled(event));
        client.on('Page.javascriptDialogOpening', event => this._onDialog(event));
        client.on('Runtime.exceptionThrown', exception => this._handleException(exception.exceptionDetails));
        client.on('Inspector.targetCrashed', () => this._onTargetCrashed());
        client.on('Performance.metrics', event => this._emitMetrics(event));
        client.on('Log.entryAdded', event => this._onLogEntryAdded(event));
        client.on('Page.fileChooserOpened', event => this._onFileChooser(event));
        this._target._isClosedPromise.then(() => {
            this.emit(Events_1.Events.Page.Close);
            this._closed = true;
        });
    }
    static async create(client, target, ignoreHTTPSErrors, defaultViewport, screenshotTaskQueue) {
        const page = new Page(client, target, ignoreHTTPSErrors, screenshotTaskQueue);
        await page._initialize();
        if (defaultViewport)
            await page.setViewport(defaultViewport);
        return page;
    }
    async _initialize() {
        await Promise.all([
            this._frameManager.initialize(),
            this._client.send('Target.setAutoAttach', { autoAttach: true, waitForDebuggerOnStart: false, flatten: true }),
            this._client.send('Performance.enable', {}),
            this._client.send('Log.enable', {}),
        ]);
    }
    async _onFileChooser(event) {
        if (!this._fileChooserInterceptors.size)
            return;
        const frame = this._frameManager.frame(event.frameId);
        const context = await frame.executionContext();
        const element = await context._adoptBackendNodeId(event.backendNodeId);
        const interceptors = Array.from(this._fileChooserInterceptors);
        this._fileChooserInterceptors.clear();
        const fileChooser = new FileChooser(this._client, element, event);
        for (const interceptor of interceptors)
            interceptor.call(null, fileChooser);
    }
    async waitForFileChooser(options = {}) {
        if (!this._fileChooserInterceptors.size)
            await this._client.send('Page.setInterceptFileChooserDialog', { enabled: true });
        const { timeout = this._timeoutSettings.timeout(), } = options;
        let callback;
        const promise = new Promise(x => callback = x);
        this._fileChooserInterceptors.add(callback);
        return helper_1.helper.waitWithTimeout(promise, 'waiting for file chooser', timeout).catch(error => {
            this._fileChooserInterceptors.delete(callback);
            throw error;
        });
    }
    async setGeolocation(options) {
        const { longitude, latitude, accuracy = 0 } = options;
        if (longitude < -180 || longitude > 180)
            throw new Error(`Invalid longitude "${longitude}": precondition -180 <= LONGITUDE <= 180 failed.`);
        if (latitude < -90 || latitude > 90)
            throw new Error(`Invalid latitude "${latitude}": precondition -90 <= LATITUDE <= 90 failed.`);
        if (accuracy < 0)
            throw new Error(`Invalid accuracy "${accuracy}": precondition 0 <= ACCURACY failed.`);
        await this._client.send('Emulation.setGeolocationOverride', { longitude, latitude, accuracy });
    }
    target() {
        return this._target;
    }
    browser() {
        return this._target.browser();
    }
    browserContext() {
        return this._target.browserContext();
    }
    _onTargetCrashed() {
        this.emit('error', new Error('Page crashed!'));
    }
    _onLogEntryAdded(event) {
        const { level, text, args, source, url, lineNumber } = event.entry;
        if (args)
            args.map(arg => helper_1.helper.releaseObject(this._client, arg));
        if (source !== 'worker')
            this.emit(Events_1.Events.Page.Console, new ConsoleMessage(level, text, [], { url, lineNumber }));
    }
    mainFrame() {
        return this._frameManager.mainFrame();
    }
    get keyboard() {
        return this._keyboard;
    }
    get touchscreen() {
        return this._touchscreen;
    }
    get coverage() {
        return this._coverage;
    }
    get tracing() {
        return this._tracing;
    }
    get accessibility() {
        return this._accessibility;
    }
    frames() {
        return this._frameManager.frames();
    }
    workers() {
        return Array.from(this._workers.values());
    }
    async setRequestInterception(value) {
        return this._frameManager.networkManager().setRequestInterception(value);
    }
    setOfflineMode(enabled) {
        return this._frameManager.networkManager().setOfflineMode(enabled);
    }
    setDefaultNavigationTimeout(timeout) {
        this._timeoutSettings.setDefaultNavigationTimeout(timeout);
    }
    setDefaultTimeout(timeout) {
        this._timeoutSettings.setDefaultTimeout(timeout);
    }
    async $(selector) {
        return this.mainFrame().$(selector);
    }
    async evaluateHandle(pageFunction, ...args) {
        const context = await this.mainFrame().executionContext();
        return context.evaluateHandle(pageFunction, ...args);
    }
    async queryObjects(prototypeHandle) {
        const context = await this.mainFrame().executionContext();
        return context.queryObjects(prototypeHandle);
    }
    async $eval(selector, pageFunction, ...args) {
        return this.mainFrame().$eval(selector, pageFunction, ...args);
    }
    async $$eval(selector, pageFunction, ...args) {
        return this.mainFrame().$$eval(selector, pageFunction, ...args);
    }
    async $$(selector) {
        return this.mainFrame().$$(selector);
    }
    async $x(expression) {
        return this.mainFrame().$x(expression);
    }
    async cookies(...urls) {
        const originalCookies = (await this._client.send('Network.getCookies', {
            urls: urls.length ? urls : [this.url()]
        })).cookies;
        const unsupportedCookieAttributes = ['priority'];
        const filterUnsupportedAttributes = (cookie) => {
            for (const attr of unsupportedCookieAttributes)
                delete cookie[attr];
            return cookie;
        };
        return originalCookies.map(filterUnsupportedAttributes);
    }
    async deleteCookie(...cookies) {
        const pageURL = this.url();
        for (const cookie of cookies) {
            const item = Object.assign({}, cookie);
            if (!cookie.url && pageURL.startsWith('http'))
                item.url = pageURL;
            await this._client.send('Network.deleteCookies', item);
        }
    }
    async setCookie(...cookies) {
        const pageURL = this.url();
        const startsWithHTTP = pageURL.startsWith('http');
        const items = cookies.map(cookie => {
            const item = Object.assign({}, cookie);
            if (!item.url && startsWithHTTP)
                item.url = pageURL;
            helper_1.assert(item.url !== 'about:blank', `Blank page can not have cookie "${item.name}"`);
            helper_1.assert(!String.prototype.startsWith.call(item.url || '', 'data:'), `Data URL page can not have cookie "${item.name}"`);
            return item;
        });
        await this.deleteCookie(...items);
        if (items.length)
            await this._client.send('Network.setCookies', { cookies: items });
    }
    async addScriptTag(options) {
        return this.mainFrame().addScriptTag(options);
    }
    async addStyleTag(options) {
        return this.mainFrame().addStyleTag(options);
    }
    async exposeFunction(name, puppeteerFunction) {
        if (this._pageBindings.has(name))
            throw new Error(`Failed to add page binding with name ${name}: window['${name}'] already exists!`);
        this._pageBindings.set(name, puppeteerFunction);
        const expression = helper_1.helper.evaluationString(addPageBinding, name);
        await this._client.send('Runtime.addBinding', { name: name });
        await this._client.send('Page.addScriptToEvaluateOnNewDocument', { source: expression });
        await Promise.all(this.frames().map(frame => frame.evaluate(expression).catch(helper_1.debugError)));
        function addPageBinding(bindingName) {
            /* Cast window to any here as we're about to add properties to it
             * via win[bindingName] which TypeScript doesn't like.
             */
            const win = window;
            const binding = win[bindingName];
            win[bindingName] = (...args) => {
                const me = window[bindingName];
                let callbacks = me['callbacks'];
                if (!callbacks) {
                    callbacks = new Map();
                    me['callbacks'] = callbacks;
                }
                const seq = (me['lastSeq'] || 0) + 1;
                me['lastSeq'] = seq;
                const promise = new Promise((resolve, reject) => callbacks.set(seq, { resolve, reject }));
                binding(JSON.stringify({ name: bindingName, seq, args }));
                return promise;
            };
        }
    }
    async authenticate(credentials) {
        return this._frameManager.networkManager().authenticate(credentials);
    }
    async setExtraHTTPHeaders(headers) {
        return this._frameManager.networkManager().setExtraHTTPHeaders(headers);
    }
    async setUserAgent(userAgent) {
        return this._frameManager.networkManager().setUserAgent(userAgent);
    }
    async metrics() {
        const response = await this._client.send('Performance.getMetrics');
        return this._buildMetricsObject(response.metrics);
    }
    _emitMetrics(event) {
        this.emit(Events_1.Events.Page.Metrics, {
            title: event.title,
            metrics: this._buildMetricsObject(event.metrics)
        });
    }
    _buildMetricsObject(metrics) {
        const result = {};
        for (const metric of metrics || []) {
            if (supportedMetrics.has(metric.name))
                result[metric.name] = metric.value;
        }
        return result;
    }
    _handleException(exceptionDetails) {
        const message = helper_1.helper.getExceptionMessage(exceptionDetails);
        const err = new Error(message);
        err.stack = ''; // Don't report clientside error with a node stack attached
        this.emit(Events_1.Events.Page.PageError, err);
    }
    async _onConsoleAPI(event) {
        if (event.executionContextId === 0) {
            // DevTools protocol stores the last 1000 console messages. These
            // messages are always reported even for removed execution contexts. In
            // this case, they are marked with executionContextId = 0 and are
            // reported upon enabling Runtime agent.
            //
            // Ignore these messages since:
            // - there's no execution context we can use to operate with message
            //   arguments
            // - these messages are reported before Puppeteer clients can subscribe
            //   to the 'console'
            //   page event.
            //
            // @see https://github.com/puppeteer/puppeteer/issues/3865
            return;
        }
        const context = this._frameManager.executionContextById(event.executionContextId);
        const values = event.args.map(arg => JSHandle_1.createJSHandle(context, arg));
        this._addConsoleMessage(event.type, values, event.stackTrace);
    }
    async _onBindingCalled(event) {
        const { name, seq, args } = JSON.parse(event.payload);
        let expression = null;
        try {
            const result = await this._pageBindings.get(name)(...args);
            expression = helper_1.helper.evaluationString(deliverResult, name, seq, result);
        }
        catch (error) {
            if (error instanceof Error)
                expression = helper_1.helper.evaluationString(deliverError, name, seq, error.message, error.stack);
            else
                expression = helper_1.helper.evaluationString(deliverErrorValue, name, seq, error);
        }
        this._client.send('Runtime.evaluate', { expression, contextId: event.executionContextId }).catch(helper_1.debugError);
        function deliverResult(name, seq, result) {
            window[name]['callbacks'].get(seq).resolve(result);
            window[name]['callbacks'].delete(seq);
        }
        function deliverError(name, seq, message, stack) {
            const error = new Error(message);
            error.stack = stack;
            window[name]['callbacks'].get(seq).reject(error);
            window[name]['callbacks'].delete(seq);
        }
        function deliverErrorValue(name, seq, value) {
            window[name]['callbacks'].get(seq).reject(value);
            window[name]['callbacks'].delete(seq);
        }
    }
    _addConsoleMessage(type, args, stackTrace) {
        if (!this.listenerCount(Events_1.Events.Page.Console)) {
            args.forEach(arg => arg.dispose());
            return;
        }
        const textTokens = [];
        for (const arg of args) {
            const remoteObject = arg._remoteObject;
            if (remoteObject.objectId)
                textTokens.push(arg.toString());
            else
                textTokens.push(helper_1.helper.valueFromRemoteObject(remoteObject));
        }
        const location = stackTrace && stackTrace.callFrames.length ? {
            url: stackTrace.callFrames[0].url,
            lineNumber: stackTrace.callFrames[0].lineNumber,
            columnNumber: stackTrace.callFrames[0].columnNumber,
        } : {};
        const message = new ConsoleMessage(type, textTokens.join(' '), args, location);
        this.emit(Events_1.Events.Page.Console, message);
    }
    _onDialog(event) {
        let dialogType = null;
        if (event.type === 'alert')
            dialogType = Dialog_1.Dialog.Type.Alert;
        else if (event.type === 'confirm')
            dialogType = Dialog_1.Dialog.Type.Confirm;
        else if (event.type === 'prompt')
            dialogType = Dialog_1.Dialog.Type.Prompt;
        else if (event.type === 'beforeunload')
            dialogType = Dialog_1.Dialog.Type.BeforeUnload;
        helper_1.assert(dialogType, 'Unknown javascript dialog type: ' + event.type);
        const dialog = new Dialog_1.Dialog(this._client, dialogType, event.message, event.defaultPrompt);
        this.emit(Events_1.Events.Page.Dialog, dialog);
    }
    url() {
        return this.mainFrame().url();
    }
    async content() {
        return await this._frameManager.mainFrame().content();
    }
    async setContent(html, options) {
        await this._frameManager.mainFrame().setContent(html, options);
    }
    async goto(url, options) {
        return await this._frameManager.mainFrame().goto(url, options);
    }
    async reload(options) {
        const result = await Promise.all([
            this.waitForNavigation(options),
            this._client.send('Page.reload')
        ]);
        return result[0];
    }
    async waitForNavigation(options = {}) {
        return await this._frameManager.mainFrame().waitForNavigation(options);
    }
    _sessionClosePromise() {
        if (!this._disconnectPromise)
            this._disconnectPromise = new Promise(fulfill => this._client.once(Events_1.Events.CDPSession.Disconnected, () => fulfill(new Error('Target closed'))));
        return this._disconnectPromise;
    }
    async waitForRequest(urlOrPredicate, options = {}) {
        const { timeout = this._timeoutSettings.timeout(), } = options;
        return helper_1.helper.waitForEvent(this._frameManager.networkManager(), Events_1.Events.NetworkManager.Request, request => {
            if (helper_1.helper.isString(urlOrPredicate))
                return (urlOrPredicate === request.url());
            if (typeof urlOrPredicate === 'function')
                return !!(urlOrPredicate(request));
            return false;
        }, timeout, this._sessionClosePromise());
    }
    async waitForResponse(urlOrPredicate, options = {}) {
        const { timeout = this._timeoutSettings.timeout(), } = options;
        return helper_1.helper.waitForEvent(this._frameManager.networkManager(), Events_1.Events.NetworkManager.Response, response => {
            if (helper_1.helper.isString(urlOrPredicate))
                return (urlOrPredicate === response.url());
            if (typeof urlOrPredicate === 'function')
                return !!(urlOrPredicate(response));
            return false;
        }, timeout, this._sessionClosePromise());
    }
    async goBack(options) {
        return this._go(-1, options);
    }
    async goForward(options) {
        return this._go(+1, options);
    }
    async _go(delta, options) {
        const history = await this._client.send('Page.getNavigationHistory');
        const entry = history.entries[history.currentIndex + delta];
        if (!entry)
            return null;
        const result = await Promise.all([
            this.waitForNavigation(options),
            this._client.send('Page.navigateToHistoryEntry', { entryId: entry.id }),
        ]);
        return result[0];
    }
    async bringToFront() {
        await this._client.send('Page.bringToFront');
    }
    async emulate(options) {
        await Promise.all([
            this.setViewport(options.viewport),
            this.setUserAgent(options.userAgent)
        ]);
    }
    async setJavaScriptEnabled(enabled) {
        if (this._javascriptEnabled === enabled)
            return;
        this._javascriptEnabled = enabled;
        await this._client.send('Emulation.setScriptExecutionDisabled', { value: !enabled });
    }
    async setBypassCSP(enabled) {
        await this._client.send('Page.setBypassCSP', { enabled });
    }
    async emulateMediaType(type) {
        helper_1.assert(type === 'screen' || type === 'print' || type === null, 'Unsupported media type: ' + type);
        await this._client.send('Emulation.setEmulatedMedia', { media: type || '' });
    }
    async emulateMediaFeatures(features) {
        if (features === null)
            await this._client.send('Emulation.setEmulatedMedia', { features: null });
        if (Array.isArray(features)) {
            features.every(mediaFeature => {
                const name = mediaFeature.name;
                helper_1.assert(/^prefers-(?:color-scheme|reduced-motion)$/.test(name), 'Unsupported media feature: ' + name);
                return true;
            });
            await this._client.send('Emulation.setEmulatedMedia', { features: features });
        }
    }
    async emulateTimezone(timezoneId) {
        try {
            await this._client.send('Emulation.setTimezoneOverride', { timezoneId: timezoneId || '' });
        }
        catch (error) {
            if (error.message.includes('Invalid timezone'))
                throw new Error(`Invalid timezone ID: ${timezoneId}`);
            throw error;
        }
    }
    async setViewport(viewport) {
        const needsReload = await this._emulationManager.emulateViewport(viewport);
        this._viewport = viewport;
        if (needsReload)
            await this.reload();
    }
    viewport() {
        return this._viewport;
    }
    async evaluate(pageFunction, ...args) {
        return this._frameManager.mainFrame().evaluate(pageFunction, ...args);
    }
    async evaluateOnNewDocument(pageFunction, ...args) {
        const source = helper_1.helper.evaluationString(pageFunction, ...args);
        await this._client.send('Page.addScriptToEvaluateOnNewDocument', { source });
    }
    async setCacheEnabled(enabled = true) {
        await this._frameManager.networkManager().setCacheEnabled(enabled);
    }
    async screenshot(options = {}) {
        let screenshotType = null;
        // options.type takes precedence over inferring the type from options.path
        // because it may be a 0-length file with no extension created beforehand (i.e. as a temp file).
        if (options.type) {
            helper_1.assert(options.type === 'png' || options.type === 'jpeg', 'Unknown options.type value: ' + options.type);
            screenshotType = options.type;
        }
        else if (options.path) {
            const mimeType = mime.getType(options.path);
            if (mimeType === 'image/png')
                screenshotType = 'png';
            else if (mimeType === 'image/jpeg')
                screenshotType = 'jpeg';
            helper_1.assert(screenshotType, 'Unsupported screenshot mime type: ' + mimeType);
        }
        if (!screenshotType)
            screenshotType = 'png';
        if (options.quality) {
            helper_1.assert(screenshotType === 'jpeg', 'options.quality is unsupported for the ' + screenshotType + ' screenshots');
            helper_1.assert(typeof options.quality === 'number', 'Expected options.quality to be a number but found ' + (typeof options.quality));
            helper_1.assert(Number.isInteger(options.quality), 'Expected options.quality to be an integer');
            helper_1.assert(options.quality >= 0 && options.quality <= 100, 'Expected options.quality to be between 0 and 100 (inclusive), got ' + options.quality);
        }
        helper_1.assert(!options.clip || !options.fullPage, 'options.clip and options.fullPage are exclusive');
        if (options.clip) {
            helper_1.assert(typeof options.clip.x === 'number', 'Expected options.clip.x to be a number but found ' + (typeof options.clip.x));
            helper_1.assert(typeof options.clip.y === 'number', 'Expected options.clip.y to be a number but found ' + (typeof options.clip.y));
            helper_1.assert(typeof options.clip.width === 'number', 'Expected options.clip.width to be a number but found ' + (typeof options.clip.width));
            helper_1.assert(typeof options.clip.height === 'number', 'Expected options.clip.height to be a number but found ' + (typeof options.clip.height));
            helper_1.assert(options.clip.width !== 0, 'Expected options.clip.width not to be 0.');
            helper_1.assert(options.clip.height !== 0, 'Expected options.clip.height not to be 0.');
        }
        return this._screenshotTaskQueue.postTask(this._screenshotTask.bind(this, screenshotType, options));
    }
    async _screenshotTask(format, options) {
        await this._client.send('Target.activateTarget', { targetId: this._target._targetId });
        let clip = options.clip ? processClip(options.clip) : undefined;
        if (options.fullPage) {
            const metrics = await this._client.send('Page.getLayoutMetrics');
            const width = Math.ceil(metrics.contentSize.width);
            const height = Math.ceil(metrics.contentSize.height);
            // Overwrite clip for full page at all times.
            clip = { x: 0, y: 0, width, height, scale: 1 };
            const { isMobile = false, deviceScaleFactor = 1, isLandscape = false } = this._viewport || {};
            const screenOrientation = isLandscape ? { angle: 90, type: 'landscapePrimary' } : { angle: 0, type: 'portraitPrimary' };
            await this._client.send('Emulation.setDeviceMetricsOverride', { mobile: isMobile, width, height, deviceScaleFactor, screenOrientation });
        }
        const shouldSetDefaultBackground = options.omitBackground && format === 'png';
        if (shouldSetDefaultBackground)
            await this._client.send('Emulation.setDefaultBackgroundColorOverride', { color: { r: 0, g: 0, b: 0, a: 0 } });
        const result = await this._client.send('Page.captureScreenshot', { format, quality: options.quality, clip });
        if (shouldSetDefaultBackground)
            await this._client.send('Emulation.setDefaultBackgroundColorOverride');
        if (options.fullPage && this._viewport)
            await this.setViewport(this._viewport);
        const buffer = options.encoding === 'base64' ? result.data : Buffer.from(result.data, 'base64');
        if (options.path)
            await writeFileAsync(options.path, buffer);
        return buffer;
        function processClip(clip) {
            const x = Math.round(clip.x);
            const y = Math.round(clip.y);
            const width = Math.round(clip.width + clip.x - x);
            const height = Math.round(clip.height + clip.y - y);
            return { x, y, width, height, scale: 1 };
        }
    }
    async pdf(options = {}) {
        const { scale = 1, displayHeaderFooter = false, headerTemplate = '', footerTemplate = '', printBackground = false, landscape = false, pageRanges = '', preferCSSPageSize = false, margin = {}, path = null } = options;
        let paperWidth = 8.5;
        let paperHeight = 11;
        if (options.format) {
            const format = paperFormats[options.format.toLowerCase()];
            helper_1.assert(format, 'Unknown paper format: ' + options.format);
            paperWidth = format.width;
            paperHeight = format.height;
        }
        else {
            paperWidth = convertPrintParameterToInches(options.width) || paperWidth;
            paperHeight = convertPrintParameterToInches(options.height) || paperHeight;
        }
        const marginTop = convertPrintParameterToInches(margin.top) || 0;
        const marginLeft = convertPrintParameterToInches(margin.left) || 0;
        const marginBottom = convertPrintParameterToInches(margin.bottom) || 0;
        const marginRight = convertPrintParameterToInches(margin.right) || 0;
        const result = await this._client.send('Page.printToPDF', {
            transferMode: 'ReturnAsStream',
            landscape,
            displayHeaderFooter,
            headerTemplate,
            footerTemplate,
            printBackground,
            scale,
            paperWidth,
            paperHeight,
            marginTop,
            marginBottom,
            marginLeft,
            marginRight,
            pageRanges,
            preferCSSPageSize
        });
        return await helper_1.helper.readProtocolStream(this._client, result.stream, path);
    }
    async title() {
        return this.mainFrame().title();
    }
    async close(options = { runBeforeUnload: undefined }) {
        helper_1.assert(!!this._client._connection, 'Protocol error: Connection closed. Most likely the page has been closed.');
        const runBeforeUnload = !!options.runBeforeUnload;
        if (runBeforeUnload) {
            await this._client.send('Page.close');
        }
        else {
            await this._client._connection.send('Target.closeTarget', { targetId: this._target._targetId });
            await this._target._isClosedPromise;
        }
    }
    isClosed() {
        return this._closed;
    }
    get mouse() {
        return this._mouse;
    }
    click(selector, options = {}) {
        return this.mainFrame().click(selector, options);
    }
    focus(selector) {
        return this.mainFrame().focus(selector);
    }
    hover(selector) {
        return this.mainFrame().hover(selector);
    }
    select(selector, ...values) {
        return this.mainFrame().select(selector, ...values);
    }
    tap(selector) {
        return this.mainFrame().tap(selector);
    }
    type(selector, text, options) {
        return this.mainFrame().type(selector, text, options);
    }
    waitFor(selectorOrFunctionOrTimeout, options = {}, ...args) {
        return this.mainFrame().waitFor(selectorOrFunctionOrTimeout, options, ...args);
    }
    waitForSelector(selector, options = {}) {
        return this.mainFrame().waitForSelector(selector, options);
    }
    waitForXPath(xpath, options = {}) {
        return this.mainFrame().waitForXPath(xpath, options);
    }
    waitForFunction(pageFunction, options = {}, ...args) {
        return this.mainFrame().waitForFunction(pageFunction, options, ...args);
    }
}
exports.Page = Page;
const supportedMetrics = new Set([
    'Timestamp',
    'Documents',
    'Frames',
    'JSEventListeners',
    'Nodes',
    'LayoutCount',
    'RecalcStyleCount',
    'LayoutDuration',
    'RecalcStyleDuration',
    'ScriptDuration',
    'TaskDuration',
    'JSHeapUsedSize',
    'JSHeapTotalSize',
]);
const unitToPixels = {
    'px': 1,
    'in': 96,
    'cm': 37.8,
    'mm': 3.78
};
function convertPrintParameterToInches(parameter) {
    if (typeof parameter === 'undefined')
        return undefined;
    let pixels;
    if (helper_1.helper.isNumber(parameter)) {
        // Treat numbers as pixel values to be aligned with phantom's paperSize.
        pixels = /** @type {number} */ (parameter);
    }
    else if (helper_1.helper.isString(parameter)) {
        const text = /** @type {string} */ (parameter);
        let unit = text.substring(text.length - 2).toLowerCase();
        let valueText = '';
        if (unitToPixels.hasOwnProperty(unit)) {
            valueText = text.substring(0, text.length - 2);
        }
        else {
            // In case of unknown unit try to parse the whole parameter as number of pixels.
            // This is consistent with phantom's paperSize behavior.
            unit = 'px';
            valueText = text;
        }
        const value = Number(valueText);
        helper_1.assert(!isNaN(value), 'Failed to parse parameter value: ' + text);
        pixels = value * unitToPixels[unit];
    }
    else {
        throw new Error('page.pdf() Cannot handle parameter type: ' + (typeof parameter));
    }
    return pixels / 96;
}
class ConsoleMessage {
    constructor(type, text, args, location = {}) {
        this._type = type;
        this._text = text;
        this._args = args;
        this._location = location;
    }
    type() {
        return this._type;
    }
    text() {
        return this._text;
    }
    args() {
        return this._args;
    }
    location() {
        return this._location;
    }
}
exports.ConsoleMessage = ConsoleMessage;
class FileChooser {
    constructor(client, element, event) {
        this._handled = false;
        this._client = client;
        this._element = element;
        this._multiple = event.mode !== 'selectSingle';
    }
    isMultiple() {
        return this._multiple;
    }
    async accept(filePaths) {
        helper_1.assert(!this._handled, 'Cannot accept FileChooser which is already handled!');
        this._handled = true;
        await this._element.uploadFile(...filePaths);
    }
    async cancel() {
        helper_1.assert(!this._handled, 'Cannot cancel FileChooser which is already handled!');
        this._handled = true;
    }
}
exports.FileChooser = FileChooser;
