/*
  @license
	Rollup.js v2.3.3
	Sat, 04 Apr 2020 22:17:48 GMT - commit d18cb37d7c328a63c36761583ce456275f164462


	https://github.com/rollup/rollup

	Released under the MIT License.
*/
'use strict';

function _interopNamespace(e) {
	if (e && e.__esModule) { return e; } else {
		var n = {};
		if (e) {
			Object.keys(e).forEach(function (k) {
				n[k] =e [k];
			});
		}
		n['default'] = e;
		return n;
	}
}

var rollup_js = require('./shared/rollup.js');
var path = require('path');
require('crypto');
require('fs');
require('events');
var mergeOptions = require('./shared/mergeOptions.js');
var commandPlugins = require('./shared/commandPlugins.js');
var url = require('url');

function supportsNativeESM() {
    return Number(/^v(\d+)/.exec(process.version)[1]) >= 13;
}
async function loadAndParseConfigFile(fileName, commandOptions = {}) {
    const configs = await loadConfigFile(fileName, commandOptions);
    const warnings = commandPlugins.batchWarnings();
    try {
        const normalizedConfigs = configs.map((config) => {
            const options = mergeOptions.mergeOptions(config, commandOptions, warnings.add);
            commandPlugins.addCommandPluginsToInputOptions(options, commandOptions);
            return options;
        });
        return { options: normalizedConfigs, warnings };
    }
    catch (err) {
        warnings.flush();
        throw err;
    }
}
async function loadConfigFile(fileName, commandOptions) {
    const extension = path.extname(fileName);
    const configFileExport = extension === '.mjs' && supportsNativeESM()
        ? (await import(url.pathToFileURL(fileName).href)).default
        : extension === '.cjs'
            ? getDefaultFromCjs(require(fileName))
            : await getDefaultFromTranspiledConfigFile(fileName, commandOptions.silent);
    return getConfigList(configFileExport, commandOptions);
}
function getDefaultFromCjs(namespace) {
    return namespace.__esModule ? namespace.default : namespace;
}
async function getDefaultFromTranspiledConfigFile(fileName, silent) {
    const warnings = commandPlugins.batchWarnings();
    const bundle = await rollup_js.rollup({
        external: (id) => (id[0] !== '.' && !path.isAbsolute(id)) || id.slice(-5, id.length) === '.json',
        input: fileName,
        onwarn: warnings.add,
        treeshake: false,
    });
    if (!silent && warnings.count > 0) {
        commandPlugins.stderr(commandPlugins.color.bold(`loaded ${rollup_js.relativeId(fileName)} with warnings`));
        warnings.flush();
    }
    const { output: [{ code }], } = await bundle.generate({
        exports: 'named',
        format: 'cjs',
    });
    return loadConfigFromBundledFile(fileName, code);
}
async function loadConfigFromBundledFile(fileName, bundledCode) {
    const extension = path.extname(fileName);
    const defaultLoader = require.extensions[extension];
    require.extensions[extension] = (module, filename) => {
        if (filename === fileName) {
            module._compile(bundledCode, filename);
        }
        else {
            defaultLoader(module, filename);
        }
    };
    delete require.cache[fileName];
    try {
        const config = getDefaultFromCjs(require(fileName));
        require.extensions[extension] = defaultLoader;
        return config;
    }
    catch (err) {
        if (err.code === 'ERR_REQUIRE_ESM') {
            return rollup_js.error({
                code: 'TRANSPILED_ESM_CONFIG',
                message: `While loading the Rollup configuration from "${rollup_js.relativeId(fileName)}", Node tried to require an ES module from a CommonJS file, which is not supported. A common cause is if there is a package.json file with "type": "module" in the same folder. You can try to fix this by changing the extension of your configuration file to ".cjs" or ".mjs" depending on the content, which will prevent Rollup from trying to preprocess the file but rather hand it to Node directly.`,
                url: 'https://rollupjs.org/guide/en/#using-untranspiled-config-files',
            });
        }
        throw err;
    }
}
async function getConfigList(configFileExport, commandOptions) {
    const config = await (typeof configFileExport === 'function'
        ? configFileExport(commandOptions)
        : configFileExport);
    if (Object.keys(config).length === 0) {
        return rollup_js.error({
            code: 'MISSING_CONFIG',
            message: 'Config file must export an options object, or an array of options objects',
            url: 'https://rollupjs.org/guide/en/#configuration-files',
        });
    }
    return Array.isArray(config) ? config : [config];
}

module.exports = loadAndParseConfigFile;
//# sourceMappingURL=loadConfigFile.js.map
