/* SPDX-License-Identifier: GPL-2.0 */
/*
 * encoder public register definitions
 *
 * Copyright (c) Imagination Technologies Ltd.
 * Copyright (c) 2021 Texas Instruments Incorporated - http://www.ti.com/
 *
 * Authors:
 *	Sunita Nadampalli <sunitan@ti.com>
 *
 * Re-written for upstreming
 *	Sidraya Jayagond <sidraya.bj@pathpartnertech.com>
 */

#ifndef __VXE_PUBLIC_REGDEFS_H__
#define __VXE_PUBLIC_REGDEFS_H__

#include <linux/io.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/jiffies.h>

/* Write to the register */
#define VXE_WR_REG32(base, offs, val) \
	(iowrite32((val), (void *)((offs) + (unsigned long)(base))))

/* Read the register */
#define VXE_RD_REG32(base, offs) \
	(ioread32((void *)((base) + (offs))))

#define VXE_POLL_REG32_ISEQ(base, offs, val, mask, cnt) \
	(ioreg32_poll_iseq((unsigned long)(base) + (offs), val, mask, cnt))

#define REG_BASE_HOST                   0x00000000
#define REG_OFFSET_TOPAZ_MTX            0x00000800
#define REG_START_TOPAZ_MTX_HOST        (REG_BASE_HOST + REG_OFFSET_TOPAZ_MTX)

static inline int ioreg32_poll_iseq(unsigned long addr,
				    unsigned int req_val, unsigned int mask, unsigned int cnt)
{
	unsigned int count, val;
	unsigned int res = 0;

	/* Add high-frequency poll loops. */
	cnt += 10;

	/*
	 * High-frequency loop (designed for shorter hardware latency such as
	 * reset).
	 */
	for (count = 0; count < cnt; count++) {
		/* Read from the device */
		val = ioread32((void *)addr);
		val = (val & mask);

		if (val == req_val) {
			res = 0;
			break;
		}

		/*
		 * Sleep to wait for hardware.
		 * Period is selected to allow for high-frequency polling
		 * (5us, e.g. reset) over the first 10 iterations, then
		 * reverting to a lower-frequency (100us, e.g. DMA) for the
		 * remainder.
		 */
		if (count < 10)
			usleep_range(5, 5);
		else
			usleep_range(100, 100);
	}

	if (res || count >= cnt) {
		pr_info("Poll failed!\n");
		res = -1;
	}

	return res;
}

/*
 * DMAC configuration values:
 */
/*! The maximum number of channels in the SoC               */
#define DMAC_MAX_CHANNELS       (1)

/* Register CR_TOPAZHP_CORE_REV */
#define TOPAZHP_TOP_CR_TOPAZHP_CORE_REV 0x03D0
#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MAINT_REV 0x000000FF
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MAINT_REV 0
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MAINT_REV 0x03D0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MAINT_REV 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MINOR_REV 0x0000FF00
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MINOR_REV 8
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MINOR_REV 0x03D0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MINOR_REV 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MAJOR_REV 0x00FF0000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MAJOR_REV 16
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MAJOR_REV 0x03D0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MAJOR_REV 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_DESIGNER 0xFF000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_DESIGNER 24
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_DESIGNER 0x03D0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_DESIGNER 0

/* Register CR_TOPAZHP_CORE_DES1 */
#define TOPAZHP_TOP_CR_TOPAZHP_CORE_DES1 0x03E0

/* Register CR_MULTICORE_HW_CFG */
#define TOPAZHP_TOP_CR_MULTICORE_HW_CFG 0x0058
#define MASK_TOPAZHP_TOP_CR_NUM_CORES_SUPPORTED 0x0000001F
#define SHIFT_TOPAZHP_TOP_CR_NUM_CORES_SUPPORTED 0
#define REGNUM_TOPAZHP_TOP_CR_NUM_CORES_SUPPORTED 0x0058
#define SIGNED_TOPAZHP_TOP_CR_NUM_CORES_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_NUM_MTX_SUPPORTED 0x00000700
#define SHIFT_TOPAZHP_TOP_CR_NUM_MTX_SUPPORTED 8
#define REGNUM_TOPAZHP_TOP_CR_NUM_MTX_SUPPORTED 0x0058
#define SIGNED_TOPAZHP_TOP_CR_NUM_MTX_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_NUM_CORES_PER_MTX 0x00070000
#define SHIFT_TOPAZHP_TOP_CR_NUM_CORES_PER_MTX 16
#define REGNUM_TOPAZHP_TOP_CR_NUM_CORES_PER_MTX 0x0058
#define SIGNED_TOPAZHP_TOP_CR_NUM_CORES_PER_MTX 0

#define MASK_TOPAZHP_TOP_CR_EXTENDED_ADDR_RANGE 0x0F000000
#define SHIFT_TOPAZHP_TOP_CR_EXTENDED_ADDR_RANGE 24
#define REGNUM_TOPAZHP_TOP_CR_EXTENDED_ADDR_RANGE 0x0058
#define SIGNED_TOPAZHP_TOP_CR_EXTENDED_ADDR_RANGE 0

/* Register CR_MULTICORE_SRST */
#define TOPAZHP_TOP_CR_MULTICORE_SRST 0x0000
#define MASK_TOPAZHP_TOP_CR_IMG_TOPAZ_MTX_SOFT_RESET 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_IMG_TOPAZ_MTX_SOFT_RESET 0
#define REGNUM_TOPAZHP_TOP_CR_IMG_TOPAZ_MTX_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_TOP_CR_IMG_TOPAZ_MTX_SOFT_RESET 0

#define MASK_TOPAZHP_TOP_CR_IMG_TOPAZ_IO_SOFT_RESET 0x00000002
#define SHIFT_TOPAZHP_TOP_CR_IMG_TOPAZ_IO_SOFT_RESET 1
#define REGNUM_TOPAZHP_TOP_CR_IMG_TOPAZ_IO_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_TOP_CR_IMG_TOPAZ_IO_SOFT_RESET 0

#define MASK_TOPAZHP_TOP_CR_IMG_TOPAZ_CORE_SOFT_RESET 0x00000004
#define SHIFT_TOPAZHP_TOP_CR_IMG_TOPAZ_CORE_SOFT_RESET 2
#define REGNUM_TOPAZHP_TOP_CR_IMG_TOPAZ_CORE_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_TOP_CR_IMG_TOPAZ_CORE_SOFT_RESET 0

/* Register CR_MULTICORE_INT_STAT */
#define TOPAZHP_TOP_CR_MULTICORE_INT_STAT 0x0004
#define MASK_TOPAZHP_TOP_CR_INT_STAT_DMAC 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_INT_STAT_DMAC 0
#define REGNUM_TOPAZHP_TOP_CR_INT_STAT_DMAC 0x0004
#define SIGNED_TOPAZHP_TOP_CR_INT_STAT_DMAC 0

#define MASK_TOPAZHP_TOP_CR_INT_STAT_MTX 0x00000002
#define SHIFT_TOPAZHP_TOP_CR_INT_STAT_MTX 1
#define REGNUM_TOPAZHP_TOP_CR_INT_STAT_MTX 0x0004
#define SIGNED_TOPAZHP_TOP_CR_INT_STAT_MTX 0

#define MASK_TOPAZHP_TOP_CR_INT_STAT_MTX_HALT 0x00000004
#define SHIFT_TOPAZHP_TOP_CR_INT_STAT_MTX_HALT 2
#define REGNUM_TOPAZHP_TOP_CR_INT_STAT_MTX_HALT 0x0004
#define SIGNED_TOPAZHP_TOP_CR_INT_STAT_MTX_HALT 0

#define MASK_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT 0x00000078
#define SHIFT_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT 3
#define REGNUM_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT 0x0004
#define SIGNED_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT 0

#define MASK_TOPAZHP_TOP_CR_INT_STAT_MTX_CORES 0x0000FF00
#define SHIFT_TOPAZHP_TOP_CR_INT_STAT_MTX_CORES 8
#define REGNUM_TOPAZHP_TOP_CR_INT_STAT_MTX_CORES 0x0004
#define SIGNED_TOPAZHP_TOP_CR_INT_STAT_MTX_CORES 0

#define MASK_TOPAZHP_TOP_CR_INT_STAT_HOST_CORES 0x00FF0000
#define SHIFT_TOPAZHP_TOP_CR_INT_STAT_HOST_CORES 16
#define REGNUM_TOPAZHP_TOP_CR_INT_STAT_HOST_CORES 0x0004
#define SIGNED_TOPAZHP_TOP_CR_INT_STAT_HOST_CORES 0

#define MASK_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT_B 0x1E000000
#define SHIFT_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT_B 25
#define REGNUM_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT_B 0x0004
#define SIGNED_TOPAZHP_TOP_CR_INT_STAT_MMU_FAULT_B 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MAS_MTX_INTS 0x40000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MAS_MTX_INTS 30
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MAS_MTX_INTS 0x0004
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MAS_MTX_INTS 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MAS_HOST_INTS 0x80000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MAS_HOST_INTS 31
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MAS_HOST_INTS 0x0004
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MAS_HOST_INTS 0

/* Register CR_MULTICORE_HOST_INT_ENAB */
#define TOPAZHP_TOP_CR_MULTICORE_HOST_INT_ENAB 0x000C
#define MASK_TOPAZHP_TOP_CR_HOST_INTEN_DMAC 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_HOST_INTEN_DMAC 0
#define REGNUM_TOPAZHP_TOP_CR_HOST_INTEN_DMAC 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_INTEN_DMAC 0

#define MASK_TOPAZHP_TOP_CR_HOST_INTEN_MTX 0x00000002
#define SHIFT_TOPAZHP_TOP_CR_HOST_INTEN_MTX 1
#define REGNUM_TOPAZHP_TOP_CR_HOST_INTEN_MTX 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_INTEN_MTX 0

#define MASK_TOPAZHP_TOP_CR_HOST_INTEN_MTX_HALT 0x00000004
#define SHIFT_TOPAZHP_TOP_CR_HOST_INTEN_MTX_HALT 2
#define REGNUM_TOPAZHP_TOP_CR_HOST_INTEN_MTX_HALT 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_INTEN_MTX_HALT 0

#define MASK_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT 0x00000078
#define SHIFT_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT 3
#define REGNUM_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT 0

#define MASK_TOPAZHP_TOP_CR_HOST_INTEN_MTX_CORES 0x0000FF00
#define SHIFT_TOPAZHP_TOP_CR_HOST_INTEN_MTX_CORES 8
#define REGNUM_TOPAZHP_TOP_CR_HOST_INTEN_MTX_CORES 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_INTEN_MTX_CORES 0

#define MASK_TOPAZHP_TOP_CR_HOST_INTEN_HOST_CORES 0x00FF0000
#define SHIFT_TOPAZHP_TOP_CR_HOST_INTEN_HOST_CORES 16
#define REGNUM_TOPAZHP_TOP_CR_HOST_INTEN_HOST_CORES 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_INTEN_HOST_CORES 0

#define MASK_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT_B 0x1E000000
#define SHIFT_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT_B 25
#define REGNUM_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT_B 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_INTEN_MMU_FAULT_B 0

#define MASK_TOPAZHP_TOP_CR_HOST_TOPAZHP_MAS_INTEN 0x80000000
#define SHIFT_TOPAZHP_TOP_CR_HOST_TOPAZHP_MAS_INTEN 31
#define REGNUM_TOPAZHP_TOP_CR_HOST_TOPAZHP_MAS_INTEN 0x000C
#define SIGNED_TOPAZHP_TOP_CR_HOST_TOPAZHP_MAS_INTEN 0

/* Register CR_MULTICORE_INT_CLEAR */
#define TOPAZHP_TOP_CR_MULTICORE_INT_CLEAR 0x0010
#define MASK_TOPAZHP_TOP_CR_INTCLR_DMAC 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_INTCLR_DMAC 0
#define REGNUM_TOPAZHP_TOP_CR_INTCLR_DMAC 0x0010
#define SIGNED_TOPAZHP_TOP_CR_INTCLR_DMAC 0

#define MASK_TOPAZHP_TOP_CR_INTCLR_MTX 0x00000002
#define SHIFT_TOPAZHP_TOP_CR_INTCLR_MTX 1
#define REGNUM_TOPAZHP_TOP_CR_INTCLR_MTX 0x0010
#define SIGNED_TOPAZHP_TOP_CR_INTCLR_MTX 0

#define MASK_TOPAZHP_TOP_CR_INTCLR_MTX_HALT 0x00000004
#define SHIFT_TOPAZHP_TOP_CR_INTCLR_MTX_HALT 2
#define REGNUM_TOPAZHP_TOP_CR_INTCLR_MTX_HALT 0x0010
#define SIGNED_TOPAZHP_TOP_CR_INTCLR_MTX_HALT 0

#define MASK_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT 0x00000078
#define SHIFT_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT 3
#define REGNUM_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT 0x0010
#define SIGNED_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT 0

#define MASK_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT_B 0x1E000000
#define SHIFT_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT_B 25
#define REGNUM_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT_B 0x0010
#define SIGNED_TOPAZHP_TOP_CR_INTCLR_MMU_FAULT_B 0

/* Register CR_TOPAZ_CMD_FIFO_FLUSH */
#define TOPAZHP_TOP_CR_TOPAZ_CMD_FIFO_FLUSH 0x0078
#define MASK_TOPAZHP_TOP_CR_CMD_FIFO_FLUSH 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_CMD_FIFO_FLUSH 0
#define REGNUM_TOPAZHP_TOP_CR_CMD_FIFO_FLUSH 0x0078
#define SIGNED_TOPAZHP_TOP_CR_CMD_FIFO_FLUSH 0

/* Register CR_MULTICORE_CMD_FIFO_WRITE */
#define TOPAZHP_TOP_CR_MULTICORE_CMD_FIFO_WRITE 0x0060
#define MASK_TOPAZHP_TOP_CR_CMD_FIFO_WDATA 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_CMD_FIFO_WDATA 0
#define REGNUM_TOPAZHP_TOP_CR_CMD_FIFO_WDATA 0x0060
#define SIGNED_TOPAZHP_TOP_CR_CMD_FIFO_WDATA 0

/* Register CR_MULTICORE_CMD_FIFO_WRITE_SPACE */
#define TOPAZHP_TOP_CR_MULTICORE_CMD_FIFO_WRITE_SPACE 0x0064
#define MASK_TOPAZHP_TOP_CR_CMD_FIFO_SPACE 0x000000FF
#define SHIFT_TOPAZHP_TOP_CR_CMD_FIFO_SPACE 0
#define REGNUM_TOPAZHP_TOP_CR_CMD_FIFO_SPACE 0x0064
#define SIGNED_TOPAZHP_TOP_CR_CMD_FIFO_SPACE 0

#define MASK_TOPAZHP_TOP_CR_CMD_FIFO_FULL 0x00000100
#define SHIFT_TOPAZHP_TOP_CR_CMD_FIFO_FULL 8
#define REGNUM_TOPAZHP_TOP_CR_CMD_FIFO_FULL 0x0064
#define SIGNED_TOPAZHP_TOP_CR_CMD_FIFO_FULL 0

/* Register CR_MULTICORE_IDLE_PWR_MAN */
#define TOPAZHP_TOP_CR_MULTICORE_IDLE_PWR_MAN 0x0118
#define MASK_TOPAZHP_TOP_CR_TOPAZ_IDLE_DISABLE 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_TOPAZ_IDLE_DISABLE 0
#define REGNUM_TOPAZHP_TOP_CR_TOPAZ_IDLE_DISABLE 0x0118
#define SIGNED_TOPAZHP_TOP_CR_TOPAZ_IDLE_DISABLE 0

/* Register CR_FIRMWARE_REG_1 */
#define TOPAZHP_TOP_CR_FIRMWARE_REG_1 0x0100
#define MASK_TOPAZHP_TOP_CR_FIRMWARE_REG_1 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_FIRMWARE_REG_1 0
#define REGNUM_TOPAZHP_TOP_CR_FIRMWARE_REG_1 0x0100
#define SIGNED_TOPAZHP_TOP_CR_FIRMWARE_REG_1 0

/* Register CR_FIRMWARE_REG_2 */
#define TOPAZHP_TOP_CR_FIRMWARE_REG_2 0x0104
#define MASK_TOPAZHP_TOP_CR_FIRMWARE_REG_2 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_FIRMWARE_REG_2 0
#define REGNUM_TOPAZHP_TOP_CR_FIRMWARE_REG_2 0x0104
#define SIGNED_TOPAZHP_TOP_CR_FIRMWARE_REG_2 0

/* Register CR_FIRMWARE_REG_3 */
#define TOPAZHP_TOP_CR_FIRMWARE_REG_3 0x0108
#define MASK_TOPAZHP_TOP_CR_FIRMWARE_REG_3 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_FIRMWARE_REG_3 0
#define REGNUM_TOPAZHP_TOP_CR_FIRMWARE_REG_3 0x0108
#define SIGNED_TOPAZHP_TOP_CR_FIRMWARE_REG_3 0

/* Register CR_FIRMWARE_REG_4 */
#define TOPAZHP_TOP_CR_FIRMWARE_REG_4 0x0300
#define MASK_TOPAZHP_TOP_CR_FIRMWARE_REG_4 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_FIRMWARE_REG_4 0
#define REGNUM_TOPAZHP_TOP_CR_FIRMWARE_REG_4 0x0300
#define SIGNED_TOPAZHP_TOP_CR_FIRMWARE_REG_4 0

/* Register CR_FIRMWARE_REG_5 */
#define TOPAZHP_TOP_CR_FIRMWARE_REG_5 0x0304
#define MASK_TOPAZHP_TOP_CR_FIRMWARE_REG_5 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_FIRMWARE_REG_5 0
#define REGNUM_TOPAZHP_TOP_CR_FIRMWARE_REG_5 0x0304
#define SIGNED_TOPAZHP_TOP_CR_FIRMWARE_REG_5 0

/* Register CR_FIRMWARE_REG_6 */
#define TOPAZHP_TOP_CR_FIRMWARE_REG_6 0x0308
#define MASK_TOPAZHP_TOP_CR_FIRMWARE_REG_6 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_FIRMWARE_REG_6 0
#define REGNUM_TOPAZHP_TOP_CR_FIRMWARE_REG_6 0x0308
#define SIGNED_TOPAZHP_TOP_CR_FIRMWARE_REG_6 0

/* Register CR_FIRMWARE_REG_7 */
#define TOPAZHP_TOP_CR_FIRMWARE_REG_7 0x030C
#define MASK_TOPAZHP_TOP_CR_FIRMWARE_REG_7 0xFFFFFFFF
#define SHIFT_TOPAZHP_TOP_CR_FIRMWARE_REG_7 0
#define REGNUM_TOPAZHP_TOP_CR_FIRMWARE_REG_7 0x030C
#define SIGNED_TOPAZHP_TOP_CR_FIRMWARE_REG_7 0

/* Register CR_MTX_DEBUG_MSTR */
#define TOPAZHP_TOP_CR_MTX_DEBUG_MSTR 0x0044
#define MASK_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_IN 0x00000003
#define SHIFT_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_IN 0
#define REGNUM_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_IN 0x0044
#define SIGNED_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_IN 0

#define MASK_TOPAZHP_TOP_CR_MTX_MSTR_DBG_IS_SLAVE 0x00000004
#define SHIFT_TOPAZHP_TOP_CR_MTX_MSTR_DBG_IS_SLAVE 2
#define REGNUM_TOPAZHP_TOP_CR_MTX_MSTR_DBG_IS_SLAVE 0x0044
#define SIGNED_TOPAZHP_TOP_CR_MTX_MSTR_DBG_IS_SLAVE 0

#define MASK_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_OUT 0x00000018
#define SHIFT_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_OUT 3
#define REGNUM_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_OUT 0x0044
#define SIGNED_TOPAZHP_TOP_CR_MTX_MSTR_DBG_GPIO_OUT 0

#define MASK_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANKS 0x00000F00
#define SHIFT_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANKS 8
#define REGNUM_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANKS 0x0044
#define SIGNED_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANKS 0

#define MASK_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANK_SIZE 0x000F0000
#define SHIFT_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANK_SIZE 16
#define REGNUM_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANK_SIZE 0x0044
#define SIGNED_TOPAZHP_TOP_CR_MTX_MSTR_RAM_BANK_SIZE 0

#define MASK_TOPAZHP_TOP_CR_MTX_MSTR_LAST_RAM_BANK_SIZE 0x0F000000
#define SHIFT_TOPAZHP_TOP_CR_MTX_MSTR_LAST_RAM_BANK_SIZE 24
#define REGNUM_TOPAZHP_TOP_CR_MTX_MSTR_LAST_RAM_BANK_SIZE 0x0044
#define SIGNED_TOPAZHP_TOP_CR_MTX_MSTR_LAST_RAM_BANK_SIZE 0

/* Register CR_MULTICORE_CORE_SEL_0 */
#define TOPAZHP_TOP_CR_MULTICORE_CORE_SEL_0 0x0050
#define MASK_TOPAZHP_TOP_CR_DMAC_MTX_SELECT 0x00000007
#define SHIFT_TOPAZHP_TOP_CR_DMAC_MTX_SELECT 0
#define REGNUM_TOPAZHP_TOP_CR_DMAC_MTX_SELECT 0x0050
#define SIGNED_TOPAZHP_TOP_CR_DMAC_MTX_SELECT 0

#define MASK_TOPAZHP_TOP_CR_WRITES_MTX_ALL 0x40000000
#define SHIFT_TOPAZHP_TOP_CR_WRITES_MTX_ALL 30
#define REGNUM_TOPAZHP_TOP_CR_WRITES_MTX_ALL 0x0050
#define SIGNED_TOPAZHP_TOP_CR_WRITES_MTX_ALL 0

#define MASK_TOPAZHP_TOP_CR_WRITES_CORE_ALL 0x80000000
#define SHIFT_TOPAZHP_TOP_CR_WRITES_CORE_ALL 31
#define REGNUM_TOPAZHP_TOP_CR_WRITES_CORE_ALL 0x0050
#define SIGNED_TOPAZHP_TOP_CR_WRITES_CORE_ALL 0

/* Register CR_TOPAZHP_AUTO_CLOCK_GATING */
#define TOPAZHP_CR_TOPAZHP_AUTO_CLOCK_GATING 0x0024
#define MASK_TOPAZHP_CR_TOPAZHP_IPE0_AUTO_CLK_GATE 0x00000001
#define SHIFT_TOPAZHP_CR_TOPAZHP_IPE0_AUTO_CLK_GATE 0
#define REGNUM_TOPAZHP_CR_TOPAZHP_IPE0_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_IPE0_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_IPE1_AUTO_CLK_GATE 0x00000002
#define SHIFT_TOPAZHP_CR_TOPAZHP_IPE1_AUTO_CLK_GATE 1
#define REGNUM_TOPAZHP_CR_TOPAZHP_IPE1_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_IPE1_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_SPE0_AUTO_CLK_GATE 0x00000004
#define SHIFT_TOPAZHP_CR_TOPAZHP_SPE0_AUTO_CLK_GATE 2
#define REGNUM_TOPAZHP_CR_TOPAZHP_SPE0_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_SPE0_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_SPE1_AUTO_CLK_GATE 0x00000008
#define SHIFT_TOPAZHP_CR_TOPAZHP_SPE1_AUTO_CLK_GATE 3
#define REGNUM_TOPAZHP_CR_TOPAZHP_SPE1_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_SPE1_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_H264COMP4X4_AUTO_CLK_GATE 0x00000010
#define SHIFT_TOPAZHP_CR_TOPAZHP_H264COMP4X4_AUTO_CLK_GATE 4
#define REGNUM_TOPAZHP_CR_TOPAZHP_H264COMP4X4_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_H264COMP4X4_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_H264COMP8X8_AUTO_CLK_GATE 0x00000020
#define SHIFT_TOPAZHP_CR_TOPAZHP_H264COMP8X8_AUTO_CLK_GATE 5
#define REGNUM_TOPAZHP_CR_TOPAZHP_H264COMP8X8_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_H264COMP8X8_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_H264COMP16X16_AUTO_CLK_GATE 0x00000040
#define SHIFT_TOPAZHP_CR_TOPAZHP_H264COMP16X16_AUTO_CLK_GATE 6
#define REGNUM_TOPAZHP_CR_TOPAZHP_H264COMP16X16_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_H264COMP16X16_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_JMCOMP_AUTO_CLK_GATE 0x00000080
#define SHIFT_TOPAZHP_CR_TOPAZHP_JMCOMP_AUTO_CLK_GATE 7
#define REGNUM_TOPAZHP_CR_TOPAZHP_JMCOMP_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_JMCOMP_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_VLC_AUTO_CLK_GATE 0x00000200
#define SHIFT_TOPAZHP_CR_TOPAZHP_VLC_AUTO_CLK_GATE 9
#define REGNUM_TOPAZHP_CR_TOPAZHP_VLC_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_VLC_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_DEB_AUTO_CLK_GATE 0x00000400
#define SHIFT_TOPAZHP_CR_TOPAZHP_DEB_AUTO_CLK_GATE 10
#define REGNUM_TOPAZHP_CR_TOPAZHP_DEB_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_DEB_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_PC_DM_AUTO_CLK_GATE 0x00000800
#define SHIFT_TOPAZHP_CR_TOPAZHP_PC_DM_AUTO_CLK_GATE 11
#define REGNUM_TOPAZHP_CR_TOPAZHP_PC_DM_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_PC_DM_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_PC_DMS_AUTO_CLK_GATE 0x00001000
#define SHIFT_TOPAZHP_CR_TOPAZHP_PC_DMS_AUTO_CLK_GATE 12
#define REGNUM_TOPAZHP_CR_TOPAZHP_PC_DMS_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_PC_DMS_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_CABAC_AUTO_CLK_GATE 0x00002000
#define SHIFT_TOPAZHP_CR_TOPAZHP_CABAC_AUTO_CLK_GATE 13
#define REGNUM_TOPAZHP_CR_TOPAZHP_CABAC_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_CABAC_AUTO_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_AUTO_CLK_GATE 0x00008000
#define SHIFT_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_AUTO_CLK_GATE 15
#define REGNUM_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_AUTO_CLK_GATE 0x0024
#define SIGNED_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_AUTO_CLK_GATE 0

/* Register CR_TOPAZHP_MAN_CLOCK_GATING */
#define TOPAZHP_CR_TOPAZHP_MAN_CLOCK_GATING 0x0028
#define MASK_TOPAZHP_CR_TOPAZHP_IPE0_MAN_CLK_GATE 0x00000001
#define SHIFT_TOPAZHP_CR_TOPAZHP_IPE0_MAN_CLK_GATE 0
#define REGNUM_TOPAZHP_CR_TOPAZHP_IPE0_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_IPE0_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_IPE1_MAN_CLK_GATE 0x00000002
#define SHIFT_TOPAZHP_CR_TOPAZHP_IPE1_MAN_CLK_GATE 1
#define REGNUM_TOPAZHP_CR_TOPAZHP_IPE1_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_IPE1_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_SPE0_MAN_CLK_GATE 0x00000004
#define SHIFT_TOPAZHP_CR_TOPAZHP_SPE0_MAN_CLK_GATE 2
#define REGNUM_TOPAZHP_CR_TOPAZHP_SPE0_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_SPE0_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_SPE1_MAN_CLK_GATE 0x00000008
#define SHIFT_TOPAZHP_CR_TOPAZHP_SPE1_MAN_CLK_GATE 3
#define REGNUM_TOPAZHP_CR_TOPAZHP_SPE1_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_SPE1_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_H264COMP4X4_MAN_CLK_GATE 0x00000010
#define SHIFT_TOPAZHP_CR_TOPAZHP_H264COMP4X4_MAN_CLK_GATE 4
#define REGNUM_TOPAZHP_CR_TOPAZHP_H264COMP4X4_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_H264COMP4X4_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_H264COMP8X8_MAN_CLK_GATE 0x00000020
#define SHIFT_TOPAZHP_CR_TOPAZHP_H264COMP8X8_MAN_CLK_GATE 5
#define REGNUM_TOPAZHP_CR_TOPAZHP_H264COMP8X8_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_H264COMP8X8_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_H264COMP16X16_MAN_CLK_GATE 0x00000040
#define SHIFT_TOPAZHP_CR_TOPAZHP_H264COMP16X16_MAN_CLK_GATE 6
#define REGNUM_TOPAZHP_CR_TOPAZHP_H264COMP16X16_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_H264COMP16X16_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_JMCOMP_MAN_CLK_GATE 0x00000080
#define SHIFT_TOPAZHP_CR_TOPAZHP_JMCOMP_MAN_CLK_GATE 7
#define REGNUM_TOPAZHP_CR_TOPAZHP_JMCOMP_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_JMCOMP_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_PREFETCH_MAN_CLK_GATE 0x00000100
#define SHIFT_TOPAZHP_CR_TOPAZHP_PREFETCH_MAN_CLK_GATE 8
#define REGNUM_TOPAZHP_CR_TOPAZHP_PREFETCH_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_PREFETCH_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_VLC_MAN_CLK_GATE 0x00000200
#define SHIFT_TOPAZHP_CR_TOPAZHP_VLC_MAN_CLK_GATE 9
#define REGNUM_TOPAZHP_CR_TOPAZHP_VLC_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_VLC_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_DEB_MAN_CLK_GATE 0x00000400
#define SHIFT_TOPAZHP_CR_TOPAZHP_DEB_MAN_CLK_GATE 10
#define REGNUM_TOPAZHP_CR_TOPAZHP_DEB_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_DEB_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_PC_DM_MAN_CLK_GATE 0x00000800
#define SHIFT_TOPAZHP_CR_TOPAZHP_PC_DM_MAN_CLK_GATE 11
#define REGNUM_TOPAZHP_CR_TOPAZHP_PC_DM_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_PC_DM_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_PC_DMS_MAN_CLK_GATE 0x00001000
#define SHIFT_TOPAZHP_CR_TOPAZHP_PC_DMS_MAN_CLK_GATE 12
#define REGNUM_TOPAZHP_CR_TOPAZHP_PC_DMS_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_PC_DMS_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_CABAC_MAN_CLK_GATE 0x00002000
#define SHIFT_TOPAZHP_CR_TOPAZHP_CABAC_MAN_CLK_GATE 13
#define REGNUM_TOPAZHP_CR_TOPAZHP_CABAC_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_CABAC_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_LRITC_MAN_CLK_GATE 0x00004000
#define SHIFT_TOPAZHP_CR_TOPAZHP_LRITC_MAN_CLK_GATE 14
#define REGNUM_TOPAZHP_CR_TOPAZHP_LRITC_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_LRITC_MAN_CLK_GATE 0

#define MASK_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_MAN_CLK_GATE 0x00008000
#define SHIFT_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_MAN_CLK_GATE 15
#define REGNUM_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_MAN_CLK_GATE 0x0028
#define SIGNED_TOPAZHP_CR_TOPAZHP_INPUT_SCALER_MAN_CLK_GATE 0

/* Register CR_TOPAZHP_SRST */
#define TOPAZHP_CR_TOPAZHP_SRST     0x0000
#define MASK_TOPAZHP_CR_TOPAZHP_IPE_SOFT_RESET 0x00000001
#define SHIFT_TOPAZHP_CR_TOPAZHP_IPE_SOFT_RESET 0
#define REGNUM_TOPAZHP_CR_TOPAZHP_IPE_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_IPE_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_SPE_SOFT_RESET 0x00000002
#define SHIFT_TOPAZHP_CR_TOPAZHP_SPE_SOFT_RESET 1
#define REGNUM_TOPAZHP_CR_TOPAZHP_SPE_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_SPE_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_PC_SOFT_RESET 0x00000004
#define SHIFT_TOPAZHP_CR_TOPAZHP_PC_SOFT_RESET 2
#define REGNUM_TOPAZHP_CR_TOPAZHP_PC_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_PC_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_H264COMP_SOFT_RESET 0x00000008
#define SHIFT_TOPAZHP_CR_TOPAZHP_H264COMP_SOFT_RESET 3
#define REGNUM_TOPAZHP_CR_TOPAZHP_H264COMP_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_H264COMP_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_JMCOMP_SOFT_RESET 0x00000010
#define SHIFT_TOPAZHP_CR_TOPAZHP_JMCOMP_SOFT_RESET 4
#define REGNUM_TOPAZHP_CR_TOPAZHP_JMCOMP_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_JMCOMP_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_PREFETCH_SOFT_RESET 0x00000020
#define SHIFT_TOPAZHP_CR_TOPAZHP_PREFETCH_SOFT_RESET 5
#define REGNUM_TOPAZHP_CR_TOPAZHP_PREFETCH_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_PREFETCH_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_VLC_SOFT_RESET 0x00000040
#define SHIFT_TOPAZHP_CR_TOPAZHP_VLC_SOFT_RESET 6
#define REGNUM_TOPAZHP_CR_TOPAZHP_VLC_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_VLC_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_DB_SOFT_RESET 0x00000080
#define SHIFT_TOPAZHP_CR_TOPAZHP_DB_SOFT_RESET 7
#define REGNUM_TOPAZHP_CR_TOPAZHP_DB_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_DB_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_LTRITC_SOFT_RESET 0x00000100
#define SHIFT_TOPAZHP_CR_TOPAZHP_LTRITC_SOFT_RESET 8
#define REGNUM_TOPAZHP_CR_TOPAZHP_LTRITC_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_LTRITC_SOFT_RESET 0

#define MASK_TOPAZHP_CR_TOPAZHP_SCALER_SOFT_RESET 0x00000200
#define SHIFT_TOPAZHP_CR_TOPAZHP_SCALER_SOFT_RESET 9
#define REGNUM_TOPAZHP_CR_TOPAZHP_SCALER_SOFT_RESET 0x0000
#define SIGNED_TOPAZHP_CR_TOPAZHP_SCALER_SOFT_RESET 0

/* Register CR_MMU_STATUS */
#define TOPAZHP_TOP_CR_MMU_STATUS   0x001C
#define MASK_TOPAZHP_TOP_CR_MMU_PF_N_RW 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_MMU_PF_N_RW 0
#define REGNUM_TOPAZHP_TOP_CR_MMU_PF_N_RW 0x001C
#define SIGNED_TOPAZHP_TOP_CR_MMU_PF_N_RW 0

#define MASK_TOPAZHP_TOP_CR_MMU_FAULT_ADDR 0xFFFFF000
#define SHIFT_TOPAZHP_TOP_CR_MMU_FAULT_ADDR 12
#define REGNUM_TOPAZHP_TOP_CR_MMU_FAULT_ADDR 0x001C
#define SIGNED_TOPAZHP_TOP_CR_MMU_FAULT_ADDR 0

/* Register CR_MMU_MEM_REQ */
#define TOPAZHP_TOP_CR_MMU_MEM_REQ  0x0020
#define MASK_TOPAZHP_TOP_CR_MEM_REQ_STAT_READS 0x000000FF
#define SHIFT_TOPAZHP_TOP_CR_MEM_REQ_STAT_READS 0
#define REGNUM_TOPAZHP_TOP_CR_MEM_REQ_STAT_READS 0x0020
#define SIGNED_TOPAZHP_TOP_CR_MEM_REQ_STAT_READS 0

/* Register CR_MMU_CONTROL0 */
#define TOPAZHP_TOP_CR_MMU_CONTROL0 0x0024
#define MASK_TOPAZHP_TOP_CR_MMU_NOREORDER 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_MMU_NOREORDER 0
#define REGNUM_TOPAZHP_TOP_CR_MMU_NOREORDER 0x0024
#define SIGNED_TOPAZHP_TOP_CR_MMU_NOREORDER 0

#define MASK_TOPAZHP_TOP_CR_MMU_PAUSE 0x00000002
#define SHIFT_TOPAZHP_TOP_CR_MMU_PAUSE 1
#define REGNUM_TOPAZHP_TOP_CR_MMU_PAUSE 0x0024
#define SIGNED_TOPAZHP_TOP_CR_MMU_PAUSE 0

#define MASK_TOPAZHP_TOP_CR_MMU_FLUSH 0x00000004
#define SHIFT_TOPAZHP_TOP_CR_MMU_FLUSH 2
#define REGNUM_TOPAZHP_TOP_CR_MMU_FLUSH 0x0024
#define SIGNED_TOPAZHP_TOP_CR_MMU_FLUSH 0

#define MASK_TOPAZHP_TOP_CR_MMU_INVALDC 0x00000008
#define SHIFT_TOPAZHP_TOP_CR_MMU_INVALDC 3
#define REGNUM_TOPAZHP_TOP_CR_MMU_INVALDC 0x0024
#define SIGNED_TOPAZHP_TOP_CR_MMU_INVALDC 0

#define MASK_TOPAZHP_TOP_CR_FLOWRATE_TOPAZ 0x00000700
#define SHIFT_TOPAZHP_TOP_CR_FLOWRATE_TOPAZ 8
#define REGNUM_TOPAZHP_TOP_CR_FLOWRATE_TOPAZ 0x0024
#define SIGNED_TOPAZHP_TOP_CR_FLOWRATE_TOPAZ 0

#define MASK_TOPAZHP_TOP_CR_MMU_BYPASS_TOPAZ 0x00010000
#define SHIFT_TOPAZHP_TOP_CR_MMU_BYPASS_TOPAZ 16
#define REGNUM_TOPAZHP_TOP_CR_MMU_BYPASS_TOPAZ 0x0024
#define SIGNED_TOPAZHP_TOP_CR_MMU_BYPASS_TOPAZ 0

/* Register CR_MMU_CONTROL1 */
#define TOPAZHP_TOP_CR_MMU_CONTROL1 0x0028
#define MASK_TOPAZHP_TOP_CR_MMU_TTE_THRESHOLD 0x00000FFF
#define SHIFT_TOPAZHP_TOP_CR_MMU_TTE_THRESHOLD 0
#define REGNUM_TOPAZHP_TOP_CR_MMU_TTE_THRESHOLD 0x0028
#define SIGNED_TOPAZHP_TOP_CR_MMU_TTE_THRESHOLD 0

#define MASK_TOPAZHP_TOP_CR_MMU_ADT_TTE 0x000FF000
#define SHIFT_TOPAZHP_TOP_CR_MMU_ADT_TTE 12
#define REGNUM_TOPAZHP_TOP_CR_MMU_ADT_TTE 0x0028
#define SIGNED_TOPAZHP_TOP_CR_MMU_ADT_TTE 0

#define MASK_TOPAZHP_TOP_CR_MMU_BEST_COUNT 0x0FF00000
#define SHIFT_TOPAZHP_TOP_CR_MMU_BEST_COUNT 20
#define REGNUM_TOPAZHP_TOP_CR_MMU_BEST_COUNT 0x0028
#define SIGNED_TOPAZHP_TOP_CR_MMU_BEST_COUNT 0

#define MASK_TOPAZHP_TOP_CR_MMU_PAGE_SIZE 0xF0000000
#define SHIFT_TOPAZHP_TOP_CR_MMU_PAGE_SIZE 28
#define REGNUM_TOPAZHP_TOP_CR_MMU_PAGE_SIZE 0x0028
#define SIGNED_TOPAZHP_TOP_CR_MMU_PAGE_SIZE 0

/* Register CR_MMU_CONTROL2 */
#define TOPAZHP_TOP_CR_MMU_CONTROL2 0x002C
#define MASK_TOPAZHP_TOP_CR_MMU_ENABLE_36BIT_ADDRESSING 0x00000001
#define SHIFT_TOPAZHP_TOP_CR_MMU_ENABLE_36BIT_ADDRESSING 0
#define REGNUM_TOPAZHP_TOP_CR_MMU_ENABLE_36BIT_ADDRESSING 0x002C
#define SIGNED_TOPAZHP_TOP_CR_MMU_ENABLE_36BIT_ADDRESSING 0

#define MASK_TOPAZHP_TOP_CR_MMU_TILING_SCHEME 0x00000008
#define SHIFT_TOPAZHP_TOP_CR_MMU_TILING_SCHEME 3
#define REGNUM_TOPAZHP_TOP_CR_MMU_TILING_SCHEME 0x002C
#define SIGNED_TOPAZHP_TOP_CR_MMU_TILING_SCHEME 0

/* Table MMU_DIR_LIST_BASE */

/* Register CR_MMU_DIR_LIST_BASE */
#define TOPAZHP_TOP_CR_MMU_DIR_LIST_BASE(X) (0x0030 + (4 * (X)))
#define MASK_TOPAZHP_TOP_CR_MMU_DIR_LIST_BASE_ADDR 0xFFFFFFF0
#define SHIFT_TOPAZHP_TOP_CR_MMU_DIR_LIST_BASE_ADDR 4
#define REGNUM_TOPAZHP_TOP_CR_MMU_DIR_LIST_BASE_ADDR 0x0030
#define SIGNED_TOPAZHP_TOP_CR_MMU_DIR_LIST_BASE_ADDR 0

/* Number of entries in table MMU_DIR_LIST_BASE */

#define TOPAZHP_TOP_MMU_DIR_LIST_BASE_SIZE_UINT32 1
#define TOPAZHP_TOP_MMU_DIR_LIST_BASE_NUM_ENTRIES 1

/* Table MMU_TILE */

/* Register CR_MMU_TILE */
#define TOPAZHP_TOP_CR_MMU_TILE(X)  (0x0038 + (4 * (X)))
#define MASK_TOPAZHP_TOP_CR_TILE_MIN_ADDR 0x00000FFF
#define SHIFT_TOPAZHP_TOP_CR_TILE_MIN_ADDR 0
#define REGNUM_TOPAZHP_TOP_CR_TILE_MIN_ADDR 0x0038
#define SIGNED_TOPAZHP_TOP_CR_TILE_MIN_ADDR 0

#define MASK_TOPAZHP_TOP_CR_TILE_MAX_ADDR 0x00FFF000
#define SHIFT_TOPAZHP_TOP_CR_TILE_MAX_ADDR 12
#define REGNUM_TOPAZHP_TOP_CR_TILE_MAX_ADDR 0x0038
#define SIGNED_TOPAZHP_TOP_CR_TILE_MAX_ADDR 0

#define MASK_TOPAZHP_TOP_CR_TILE_STRIDE 0x07000000
#define SHIFT_TOPAZHP_TOP_CR_TILE_STRIDE 24
#define REGNUM_TOPAZHP_TOP_CR_TILE_STRIDE 0x0038
#define SIGNED_TOPAZHP_TOP_CR_TILE_STRIDE 0

#define MASK_TOPAZHP_TOP_CR_TILE_ENABLE 0x10000000
#define SHIFT_TOPAZHP_TOP_CR_TILE_ENABLE 28
#define REGNUM_TOPAZHP_TOP_CR_TILE_ENABLE 0x0038
#define SIGNED_TOPAZHP_TOP_CR_TILE_ENABLE 0

#define MASK_TOPAZHP_TOP_CR_TILE_128BYTE_INTERLEAVE 0x20000000
#define SHIFT_TOPAZHP_TOP_CR_TILE_128BYTE_INTERLEAVE 29
#define REGNUM_TOPAZHP_TOP_CR_TILE_128BYTE_INTERLEAVE 0x0038
#define SIGNED_TOPAZHP_TOP_CR_TILE_128BYTE_INTERLEAVE 0

/* Number of entries in table MMU_TILE */

#define TOPAZHP_TOP_MMU_TILE_SIZE_UINT32 2
#define TOPAZHP_TOP_MMU_TILE_NUM_ENTRIES 2

/* Table MMU_TILE_EXT */

/* Register CR_MMU_TILE_EXT */
#define TOPAZHP_TOP_CR_MMU_TILE_EXT(X) (0x0080 + (4 * (X)))
#define MASK_TOPAZHP_TOP_CR_TILE_MIN_ADDR_EXT 0x000000FF
#define SHIFT_TOPAZHP_TOP_CR_TILE_MIN_ADDR_EXT 0
#define REGNUM_TOPAZHP_TOP_CR_TILE_MIN_ADDR_EXT 0x0080
#define SIGNED_TOPAZHP_TOP_CR_TILE_MIN_ADDR_EXT 0

#define MASK_TOPAZHP_TOP_CR_TILE_MAX_ADDR_EXT 0x0000FF00
#define SHIFT_TOPAZHP_TOP_CR_TILE_MAX_ADDR_EXT 8
#define REGNUM_TOPAZHP_TOP_CR_TILE_MAX_ADDR_EXT 0x0080
#define SIGNED_TOPAZHP_TOP_CR_TILE_MAX_ADDR_EXT 0

/* Number of entries in table MMU_TILE_EXT */

#define TOPAZHP_TOP_MMU_TILE_EXT_SIZE_UINT32 2
#define TOPAZHP_TOP_MMU_TILE_EXT_NUM_ENTRIES 2

#define TOPAZHP_CR_PROC_ESB_ACCESS_WORD0 0x00F0

/* Register CR_PROC_ESB_ACCESS_CONTROL */
#define TOPAZHP_CR_PROC_ESB_ACCESS_CONTROL 0x00EC
#define MASK_TOPAZHP_CR_PROC_ESB_ADDR 0x00003FF0
#define SHIFT_TOPAZHP_CR_PROC_ESB_ADDR 4
#define REGNUM_TOPAZHP_CR_PROC_ESB_ADDR 0x00EC
#define SIGNED_TOPAZHP_CR_PROC_ESB_ADDR 0

#define MASK_TOPAZHP_CR_PROC_ESB_READ_N_WRITE 0x00010000
#define SHIFT_TOPAZHP_CR_PROC_ESB_READ_N_WRITE 16
#define REGNUM_TOPAZHP_CR_PROC_ESB_READ_N_WRITE 0x00EC
#define SIGNED_TOPAZHP_CR_PROC_ESB_READ_N_WRITE 0

#define MASK_TOPAZHP_CR_PROC_ESB_OP_VALID 0x00020000
#define SHIFT_TOPAZHP_CR_PROC_ESB_OP_VALID 17
#define REGNUM_TOPAZHP_CR_PROC_ESB_OP_VALID 0x00EC
#define SIGNED_TOPAZHP_CR_PROC_ESB_OP_VALID 0

#define MASK_TOPAZHP_CR_PROC_ACCESS_FLAG 0x03000000
#define SHIFT_TOPAZHP_CR_PROC_ACCESS_FLAG 24
#define REGNUM_TOPAZHP_CR_PROC_ACCESS_FLAG 0x00EC
#define SIGNED_TOPAZHP_CR_PROC_ACCESS_FLAG 0

/* Register CR_SECURE_CONFIG */
#define TOPAZHP_TOP_CR_SECURE_CONFIG 0x0200

/* Register CR_VLC_MPEG4_CFG */
#define TOPAZ_VLC_CR_VLC_MPEG4_CFG  0x0064
#define MASK_TOPAZ_VLC_CR_RSIZE     0x00000007
#define SHIFT_TOPAZ_VLC_CR_RSIZE    0
#define REGNUM_TOPAZ_VLC_CR_RSIZE   0x0064
#define SIGNED_TOPAZ_VLC_CR_RSIZE   0

/* RC Config registers and tables */
#define TOPAZHP_TOP_CR_RC_CONFIG_TABLE7 0x012C
#define TOPAZHP_TOP_CR_RC_CONFIG_TABLE6 0x0124
#define TOPAZHP_TOP_CR_RC_CONFIG_TABLE4 0x0128
#define TOPAZHP_TOP_CR_RC_CONFIG_TABLE5 0x0130
#define TOPAZHP_CR_RC_CONFIG_REG8       0x0344
#define TOPAZHP_CR_RC_CONFIG_REG9       0x0184
#define TOPAZHP_CR_JMCOMP_RC_STATS      0x0340

/* Register CR_TOPAZHP_CORE_DES1 */
#define TOPAZHP_TOP_CR_TOPAZHP_CORE_DES1 0x03E0
#define MASK_TOPAZHP_TOP_CR_TOPAZHP_SCALER_SUPPORTED 0x00000080
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_SCALER_SUPPORTED 7
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_SCALER_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_SCALER_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_GENERATE_PERFORMANCE_STORE 0x00000100
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_GENERATE_PERFORMANCE_STORE 8
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_GENERATE_PERFORMANCE_STORE 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_GENERATE_PERFORMANCE_STORE 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_LOSSLESS_SUPPORTED 0x00000200
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_LOSSLESS_SUPPORTED 9
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_LOSSLESS_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_LOSSLESS_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_CUSTOM_QUANT_SUPPORTED 0x00000400
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_CUSTOM_QUANT_SUPPORTED 10
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_CUSTOM_QUANT_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_CUSTOM_QUANT_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MPEG2_SUPPORTED 0x00000800
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MPEG2_SUPPORTED 11
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MPEG2_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MPEG2_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_SUBSET 0x00001000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_SUBSET 12
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_SUBSET 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_SUBSET 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_ALL 0x00002000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_ALL 13
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_ALL 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_SIGNATURES_SUPPORTED_ALL 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_ME_SUPPORTED 0x00004000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_ME_SUPPORTED 14
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_ME_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_ME_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_SUPPORTED 0x00008000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_SUPPORTED 15
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_WEIGHTED_PRED_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_2_REF_ON_P_PIC_SUPPORTED 0x00010000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_2_REF_ON_P_PIC_SUPPORTED 16
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_2_REF_ON_P_PIC_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_2_REF_ON_P_PIC_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_SPATIAL_DIRECT_SUPPORTED 0x00020000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_SPATIAL_DIRECT_SUPPORTED 17
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_SPATIAL_DIRECT_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_SPATIAL_DIRECT_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_MULTIPASS_SUPPORTED 0x00040000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_MULTIPASS_SUPPORTED 18
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_MULTIPASS_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_MULTIPASS_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_DEFAULT_TABLES_SUPPORTED 0x00080000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_DEFAULT_TABLES_SUPPORTED 19
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_DEFAULT_TABLES_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_DEFAULT_TABLES_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_8X8_TRANSFORM_SUPPORTED 0x00100000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_8X8_TRANSFORM_SUPPORTED 20
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_8X8_TRANSFORM_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_8X8_TRANSFORM_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_INTERLACED_SUPPORTED 0x00200000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_INTERLACED_SUPPORTED 21
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_INTERLACED_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_INTERLACED_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_B_PIC_SUPPORTED 0x00400000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_B_PIC_SUPPORTED 22
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_B_PIC_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_B_PIC_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_16X8_8X16_SUPPORTED 0x00800000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_16X8_8X16_SUPPORTED 23
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_16X8_8X16_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_16X8_8X16_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_CABAC_SUPPORTED 0x01000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_CABAC_SUPPORTED 24
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_CABAC_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_CABAC_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_SLAVE_JPEG_SUPPORTED 0x02000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_SLAVE_JPEG_SUPPORTED 25
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_SLAVE_JPEG_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_SLAVE_JPEG_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_JPEG_SUPPORTED 0x04000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_JPEG_SUPPORTED 26
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_JPEG_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_JPEG_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H263_SUPPORTED 0x08000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H263_SUPPORTED 27
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H263_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H263_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MPEG4_SUPPORTED 0x10000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MPEG4_SUPPORTED 28
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MPEG4_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MPEG4_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_H264_SUPPORTED 0x20000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_H264_SUPPORTED 29
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_H264_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_H264_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_DMAC_SUPPORTED 0x40000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_DMAC_SUPPORTED 30
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_DMAC_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_DMAC_SUPPORTED 0

#define MASK_TOPAZHP_TOP_CR_TOPAZHP_MMU_SUPPORTED 0x80000000
#define SHIFT_TOPAZHP_TOP_CR_TOPAZHP_MMU_SUPPORTED 31
#define REGNUM_TOPAZHP_TOP_CR_TOPAZHP_MMU_SUPPORTED 0x03E0
#define SIGNED_TOPAZHP_TOP_CR_TOPAZHP_MMU_SUPPORTED 0
#endif
