// SPDX-License-Identifier: GPL-2.0
/*
 * Self tests for device tree subsystem
 */

#define pr_fmt(fmt) "### dt-test ### " fmt

#include <linux/memblock.h>
#include <linux/clk.h>
#include <linux/dma-direct.h> /* to test phys_to_dma/dma_to_phys */
#include <linux/err.h>
#include <linux/errno.h>
#include <linux/hashtable.h>
#include <linux/libfdt.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_fdt.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/list.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/kernel.h>

#include <linux/i2c.h>
#include <linux/i2c-mux.h>
#include <linux/gpio/driver.h>

#include <linux/bitops.h>

#include "of_private.h"

static struct unittest_results {
	int passed;
	int failed;
} unittest_results;

#define unittest(result, fmt, ...) ({ \
	bool failed = !(result); \
	if (failed) { \
		unittest_results.failed++; \
		pr_err("FAIL %s():%i " fmt, __func__, __LINE__, ##__VA_ARGS__); \
	} else { \
		unittest_results.passed++; \
		pr_debug("pass %s():%i\n", __func__, __LINE__); \
	} \
	failed; \
})

/*
 * Expected message may have a message level other than KERN_INFO.
 * Print the expected message only if the current loglevel will allow
 * the actual message to print.
 *
 * Do not use EXPECT_BEGIN() or EXPECT_END() for messages generated by
 * pr_debug().
 */
#define EXPECT_BEGIN(level, fmt, ...) \
	printk(level pr_fmt("EXPECT \\ : ") fmt, ##__VA_ARGS__)

#define EXPECT_END(level, fmt, ...) \
	printk(level pr_fmt("EXPECT / : ") fmt, ##__VA_ARGS__)

static void __init of_unittest_find_node_by_name(void)
{
	struct device_node *np;
	const char *options, *name;

	np = of_find_node_by_path("/testcase-data");
	name = kasprintf(GFP_KERNEL, "%pOF", np);
	unittest(np && !strcmp("/testcase-data", name),
		"find /testcase-data failed\n");
	of_node_put(np);
	kfree(name);

	/* Test if trailing '/' works */
	np = of_find_node_by_path("/testcase-data/");
	unittest(!np, "trailing '/' on /testcase-data/ should fail\n");

	np = of_find_node_by_path("/testcase-data/phandle-tests/consumer-a");
	name = kasprintf(GFP_KERNEL, "%pOF", np);
	unittest(np && !strcmp("/testcase-data/phandle-tests/consumer-a", name),
		"find /testcase-data/phandle-tests/consumer-a failed\n");
	of_node_put(np);
	kfree(name);

	np = of_find_node_by_path("testcase-alias");
	name = kasprintf(GFP_KERNEL, "%pOF", np);
	unittest(np && !strcmp("/testcase-data", name),
		"find testcase-alias failed\n");
	of_node_put(np);
	kfree(name);

	/* Test if trailing '/' works on aliases */
	np = of_find_node_by_path("testcase-alias/");
	unittest(!np, "trailing '/' on testcase-alias/ should fail\n");

	np = of_find_node_by_path("testcase-alias/phandle-tests/consumer-a");
	name = kasprintf(GFP_KERNEL, "%pOF", np);
	unittest(np && !strcmp("/testcase-data/phandle-tests/consumer-a", name),
		"find testcase-alias/phandle-tests/consumer-a failed\n");
	of_node_put(np);
	kfree(name);

	np = of_find_node_by_path("/testcase-data/missing-path");
	unittest(!np, "non-existent path returned node %pOF\n", np);
	of_node_put(np);

	np = of_find_node_by_path("missing-alias");
	unittest(!np, "non-existent alias returned node %pOF\n", np);
	of_node_put(np);

	np = of_find_node_by_path("testcase-alias/missing-path");
	unittest(!np, "non-existent alias with relative path returned node %pOF\n", np);
	of_node_put(np);

	np = of_find_node_opts_by_path("/testcase-data:testoption", &options);
	unittest(np && !strcmp("testoption", options),
		 "option path test failed\n");
	of_node_put(np);

	np = of_find_node_opts_by_path("/testcase-data:test/option", &options);
	unittest(np && !strcmp("test/option", options),
		 "option path test, subcase #1 failed\n");
	of_node_put(np);

	np = of_find_node_opts_by_path("/testcase-data/testcase-device1:test/option", &options);
	unittest(np && !strcmp("test/option", options),
		 "option path test, subcase #2 failed\n");
	of_node_put(np);

	np = of_find_node_opts_by_path("/testcase-data:testoption", NULL);
	unittest(np, "NULL option path test failed\n");
	of_node_put(np);

	np = of_find_node_opts_by_path("testcase-alias:testaliasoption",
				       &options);
	unittest(np && !strcmp("testaliasoption", options),
		 "option alias path test failed\n");
	of_node_put(np);

	np = of_find_node_opts_by_path("testcase-alias:test/alias/option",
				       &options);
	unittest(np && !strcmp("test/alias/option", options),
		 "option alias path test, subcase #1 failed\n");
	of_node_put(np);

	np = of_find_node_opts_by_path("testcase-alias:testaliasoption", NULL);
	unittest(np, "NULL option alias path test failed\n");
	of_node_put(np);

	options = "testoption";
	np = of_find_node_opts_by_path("testcase-alias", &options);
	unittest(np && !options, "option clearing test failed\n");
	of_node_put(np);

	options = "testoption";
	np = of_find_node_opts_by_path("/", &options);
	unittest(np && !options, "option clearing root node test failed\n");
	of_node_put(np);
}

static void __init of_unittest_dynamic(void)
{
	struct device_node *np;
	struct property *prop;

	np = of_find_node_by_path("/testcase-data");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	/* Array of 4 properties for the purpose of testing */
	prop = kcalloc(4, sizeof(*prop), GFP_KERNEL);
	if (!prop) {
		unittest(0, "kzalloc() failed\n");
		return;
	}

	/* Add a new property - should pass*/
	prop->name = "new-property";
	prop->value = "new-property-data";
	prop->length = strlen(prop->value) + 1;
	unittest(of_add_property(np, prop) == 0, "Adding a new property failed\n");

	/* Try to add an existing property - should fail */
	prop++;
	prop->name = "new-property";
	prop->value = "new-property-data-should-fail";
	prop->length = strlen(prop->value) + 1;
	unittest(of_add_property(np, prop) != 0,
		 "Adding an existing property should have failed\n");

	/* Try to modify an existing property - should pass */
	prop->value = "modify-property-data-should-pass";
	prop->length = strlen(prop->value) + 1;
	unittest(of_update_property(np, prop) == 0,
		 "Updating an existing property should have passed\n");

	/* Try to modify non-existent property - should pass*/
	prop++;
	prop->name = "modify-property";
	prop->value = "modify-missing-property-data-should-pass";
	prop->length = strlen(prop->value) + 1;
	unittest(of_update_property(np, prop) == 0,
		 "Updating a missing property should have passed\n");

	/* Remove property - should pass */
	unittest(of_remove_property(np, prop) == 0,
		 "Removing a property should have passed\n");

	/* Adding very large property - should pass */
	prop++;
	prop->name = "large-property-PAGE_SIZEx8";
	prop->length = PAGE_SIZE * 8;
	prop->value = kzalloc(prop->length, GFP_KERNEL);
	unittest(prop->value != NULL, "Unable to allocate large buffer\n");
	if (prop->value)
		unittest(of_add_property(np, prop) == 0,
			 "Adding a large property should have passed\n");
}

static int __init of_unittest_check_node_linkage(struct device_node *np)
{
	struct device_node *child;
	int count = 0, rc;

	for_each_child_of_node(np, child) {
		if (child->parent != np) {
			pr_err("Child node %pOFn links to wrong parent %pOFn\n",
				 child, np);
			rc = -EINVAL;
			goto put_child;
		}

		rc = of_unittest_check_node_linkage(child);
		if (rc < 0)
			goto put_child;
		count += rc;
	}

	return count + 1;
put_child:
	of_node_put(child);
	return rc;
}

static void __init of_unittest_check_tree_linkage(void)
{
	struct device_node *np;
	int allnode_count = 0, child_count;

	if (!of_root)
		return;

	for_each_of_allnodes(np)
		allnode_count++;
	child_count = of_unittest_check_node_linkage(of_root);

	unittest(child_count > 0, "Device node data structure is corrupted\n");
	unittest(child_count == allnode_count,
		 "allnodes list size (%i) doesn't match sibling lists size (%i)\n",
		 allnode_count, child_count);
	pr_debug("allnodes list size (%i); sibling lists size (%i)\n", allnode_count, child_count);
}

static void __init of_unittest_printf_one(struct device_node *np, const char *fmt,
					  const char *expected)
{
	unsigned char *buf;
	int buf_size;
	int size, i;

	buf_size = strlen(expected) + 10;
	buf = kmalloc(buf_size, GFP_KERNEL);
	if (!buf)
		return;

	/* Baseline; check conversion with a large size limit */
	memset(buf, 0xff, buf_size);
	size = snprintf(buf, buf_size - 2, fmt, np);

	/* use strcmp() instead of strncmp() here to be absolutely sure strings match */
	unittest((strcmp(buf, expected) == 0) && (buf[size+1] == 0xff),
		"sprintf failed; fmt='%s' expected='%s' rslt='%s'\n",
		fmt, expected, buf);

	/* Make sure length limits work */
	size++;
	for (i = 0; i < 2; i++, size--) {
		/* Clear the buffer, and make sure it works correctly still */
		memset(buf, 0xff, buf_size);
		snprintf(buf, size+1, fmt, np);
		unittest(strncmp(buf, expected, size) == 0 && (buf[size+1] == 0xff),
			"snprintf failed; size=%i fmt='%s' expected='%s' rslt='%s'\n",
			size, fmt, expected, buf);
	}
	kfree(buf);
}

static void __init of_unittest_printf(void)
{
	struct device_node *np;
	const char *full_name = "/testcase-data/platform-tests/test-device@1/dev@100";
	char phandle_str[16] = "";

	np = of_find_node_by_path(full_name);
	if (!np) {
		unittest(np, "testcase data missing\n");
		return;
	}

	num_to_str(phandle_str, sizeof(phandle_str), np->phandle, 0);

	of_unittest_printf_one(np, "%pOF",  full_name);
	of_unittest_printf_one(np, "%pOFf", full_name);
	of_unittest_printf_one(np, "%pOFn", "dev");
	of_unittest_printf_one(np, "%2pOFn", "dev");
	of_unittest_printf_one(np, "%5pOFn", "  dev");
	of_unittest_printf_one(np, "%pOFnc", "dev:test-sub-device");
	of_unittest_printf_one(np, "%pOFp", phandle_str);
	of_unittest_printf_one(np, "%pOFP", "dev@100");
	of_unittest_printf_one(np, "ABC %pOFP ABC", "ABC dev@100 ABC");
	of_unittest_printf_one(np, "%10pOFP", "   dev@100");
	of_unittest_printf_one(np, "%-10pOFP", "dev@100   ");
	of_unittest_printf_one(of_root, "%pOFP", "/");
	of_unittest_printf_one(np, "%pOFF", "----");
	of_unittest_printf_one(np, "%pOFPF", "dev@100:----");
	of_unittest_printf_one(np, "%pOFPFPc", "dev@100:----:dev@100:test-sub-device");
	of_unittest_printf_one(np, "%pOFc", "test-sub-device");
	of_unittest_printf_one(np, "%pOFC",
			"\"test-sub-device\",\"test-compat2\",\"test-compat3\"");
}

struct node_hash {
	struct hlist_node node;
	struct device_node *np;
};

static DEFINE_HASHTABLE(phandle_ht, 8);
static void __init of_unittest_check_phandles(void)
{
	struct device_node *np;
	struct node_hash *nh;
	struct hlist_node *tmp;
	int i, dup_count = 0, phandle_count = 0;

	for_each_of_allnodes(np) {
		if (!np->phandle)
			continue;

		hash_for_each_possible(phandle_ht, nh, node, np->phandle) {
			if (nh->np->phandle == np->phandle) {
				pr_info("Duplicate phandle! %i used by %pOF and %pOF\n",
					np->phandle, nh->np, np);
				dup_count++;
				break;
			}
		}

		nh = kzalloc(sizeof(*nh), GFP_KERNEL);
		if (!nh)
			return;

		nh->np = np;
		hash_add(phandle_ht, &nh->node, np->phandle);
		phandle_count++;
	}
	unittest(dup_count == 0, "Found %i duplicates in %i phandles\n",
		 dup_count, phandle_count);

	/* Clean up */
	hash_for_each_safe(phandle_ht, i, tmp, nh, node) {
		hash_del(&nh->node);
		kfree(nh);
	}
}

static void __init of_unittest_parse_phandle_with_args(void)
{
	struct device_node *np;
	struct of_phandle_args args;
	int i, rc;

	np = of_find_node_by_path("/testcase-data/phandle-tests/consumer-a");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	rc = of_count_phandle_with_args(np, "phandle-list", "#phandle-cells");
	unittest(rc == 7, "of_count_phandle_with_args() returned %i, expected 7\n", rc);

	for (i = 0; i < 8; i++) {
		bool passed = true;

		memset(&args, 0, sizeof(args));
		rc = of_parse_phandle_with_args(np, "phandle-list",
						"#phandle-cells", i, &args);

		/* Test the values from tests-phandle.dtsi */
		switch (i) {
		case 0:
			passed &= !rc;
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == (i + 1));
			break;
		case 1:
			passed &= !rc;
			passed &= (args.args_count == 2);
			passed &= (args.args[0] == (i + 1));
			passed &= (args.args[1] == 0);
			break;
		case 2:
			passed &= (rc == -ENOENT);
			break;
		case 3:
			passed &= !rc;
			passed &= (args.args_count == 3);
			passed &= (args.args[0] == (i + 1));
			passed &= (args.args[1] == 4);
			passed &= (args.args[2] == 3);
			break;
		case 4:
			passed &= !rc;
			passed &= (args.args_count == 2);
			passed &= (args.args[0] == (i + 1));
			passed &= (args.args[1] == 100);
			break;
		case 5:
			passed &= !rc;
			passed &= (args.args_count == 0);
			break;
		case 6:
			passed &= !rc;
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == (i + 1));
			break;
		case 7:
			passed &= (rc == -ENOENT);
			break;
		default:
			passed = false;
		}

		unittest(passed, "index %i - data error on node %pOF rc=%i\n",
			 i, args.np, rc);
	}

	/* Check for missing list property */
	memset(&args, 0, sizeof(args));
	rc = of_parse_phandle_with_args(np, "phandle-list-missing",
					"#phandle-cells", 0, &args);
	unittest(rc == -ENOENT, "expected:%i got:%i\n", -ENOENT, rc);
	rc = of_count_phandle_with_args(np, "phandle-list-missing",
					"#phandle-cells");
	unittest(rc == -ENOENT, "expected:%i got:%i\n", -ENOENT, rc);

	/* Check for missing cells property */
	memset(&args, 0, sizeof(args));

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-a: could not get #phandle-cells-missing for /testcase-data/phandle-tests/provider1");

	rc = of_parse_phandle_with_args(np, "phandle-list",
					"#phandle-cells-missing", 0, &args);

	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-a: could not get #phandle-cells-missing for /testcase-data/phandle-tests/provider1");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-a: could not get #phandle-cells-missing for /testcase-data/phandle-tests/provider1");

	rc = of_count_phandle_with_args(np, "phandle-list",
					"#phandle-cells-missing");

	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-a: could not get #phandle-cells-missing for /testcase-data/phandle-tests/provider1");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);

	/* Check for bad phandle in list */
	memset(&args, 0, sizeof(args));

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-a: could not find phandle");

	rc = of_parse_phandle_with_args(np, "phandle-list-bad-phandle",
					"#phandle-cells", 0, &args);

	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-a: could not find phandle");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-a: could not find phandle");

	rc = of_count_phandle_with_args(np, "phandle-list-bad-phandle",
					"#phandle-cells");

	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-a: could not find phandle");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);

	/* Check for incorrectly formed argument list */
	memset(&args, 0, sizeof(args));

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-a: #phandle-cells = 3 found -1");

	rc = of_parse_phandle_with_args(np, "phandle-list-bad-args",
					"#phandle-cells", 1, &args);

	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-a: #phandle-cells = 3 found -1");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-a: #phandle-cells = 3 found -1");

	rc = of_count_phandle_with_args(np, "phandle-list-bad-args",
					"#phandle-cells");

	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-a: #phandle-cells = 3 found -1");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);
}

static void __init of_unittest_parse_phandle_with_args_map(void)
{
	struct device_node *np, *p0, *p1, *p2, *p3;
	struct of_phandle_args args;
	int i, rc;

	np = of_find_node_by_path("/testcase-data/phandle-tests/consumer-b");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	p0 = of_find_node_by_path("/testcase-data/phandle-tests/provider0");
	if (!p0) {
		pr_err("missing testcase data\n");
		return;
	}

	p1 = of_find_node_by_path("/testcase-data/phandle-tests/provider1");
	if (!p1) {
		pr_err("missing testcase data\n");
		return;
	}

	p2 = of_find_node_by_path("/testcase-data/phandle-tests/provider2");
	if (!p2) {
		pr_err("missing testcase data\n");
		return;
	}

	p3 = of_find_node_by_path("/testcase-data/phandle-tests/provider3");
	if (!p3) {
		pr_err("missing testcase data\n");
		return;
	}

	rc = of_count_phandle_with_args(np, "phandle-list", "#phandle-cells");
	unittest(rc == 7, "of_count_phandle_with_args() returned %i, expected 7\n", rc);

	for (i = 0; i < 8; i++) {
		bool passed = true;

		memset(&args, 0, sizeof(args));
		rc = of_parse_phandle_with_args_map(np, "phandle-list",
						    "phandle", i, &args);

		/* Test the values from tests-phandle.dtsi */
		switch (i) {
		case 0:
			passed &= !rc;
			passed &= (args.np == p1);
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == 1);
			break;
		case 1:
			passed &= !rc;
			passed &= (args.np == p3);
			passed &= (args.args_count == 3);
			passed &= (args.args[0] == 2);
			passed &= (args.args[1] == 5);
			passed &= (args.args[2] == 3);
			break;
		case 2:
			passed &= (rc == -ENOENT);
			break;
		case 3:
			passed &= !rc;
			passed &= (args.np == p0);
			passed &= (args.args_count == 0);
			break;
		case 4:
			passed &= !rc;
			passed &= (args.np == p1);
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == 3);
			break;
		case 5:
			passed &= !rc;
			passed &= (args.np == p0);
			passed &= (args.args_count == 0);
			break;
		case 6:
			passed &= !rc;
			passed &= (args.np == p2);
			passed &= (args.args_count == 2);
			passed &= (args.args[0] == 15);
			passed &= (args.args[1] == 0x20);
			break;
		case 7:
			passed &= (rc == -ENOENT);
			break;
		default:
			passed = false;
		}

		unittest(passed, "index %i - data error on node %s rc=%i\n",
			 i, args.np->full_name, rc);
	}

	/* Check for missing list property */
	memset(&args, 0, sizeof(args));
	rc = of_parse_phandle_with_args_map(np, "phandle-list-missing",
					    "phandle", 0, &args);
	unittest(rc == -ENOENT, "expected:%i got:%i\n", -ENOENT, rc);

	/* Check for missing cells,map,mask property */
	memset(&args, 0, sizeof(args));

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-b: could not get #phandle-missing-cells for /testcase-data/phandle-tests/provider1");

	rc = of_parse_phandle_with_args_map(np, "phandle-list",
					    "phandle-missing", 0, &args);
	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-b: could not get #phandle-missing-cells for /testcase-data/phandle-tests/provider1");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);

	/* Check for bad phandle in list */
	memset(&args, 0, sizeof(args));

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-b: could not find phandle");

	rc = of_parse_phandle_with_args_map(np, "phandle-list-bad-phandle",
					    "phandle", 0, &args);
	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-b: could not find phandle");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);

	/* Check for incorrectly formed argument list */
	memset(&args, 0, sizeof(args));

	EXPECT_BEGIN(KERN_INFO,
		     "OF: /testcase-data/phandle-tests/consumer-b: #phandle-cells = 2 found -1");

	rc = of_parse_phandle_with_args_map(np, "phandle-list-bad-args",
					    "phandle", 1, &args);
	EXPECT_END(KERN_INFO,
		   "OF: /testcase-data/phandle-tests/consumer-b: #phandle-cells = 2 found -1");

	unittest(rc == -EINVAL, "expected:%i got:%i\n", -EINVAL, rc);
}

static void __init of_unittest_property_string(void)
{
	const char *strings[4];
	struct device_node *np;
	int rc;

	np = of_find_node_by_path("/testcase-data/phandle-tests/consumer-a");
	if (!np) {
		pr_err("No testcase data in device tree\n");
		return;
	}

	rc = of_property_match_string(np, "phandle-list-names", "first");
	unittest(rc == 0, "first expected:0 got:%i\n", rc);
	rc = of_property_match_string(np, "phandle-list-names", "second");
	unittest(rc == 1, "second expected:1 got:%i\n", rc);
	rc = of_property_match_string(np, "phandle-list-names", "third");
	unittest(rc == 2, "third expected:2 got:%i\n", rc);
	rc = of_property_match_string(np, "phandle-list-names", "fourth");
	unittest(rc == -ENODATA, "unmatched string; rc=%i\n", rc);
	rc = of_property_match_string(np, "missing-property", "blah");
	unittest(rc == -EINVAL, "missing property; rc=%i\n", rc);
	rc = of_property_match_string(np, "empty-property", "blah");
	unittest(rc == -ENODATA, "empty property; rc=%i\n", rc);
	rc = of_property_match_string(np, "unterminated-string", "blah");
	unittest(rc == -EILSEQ, "unterminated string; rc=%i\n", rc);

	/* of_property_count_strings() tests */
	rc = of_property_count_strings(np, "string-property");
	unittest(rc == 1, "Incorrect string count; rc=%i\n", rc);
	rc = of_property_count_strings(np, "phandle-list-names");
	unittest(rc == 3, "Incorrect string count; rc=%i\n", rc);
	rc = of_property_count_strings(np, "unterminated-string");
	unittest(rc == -EILSEQ, "unterminated string; rc=%i\n", rc);
	rc = of_property_count_strings(np, "unterminated-string-list");
	unittest(rc == -EILSEQ, "unterminated string array; rc=%i\n", rc);

	/* of_property_read_string_index() tests */
	rc = of_property_read_string_index(np, "string-property", 0, strings);
	unittest(rc == 0 && !strcmp(strings[0], "foobar"), "of_property_read_string_index() failure; rc=%i\n", rc);
	strings[0] = NULL;
	rc = of_property_read_string_index(np, "string-property", 1, strings);
	unittest(rc == -ENODATA && strings[0] == NULL, "of_property_read_string_index() failure; rc=%i\n", rc);
	rc = of_property_read_string_index(np, "phandle-list-names", 0, strings);
	unittest(rc == 0 && !strcmp(strings[0], "first"), "of_property_read_string_index() failure; rc=%i\n", rc);
	rc = of_property_read_string_index(np, "phandle-list-names", 1, strings);
	unittest(rc == 0 && !strcmp(strings[0], "second"), "of_property_read_string_index() failure; rc=%i\n", rc);
	rc = of_property_read_string_index(np, "phandle-list-names", 2, strings);
	unittest(rc == 0 && !strcmp(strings[0], "third"), "of_property_read_string_index() failure; rc=%i\n", rc);
	strings[0] = NULL;
	rc = of_property_read_string_index(np, "phandle-list-names", 3, strings);
	unittest(rc == -ENODATA && strings[0] == NULL, "of_property_read_string_index() failure; rc=%i\n", rc);
	strings[0] = NULL;
	rc = of_property_read_string_index(np, "unterminated-string", 0, strings);
	unittest(rc == -EILSEQ && strings[0] == NULL, "of_property_read_string_index() failure; rc=%i\n", rc);
	rc = of_property_read_string_index(np, "unterminated-string-list", 0, strings);
	unittest(rc == 0 && !strcmp(strings[0], "first"), "of_property_read_string_index() failure; rc=%i\n", rc);
	strings[0] = NULL;
	rc = of_property_read_string_index(np, "unterminated-string-list", 2, strings); /* should fail */
	unittest(rc == -EILSEQ && strings[0] == NULL, "of_property_read_string_index() failure; rc=%i\n", rc);
	strings[1] = NULL;

	/* of_property_read_string_array() tests */
	rc = of_property_read_string_array(np, "string-property", strings, 4);
	unittest(rc == 1, "Incorrect string count; rc=%i\n", rc);
	rc = of_property_read_string_array(np, "phandle-list-names", strings, 4);
	unittest(rc == 3, "Incorrect string count; rc=%i\n", rc);
	rc = of_property_read_string_array(np, "unterminated-string", strings, 4);
	unittest(rc == -EILSEQ, "unterminated string; rc=%i\n", rc);
	/* -- An incorrectly formed string should cause a failure */
	rc = of_property_read_string_array(np, "unterminated-string-list", strings, 4);
	unittest(rc == -EILSEQ, "unterminated string array; rc=%i\n", rc);
	/* -- parsing the correctly formed strings should still work: */
	strings[2] = NULL;
	rc = of_property_read_string_array(np, "unterminated-string-list", strings, 2);
	unittest(rc == 2 && strings[2] == NULL, "of_property_read_string_array() failure; rc=%i\n", rc);
	strings[1] = NULL;
	rc = of_property_read_string_array(np, "phandle-list-names", strings, 1);
	unittest(rc == 1 && strings[1] == NULL, "Overwrote end of string array; rc=%i, str='%s'\n", rc, strings[1]);
}

#define propcmp(p1, p2) (((p1)->length == (p2)->length) && \
			(p1)->value && (p2)->value && \
			!memcmp((p1)->value, (p2)->value, (p1)->length) && \
			!strcmp((p1)->name, (p2)->name))
static void __init of_unittest_property_copy(void)
{
#ifdef CONFIG_OF_DYNAMIC
	struct property p1 = { .name = "p1", .length = 0, .value = "" };
	struct property p2 = { .name = "p2", .length = 5, .value = "abcd" };
	struct property *new;

	new = __of_prop_dup(&p1, GFP_KERNEL);
	unittest(new && propcmp(&p1, new), "empty property didn't copy correctly\n");
	kfree(new->value);
	kfree(new->name);
	kfree(new);

	new = __of_prop_dup(&p2, GFP_KERNEL);
	unittest(new && propcmp(&p2, new), "non-empty property didn't copy correctly\n");
	kfree(new->value);
	kfree(new->name);
	kfree(new);
#endif
}

static void __init of_unittest_changeset(void)
{
#ifdef CONFIG_OF_DYNAMIC
	struct property *ppadd, padd = { .name = "prop-add", .length = 1, .value = "" };
	struct property *ppname_n1,  pname_n1  = { .name = "name", .length = 3, .value = "n1"  };
	struct property *ppname_n2,  pname_n2  = { .name = "name", .length = 3, .value = "n2"  };
	struct property *ppname_n21, pname_n21 = { .name = "name", .length = 3, .value = "n21" };
	struct property *ppupdate, pupdate = { .name = "prop-update", .length = 5, .value = "abcd" };
	struct property *ppremove;
	struct device_node *n1, *n2, *n21, *nchangeset, *nremove, *parent, *np;
	struct of_changeset chgset;

	n1 = __of_node_dup(NULL, "n1");
	unittest(n1, "testcase setup failure\n");

	n2 = __of_node_dup(NULL, "n2");
	unittest(n2, "testcase setup failure\n");

	n21 = __of_node_dup(NULL, "n21");
	unittest(n21, "testcase setup failure %p\n", n21);

	nchangeset = of_find_node_by_path("/testcase-data/changeset");
	nremove = of_get_child_by_name(nchangeset, "node-remove");
	unittest(nremove, "testcase setup failure\n");

	ppadd = __of_prop_dup(&padd, GFP_KERNEL);
	unittest(ppadd, "testcase setup failure\n");

	ppname_n1  = __of_prop_dup(&pname_n1, GFP_KERNEL);
	unittest(ppname_n1, "testcase setup failure\n");

	ppname_n2  = __of_prop_dup(&pname_n2, GFP_KERNEL);
	unittest(ppname_n2, "testcase setup failure\n");

	ppname_n21 = __of_prop_dup(&pname_n21, GFP_KERNEL);
	unittest(ppname_n21, "testcase setup failure\n");

	ppupdate = __of_prop_dup(&pupdate, GFP_KERNEL);
	unittest(ppupdate, "testcase setup failure\n");

	parent = nchangeset;
	n1->parent = parent;
	n2->parent = parent;
	n21->parent = n2;

	ppremove = of_find_property(parent, "prop-remove", NULL);
	unittest(ppremove, "failed to find removal prop");

	of_changeset_init(&chgset);

	unittest(!of_changeset_attach_node(&chgset, n1), "fail attach n1\n");
	unittest(!of_changeset_add_property(&chgset, n1, ppname_n1), "fail add prop name\n");

	unittest(!of_changeset_attach_node(&chgset, n2), "fail attach n2\n");
	unittest(!of_changeset_add_property(&chgset, n2, ppname_n2), "fail add prop name\n");

	unittest(!of_changeset_detach_node(&chgset, nremove), "fail remove node\n");
	unittest(!of_changeset_add_property(&chgset, n21, ppname_n21), "fail add prop name\n");

	unittest(!of_changeset_attach_node(&chgset, n21), "fail attach n21\n");

	unittest(!of_changeset_add_property(&chgset, parent, ppadd), "fail add prop prop-add\n");
	unittest(!of_changeset_update_property(&chgset, parent, ppupdate), "fail update prop\n");
	unittest(!of_changeset_remove_property(&chgset, parent, ppremove), "fail remove prop\n");

	unittest(!of_changeset_apply(&chgset), "apply failed\n");

	of_node_put(nchangeset);

	/* Make sure node names are constructed correctly */
	unittest((np = of_find_node_by_path("/testcase-data/changeset/n2/n21")),
		 "'%pOF' not added\n", n21);
	of_node_put(np);

	unittest(!of_changeset_revert(&chgset), "revert failed\n");

	of_changeset_destroy(&chgset);

	of_node_put(n1);
	of_node_put(n2);
	of_node_put(n21);
#endif
}

static void __init of_unittest_dma_get_max_cpu_address(void)
{
	struct device_node *np;
	phys_addr_t cpu_addr;

	if (!IS_ENABLED(CONFIG_OF_ADDRESS))
		return;

	np = of_find_node_by_path("/testcase-data/address-tests");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	cpu_addr = of_dma_get_max_cpu_address(np);
	unittest(cpu_addr == 0x4fffffff,
		 "of_dma_get_max_cpu_address: wrong CPU addr %pad (expecting %x)\n",
		 &cpu_addr, 0x4fffffff);
}

static void __init of_unittest_dma_ranges_one(const char *path,
		u64 expect_dma_addr, u64 expect_paddr)
{
#ifdef CONFIG_HAS_DMA
	struct device_node *np;
	const struct bus_dma_region *map = NULL;
	int rc;

	np = of_find_node_by_path(path);
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	rc = of_dma_get_range(np, &map);

	unittest(!rc, "of_dma_get_range failed on node %pOF rc=%i\n", np, rc);

	if (!rc) {
		phys_addr_t	paddr;
		dma_addr_t	dma_addr;
		struct device	dev_bogus;

		dev_bogus.dma_range_map = map;
		paddr = dma_to_phys(&dev_bogus, expect_dma_addr);
		dma_addr = phys_to_dma(&dev_bogus, expect_paddr);

		unittest(paddr == expect_paddr,
			 "of_dma_get_range: wrong phys addr %pap (expecting %llx) on node %pOF\n",
			 &paddr, expect_paddr, np);
		unittest(dma_addr == expect_dma_addr,
			 "of_dma_get_range: wrong DMA addr %pad (expecting %llx) on node %pOF\n",
			 &dma_addr, expect_dma_addr, np);

		kfree(map);
	}
	of_node_put(np);
#endif
}

static void __init of_unittest_parse_dma_ranges(void)
{
	of_unittest_dma_ranges_one("/testcase-data/address-tests/device@70000000",
		0x0, 0x20000000);
	of_unittest_dma_ranges_one("/testcase-data/address-tests/bus@80000000/device@1000",
		0x100000000, 0x20000000);
	of_unittest_dma_ranges_one("/testcase-data/address-tests/pci@90000000",
		0x80000000, 0x20000000);
}

static void __init of_unittest_pci_dma_ranges(void)
{
	struct device_node *np;
	struct of_pci_range range;
	struct of_pci_range_parser parser;
	int i = 0;

	if (!IS_ENABLED(CONFIG_PCI))
		return;

	np = of_find_node_by_path("/testcase-data/address-tests/pci@90000000");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	if (of_pci_dma_range_parser_init(&parser, np)) {
		pr_err("missing dma-ranges property\n");
		return;
	}

	/*
	 * Get the dma-ranges from the device tree
	 */
	for_each_of_pci_range(&parser, &range) {
		if (!i) {
			unittest(range.size == 0x10000000,
				 "for_each_of_pci_range wrong size on node %pOF size=%llx\n",
				 np, range.size);
			unittest(range.cpu_addr == 0x20000000,
				 "for_each_of_pci_range wrong CPU addr (%llx) on node %pOF",
				 range.cpu_addr, np);
			unittest(range.pci_addr == 0x80000000,
				 "for_each_of_pci_range wrong DMA addr (%llx) on node %pOF",
				 range.pci_addr, np);
		} else {
			unittest(range.size == 0x10000000,
				 "for_each_of_pci_range wrong size on node %pOF size=%llx\n",
				 np, range.size);
			unittest(range.cpu_addr == 0x40000000,
				 "for_each_of_pci_range wrong CPU addr (%llx) on node %pOF",
				 range.cpu_addr, np);
			unittest(range.pci_addr == 0xc0000000,
				 "for_each_of_pci_range wrong DMA addr (%llx) on node %pOF",
				 range.pci_addr, np);
		}
		i++;
	}

	of_node_put(np);
}

static void __init of_unittest_parse_interrupts(void)
{
	struct device_node *np;
	struct of_phandle_args args;
	int i, rc;

	if (of_irq_workarounds & OF_IMAP_OLDWORLD_MAC)
		return;

	np = of_find_node_by_path("/testcase-data/interrupts/interrupts0");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	for (i = 0; i < 4; i++) {
		bool passed = true;

		memset(&args, 0, sizeof(args));
		rc = of_irq_parse_one(np, i, &args);

		passed &= !rc;
		passed &= (args.args_count == 1);
		passed &= (args.args[0] == (i + 1));

		unittest(passed, "index %i - data error on node %pOF rc=%i\n",
			 i, args.np, rc);
	}
	of_node_put(np);

	np = of_find_node_by_path("/testcase-data/interrupts/interrupts1");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	for (i = 0; i < 4; i++) {
		bool passed = true;

		memset(&args, 0, sizeof(args));
		rc = of_irq_parse_one(np, i, &args);

		/* Test the values from tests-phandle.dtsi */
		switch (i) {
		case 0:
			passed &= !rc;
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == 9);
			break;
		case 1:
			passed &= !rc;
			passed &= (args.args_count == 3);
			passed &= (args.args[0] == 10);
			passed &= (args.args[1] == 11);
			passed &= (args.args[2] == 12);
			break;
		case 2:
			passed &= !rc;
			passed &= (args.args_count == 2);
			passed &= (args.args[0] == 13);
			passed &= (args.args[1] == 14);
			break;
		case 3:
			passed &= !rc;
			passed &= (args.args_count == 2);
			passed &= (args.args[0] == 15);
			passed &= (args.args[1] == 16);
			break;
		default:
			passed = false;
		}
		unittest(passed, "index %i - data error on node %pOF rc=%i\n",
			 i, args.np, rc);
	}
	of_node_put(np);
}

static void __init of_unittest_parse_interrupts_extended(void)
{
	struct device_node *np;
	struct of_phandle_args args;
	int i, rc;

	if (of_irq_workarounds & OF_IMAP_OLDWORLD_MAC)
		return;

	np = of_find_node_by_path("/testcase-data/interrupts/interrupts-extended0");
	if (!np) {
		pr_err("missing testcase data\n");
		return;
	}

	for (i = 0; i < 7; i++) {
		bool passed = true;

		memset(&args, 0, sizeof(args));
		rc = of_irq_parse_one(np, i, &args);

		/* Test the values from tests-phandle.dtsi */
		switch (i) {
		case 0:
			passed &= !rc;
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == 1);
			break;
		case 1:
			passed &= !rc;
			passed &= (args.args_count == 3);
			passed &= (args.args[0] == 2);
			passed &= (args.args[1] == 3);
			passed &= (args.args[2] == 4);
			break;
		case 2:
			passed &= !rc;
			passed &= (args.args_count == 2);
			passed &= (args.args[0] == 5);
			passed &= (args.args[1] == 6);
			break;
		case 3:
			passed &= !rc;
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == 9);
			break;
		case 4:
			passed &= !rc;
			passed &= (args.args_count == 3);
			passed &= (args.args[0] == 10);
			passed &= (args.args[1] == 11);
			passed &= (args.args[2] == 12);
			break;
		case 5:
			passed &= !rc;
			passed &= (args.args_count == 2);
			passed &= (args.args[0] == 13);
			passed &= (args.args[1] == 14);
			break;
		case 6:
			/*
			 * Tests child node that is missing property
			 * #address-cells.  See the comments in
			 * drivers/of/unittest-data/tests-interrupts.dtsi
			 * nodes intmap1 and interrupts-extended0
			 */
			passed &= !rc;
			passed &= (args.args_count == 1);
			passed &= (args.args[0] == 15);
			break;
		default:
			passed = false;
		}

		unittest(passed, "index %i - data error on node %pOF rc=%i\n",
			 i, args.np, rc);
	}
	of_node_put(np);
}

static const struct of_device_id match_node_table[] = {
	{ .data = "A", .name = "name0", }, /* Name alone is lowest priority */
	{ .data = "B", .type = "type1", }, /* followed by type alone */

	{ .data = "Ca", .name = "name2", .type = "type1", }, /* followed by both together */
	{ .data = "Cb", .name = "name2", }, /* Only match when type doesn't match */
	{ .data = "Cc", .name = "name2", .type = "type2", },

	{ .data = "E", .compatible = "compat3" },
	{ .data = "G", .compatible = "compat2", },
	{ .data = "H", .compatible = "compat2", .name = "name5", },
	{ .data = "I", .compatible = "compat2", .type = "type1", },
	{ .data = "J", .compatible = "compat2", .type = "type1", .name = "name8", },
	{ .data = "K", .compatible = "compat2", .name = "name9", },
	{}
};

static struct {
	const char *path;
	const char *data;
} match_node_tests[] = {
	{ .path = "/testcase-data/match-node/name0", .data = "A", },
	{ .path = "/testcase-data/match-node/name1", .data = "B", },
	{ .path = "/testcase-data/match-node/a/name2", .data = "Ca", },
	{ .path = "/testcase-data/match-node/b/name2", .data = "Cb", },
	{ .path = "/testcase-data/match-node/c/name2", .data = "Cc", },
	{ .path = "/testcase-data/match-node/name3", .data = "E", },
	{ .path = "/testcase-data/match-node/name4", .data = "G", },
	{ .path = "/testcase-data/match-node/name5", .data = "H", },
	{ .path = "/testcase-data/match-node/name6", .data = "G", },
	{ .path = "/testcase-data/match-node/name7", .data = "I", },
	{ .path = "/testcase-data/match-node/name8", .data = "J", },
	{ .path = "/testcase-data/match-node/name9", .data = "K", },
};

static void __init of_unittest_match_node(void)
{
	struct device_node *np;
	const struct of_device_id *match;
	int i;

	for (i = 0; i < ARRAY_SIZE(match_node_tests); i++) {
		np = of_find_node_by_path(match_node_tests[i].path);
		if (!np) {
			unittest(0, "missing testcase node %s\n",
				match_node_tests[i].path);
			continue;
		}

		match = of_match_node(match_node_table, np);
		if (!match) {
			unittest(0, "%s didn't match anything\n",
				match_node_tests[i].path);
			continue;
		}

		if (strcmp(match->data, match_node_tests[i].data) != 0) {
			unittest(0, "%s got wrong match. expected %s, got %s\n",
				match_node_tests[i].path, match_node_tests[i].data,
				(const char *)match->data);
			continue;
		}
		unittest(1, "passed");
	}
}

static struct resource test_bus_res = DEFINE_RES_MEM(0xfffffff8, 2);
static const struct platform_device_info test_bus_info = {
	.name = "unittest-bus",
};
static void __init of_unittest_platform_populate(void)
{
	int irq, rc;
	struct device_node *np, *child, *grandchild;
	struct platform_device *pdev, *test_bus;
	const struct of_device_id match[] = {
		{ .compatible = "test-device", },
		{}
	};

	np = of_find_node_by_path("/testcase-data");
	of_platform_default_populate(np, NULL, NULL);

	/* Test that a missing irq domain returns -EPROBE_DEFER */
	np = of_find_node_by_path("/testcase-data/testcase-device1");
	pdev = of_find_device_by_node(np);
	unittest(pdev, "device 1 creation failed\n");

	if (!(of_irq_workarounds & OF_IMAP_OLDWORLD_MAC)) {
		irq = platform_get_irq(pdev, 0);
		unittest(irq == -EPROBE_DEFER,
			 "device deferred probe failed - %d\n", irq);

		/* Test that a parsing failure does not return -EPROBE_DEFER */
		np = of_find_node_by_path("/testcase-data/testcase-device2");
		pdev = of_find_device_by_node(np);
		unittest(pdev, "device 2 creation failed\n");

		EXPECT_BEGIN(KERN_INFO,
			     "platform testcase-data:testcase-device2: IRQ index 0 not found");

		irq = platform_get_irq(pdev, 0);

		EXPECT_END(KERN_INFO,
			   "platform testcase-data:testcase-device2: IRQ index 0 not found");

		unittest(irq < 0 && irq != -EPROBE_DEFER,
			 "device parsing error failed - %d\n", irq);
	}

	np = of_find_node_by_path("/testcase-data/platform-tests");
	unittest(np, "No testcase data in device tree\n");
	if (!np)
		return;

	test_bus = platform_device_register_full(&test_bus_info);
	rc = PTR_ERR_OR_ZERO(test_bus);
	unittest(!rc, "testbus registration failed; rc=%i\n", rc);
	if (rc) {
		of_node_put(np);
		return;
	}
	test_bus->dev.of_node = np;

	/*
	 * Add a dummy resource to the test bus node after it is
	 * registered to catch problems with un-inserted resources. The
	 * DT code doesn't insert the resources, and it has caused the
	 * kernel to oops in the past. This makes sure the same bug
	 * doesn't crop up again.
	 */
	platform_device_add_resources(test_bus, &test_bus_res, 1);

	of_platform_populate(np, match, NULL, &test_bus->dev);
	for_each_child_of_node(np, child) {
		for_each_child_of_node(child, grandchild) {
			pdev = of_find_device_by_node(grandchild);
			unittest(pdev,
				 "Could not create device for node '%pOFn'\n",
				 grandchild);
			platform_device_put(pdev);
		}
	}

	of_platform_depopulate(&test_bus->dev);
	for_each_child_of_node(np, child) {
		for_each_child_of_node(child, grandchild)
			unittest(!of_find_device_by_node(grandchild),
				 "device didn't get destroyed '%pOFn'\n",
				 grandchild);
	}

	platform_device_unregister(test_bus);
	of_node_put(np);
}

/**
 *	update_node_properties - adds the properties
 *	of np into dup node (present in live tree) and
 *	updates parent of children of np to dup.
 *
 *	@np:	node whose properties are being added to the live tree
 *	@dup:	node present in live tree to be updated
 */
static void update_node_properties(struct device_node *np,
					struct device_node *dup)
{
	struct property *prop;
	struct property *save_next;
	struct device_node *child;
	int ret;

	for_each_child_of_node(np, child)
		child->parent = dup;

	/*
	 * "unittest internal error: unable to add testdata property"
	 *
	 *    If this message reports a property in node '/__symbols__' then
	 *    the respective unittest overlay contains a label that has the
	 *    same name as a label in the live devicetree.  The label will
	 *    be in the live devicetree only if the devicetree source was
	 *    compiled with the '-@' option.  If you encounter this error,
	 *    please consider renaming __all__ of the labels in the unittest
	 *    overlay dts files with an odd prefix that is unlikely to be
	 *    used in a real devicetree.
	 */

	/*
	 * open code for_each_property_of_node() because of_add_property()
	 * sets prop->next to NULL
	 */
	for (prop = np->properties; prop != NULL; prop = save_next) {
		save_next = prop->next;
		ret = of_add_property(dup, prop);
		if (ret) {
			if (ret == -EEXIST && !strcmp(prop->name, "name"))
				continue;
			pr_err("unittest internal error: unable to add testdata property %pOF/%s",
			       np, prop->name);
		}
	}
}

/**
 *	attach_node_and_children - attaches nodes
 *	and its children to live tree.
 *	CAUTION: misleading function name - if node @np already exists in
 *	the live tree then children of @np are *not* attached to the live
 *	tree.  This works for the current test devicetree nodes because such
 *	nodes do not have child nodes.
 *
 *	@np:	Node to attach to live tree
 */
static void attach_node_and_children(struct device_node *np)
{
	struct device_node *next, *dup, *child;
	unsigned long flags;
	const char *full_name;

	full_name = kasprintf(GFP_KERNEL, "%pOF", np);

	if (!strcmp(full_name, "/__local_fixups__") ||
	    !strcmp(full_name, "/__fixups__")) {
		kfree(full_name);
		return;
	}

	dup = of_find_node_by_path(full_name);
	kfree(full_name);
	if (dup) {
		update_node_properties(np, dup);
		return;
	}

	child = np->child;
	np->child = NULL;

	mutex_lock(&of_mutex);
	raw_spin_lock_irqsave(&devtree_lock, flags);
	np->sibling = np->parent->child;
	np->parent->child = np;
	of_node_clear_flag(np, OF_DETACHED);
	raw_spin_unlock_irqrestore(&devtree_lock, flags);

	__of_attach_node_sysfs(np);
	mutex_unlock(&of_mutex);

	while (child) {
		next = child->sibling;
		attach_node_and_children(child);
		child = next;
	}
}

/**
 *	unittest_data_add - Reads, copies data from
 *	linked tree and attaches it to the live tree
 */
static int __init unittest_data_add(void)
{
	void *unittest_data;
	void *unittest_data_align;
	struct device_node *unittest_data_node = NULL, *np;
	/*
	 * __dtb_testcases_begin[] and __dtb_testcases_end[] are magically
	 * created by cmd_dt_S_dtb in scripts/Makefile.lib
	 */
	extern uint8_t __dtb_testcases_begin[];
	extern uint8_t __dtb_testcases_end[];
	const int size = __dtb_testcases_end - __dtb_testcases_begin;
	int rc;
	void *ret;

	if (!size) {
		pr_warn("%s: testcases is empty\n", __func__);
		return -ENODATA;
	}

	/* creating copy */
	unittest_data = kmalloc(size + FDT_ALIGN_SIZE, GFP_KERNEL);
	if (!unittest_data)
		return -ENOMEM;

	unittest_data_align = PTR_ALIGN(unittest_data, FDT_ALIGN_SIZE);
	memcpy(unittest_data_align, __dtb_testcases_begin, size);

	ret = of_fdt_unflatten_tree(unittest_data_align, NULL, &unittest_data_node);
	if (!ret) {
		pr_warn("%s: unflatten testcases tree failed\n", __func__);
		kfree(unittest_data);
		return -ENODATA;
	}
	if (!unittest_data_node) {
		pr_warn("%s: testcases tree is empty\n", __func__);
		kfree(unittest_data);
		return -ENODATA;
	}

	/*
	 * This lock normally encloses of_resolve_phandles()
	 */
	of_overlay_mutex_lock();

	rc = of_resolve_phandles(unittest_data_node);
	if (rc) {
		pr_err("%s: Failed to resolve phandles (rc=%i)\n", __func__, rc);
		of_overlay_mutex_unlock();
		return -EINVAL;
	}

	if (!of_root) {
		of_root = unittest_data_node;
		for_each_of_allnodes(np)
			__of_attach_node_sysfs(np);
		of_aliases = of_find_node_by_path("/aliases");
		of_chosen = of_find_node_by_path("/chosen");
		of_overlay_mutex_unlock();
		return 0;
	}

	EXPECT_BEGIN(KERN_INFO,
		     "Duplicate name in testcase-data, renamed to \"duplicate-name#1\"");

	/* attach the sub-tree to live tree */
	np = unittest_data_node->child;
	while (np) {
		struct device_node *next = np->sibling;

		np->parent = of_root;
		attach_node_and_children(np);
		np = next;
	}

	EXPECT_END(KERN_INFO,
		   "Duplicate name in testcase-data, renamed to \"duplicate-name#1\"");

	of_overlay_mutex_unlock();

	return 0;
}

#ifdef CONFIG_OF_OVERLAY
static int __init overlay_data_apply(const char *overlay_name, int *overlay_id);

static int unittest_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;

	if (np == NULL) {
		dev_err(dev, "No OF data for device\n");
		return -EINVAL;

	}

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);

	of_platform_populate(np, NULL, NULL, &pdev->dev);

	return 0;
}

static int unittest_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);
	return 0;
}

static const struct of_device_id unittest_match[] = {
	{ .compatible = "unittest", },
	{},
};

static struct platform_driver unittest_driver = {
	.probe			= unittest_probe,
	.remove			= unittest_remove,
	.driver = {
		.name		= "unittest",
		.of_match_table	= of_match_ptr(unittest_match),
	},
};

/* get the platform device instantiated at the path */
static struct platform_device *of_path_to_platform_device(const char *path)
{
	struct device_node *np;
	struct platform_device *pdev;

	np = of_find_node_by_path(path);
	if (np == NULL)
		return NULL;

	pdev = of_find_device_by_node(np);
	of_node_put(np);

	return pdev;
}

/* find out if a platform device exists at that path */
static int of_path_platform_device_exists(const char *path)
{
	struct platform_device *pdev;

	pdev = of_path_to_platform_device(path);
	platform_device_put(pdev);
	return pdev != NULL;
}

#ifdef CONFIG_OF_GPIO

struct unittest_gpio_dev {
	struct gpio_chip chip;
};

static int unittest_gpio_chip_request_count;
static int unittest_gpio_probe_count;
static int unittest_gpio_probe_pass_count;

static int unittest_gpio_chip_request(struct gpio_chip *chip, unsigned int offset)
{
	unittest_gpio_chip_request_count++;

	pr_debug("%s(): %s %d %d\n", __func__, chip->label, offset,
		 unittest_gpio_chip_request_count);
	return 0;
}

static int unittest_gpio_probe(struct platform_device *pdev)
{
	struct unittest_gpio_dev *devptr;
	int ret;

	unittest_gpio_probe_count++;

	devptr = kzalloc(sizeof(*devptr), GFP_KERNEL);
	if (!devptr)
		return -ENOMEM;

	platform_set_drvdata(pdev, devptr);

	devptr->chip.of_node = pdev->dev.of_node;
	devptr->chip.label = "of-unittest-gpio";
	devptr->chip.base = -1; /* dynamic allocation */
	devptr->chip.ngpio = 5;
	devptr->chip.request = unittest_gpio_chip_request;

	ret = gpiochip_add_data(&devptr->chip, NULL);

	unittest(!ret,
		 "gpiochip_add_data() for node @%pOF failed, ret = %d\n", devptr->chip.of_node, ret);

	if (!ret)
		unittest_gpio_probe_pass_count++;
	return ret;
}

static int unittest_gpio_remove(struct platform_device *pdev)
{
	struct unittest_gpio_dev *gdev = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;
	struct device_node *np = pdev->dev.of_node;

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);

	if (!gdev)
		return -EINVAL;

	if (gdev->chip.base != -1)
		gpiochip_remove(&gdev->chip);

	platform_set_drvdata(pdev, NULL);
	kfree(gdev);

	return 0;
}

static const struct of_device_id unittest_gpio_id[] = {
	{ .compatible = "unittest-gpio", },
	{}
};

static struct platform_driver unittest_gpio_driver = {
	.probe	= unittest_gpio_probe,
	.remove	= unittest_gpio_remove,
	.driver	= {
		.name		= "unittest-gpio",
		.of_match_table	= of_match_ptr(unittest_gpio_id),
	},
};

static void __init of_unittest_overlay_gpio(void)
{
	int chip_request_count;
	int probe_pass_count;
	int ret;

	/*
	 * tests: apply overlays before registering driver
	 * Similar to installing a driver as a module, the
	 * driver is registered after applying the overlays.
	 *
	 * The overlays are applied by overlay_data_apply()
	 * instead of of_unittest_apply_overlay() so that they
	 * will not be tracked.  Thus they will not be removed
	 * by of_unittest_destroy_tracked_overlays().
	 *
	 * - apply overlay_gpio_01
	 * - apply overlay_gpio_02a
	 * - apply overlay_gpio_02b
	 * - register driver
	 *
	 * register driver will result in
	 *   - probe and processing gpio hog for overlay_gpio_01
	 *   - probe for overlay_gpio_02a
	 *   - processing gpio for overlay_gpio_02b
	 */

	probe_pass_count = unittest_gpio_probe_pass_count;
	chip_request_count = unittest_gpio_chip_request_count;

	/*
	 * overlay_gpio_01 contains gpio node and child gpio hog node
	 * overlay_gpio_02a contains gpio node
	 * overlay_gpio_02b contains child gpio hog node
	 */

	unittest(overlay_data_apply("overlay_gpio_01", NULL),
		 "Adding overlay 'overlay_gpio_01' failed\n");

	unittest(overlay_data_apply("overlay_gpio_02a", NULL),
		 "Adding overlay 'overlay_gpio_02a' failed\n");

	unittest(overlay_data_apply("overlay_gpio_02b", NULL),
		 "Adding overlay 'overlay_gpio_02b' failed\n");

	/*
	 * messages are the result of the probes, after the
	 * driver is registered
	 */

	EXPECT_BEGIN(KERN_INFO,
		     "gpio-<<int>> (line-B-input): hogged as input\n");

	EXPECT_BEGIN(KERN_INFO,
		     "gpio-<<int>> (line-A-input): hogged as input\n");

	ret = platform_driver_register(&unittest_gpio_driver);
	if (unittest(ret == 0, "could not register unittest gpio driver\n"))
		return;

	EXPECT_END(KERN_INFO,
		   "gpio-<<int>> (line-A-input): hogged as input\n");
	EXPECT_END(KERN_INFO,
		   "gpio-<<int>> (line-B-input): hogged as input\n");

	unittest(probe_pass_count + 2 == unittest_gpio_probe_pass_count,
		 "unittest_gpio_probe() failed or not called\n");

	unittest(chip_request_count + 2 == unittest_gpio_chip_request_count,
		 "unittest_gpio_chip_request() called %d times (expected 1 time)\n",
		 unittest_gpio_chip_request_count - chip_request_count);

	/*
	 * tests: apply overlays after registering driver
	 *
	 * Similar to a driver built-in to the kernel, the
	 * driver is registered before applying the overlays.
	 *
	 * overlay_gpio_03 contains gpio node and child gpio hog node
	 *
	 * - apply overlay_gpio_03
	 *
	 * apply overlay will result in
	 *   - probe and processing gpio hog.
	 */

	probe_pass_count = unittest_gpio_probe_pass_count;
	chip_request_count = unittest_gpio_chip_request_count;

	EXPECT_BEGIN(KERN_INFO,
		     "gpio-<<int>> (line-D-input): hogged as input\n");

	/* overlay_gpio_03 contains gpio node and child gpio hog node */

	unittest(overlay_data_apply("overlay_gpio_03", NULL),
		 "Adding overlay 'overlay_gpio_03' failed\n");

	EXPECT_END(KERN_INFO,
		   "gpio-<<int>> (line-D-input): hogged as input\n");

	unittest(probe_pass_count + 1 == unittest_gpio_probe_pass_count,
		 "unittest_gpio_probe() failed or not called\n");

	unittest(chip_request_count + 1 == unittest_gpio_chip_request_count,
		 "unittest_gpio_chip_request() called %d times (expected 1 time)\n",
		 unittest_gpio_chip_request_count - chip_request_count);

	/*
	 * overlay_gpio_04a contains gpio node
	 *
	 * - apply overlay_gpio_04a
	 *
	 * apply the overlay will result in
	 *   - probe for overlay_gpio_04a
	 */

	probe_pass_count = unittest_gpio_probe_pass_count;
	chip_request_count = unittest_gpio_chip_request_count;

	/* overlay_gpio_04a contains gpio node */

	unittest(overlay_data_apply("overlay_gpio_04a", NULL),
		 "Adding overlay 'overlay_gpio_04a' failed\n");

	unittest(probe_pass_count + 1 == unittest_gpio_probe_pass_count,
		 "unittest_gpio_probe() failed or not called\n");

	/*
	 * overlay_gpio_04b contains child gpio hog node
	 *
	 * - apply overlay_gpio_04b
	 *
	 * apply the overlay will result in
	 *   - processing gpio for overlay_gpio_04b
	 */

	EXPECT_BEGIN(KERN_INFO,
		     "gpio-<<int>> (line-C-input): hogged as input\n");

	/* overlay_gpio_04b contains child gpio hog node */

	unittest(overlay_data_apply("overlay_gpio_04b", NULL),
		 "Adding overlay 'overlay_gpio_04b' failed\n");

	EXPECT_END(KERN_INFO,
		   "gpio-<<int>> (line-C-input): hogged as input\n");

	unittest(chip_request_count + 1 == unittest_gpio_chip_request_count,
		 "unittest_gpio_chip_request() called %d times (expected 1 time)\n",
		 unittest_gpio_chip_request_count - chip_request_count);
}

#else

static void __init of_unittest_overlay_gpio(void)
{
	/* skip tests */
}

#endif

#if IS_BUILTIN(CONFIG_I2C)

/* get the i2c client device instantiated at the path */
static struct i2c_client *of_path_to_i2c_client(const char *path)
{
	struct device_node *np;
	struct i2c_client *client;

	np = of_find_node_by_path(path);
	if (np == NULL)
		return NULL;

	client = of_find_i2c_device_by_node(np);
	of_node_put(np);

	return client;
}

/* find out if a i2c client device exists at that path */
static int of_path_i2c_client_exists(const char *path)
{
	struct i2c_client *client;

	client = of_path_to_i2c_client(path);
	if (client)
		put_device(&client->dev);
	return client != NULL;
}
#else
static int of_path_i2c_client_exists(const char *path)
{
	return 0;
}
#endif

enum overlay_type {
	PDEV_OVERLAY,
	I2C_OVERLAY
};

static int of_path_device_type_exists(const char *path,
		enum overlay_type ovtype)
{
	switch (ovtype) {
	case PDEV_OVERLAY:
		return of_path_platform_device_exists(path);
	case I2C_OVERLAY:
		return of_path_i2c_client_exists(path);
	}
	return 0;
}

static const char *unittest_path(int nr, enum overlay_type ovtype)
{
	const char *base;
	static char buf[256];

	switch (ovtype) {
	case PDEV_OVERLAY:
		base = "/testcase-data/overlay-node/test-bus";
		break;
	case I2C_OVERLAY:
		base = "/testcase-data/overlay-node/test-bus/i2c-test-bus";
		break;
	default:
		buf[0] = '\0';
		return buf;
	}
	snprintf(buf, sizeof(buf) - 1, "%s/test-unittest%d", base, nr);
	buf[sizeof(buf) - 1] = '\0';
	return buf;
}

static int of_unittest_device_exists(int unittest_nr, enum overlay_type ovtype)
{
	const char *path;

	path = unittest_path(unittest_nr, ovtype);

	switch (ovtype) {
	case PDEV_OVERLAY:
		return of_path_platform_device_exists(path);
	case I2C_OVERLAY:
		return of_path_i2c_client_exists(path);
	}
	return 0;
}

static const char *overlay_name_from_nr(int nr)
{
	static char buf[256];

	snprintf(buf, sizeof(buf) - 1,
		"overlay_%d", nr);
	buf[sizeof(buf) - 1] = '\0';

	return buf;
}

static const char *bus_path = "/testcase-data/overlay-node/test-bus";

/* FIXME: it is NOT guaranteed that overlay ids are assigned in sequence */

#define MAX_UNITTEST_OVERLAYS	256
static unsigned long overlay_id_bits[BITS_TO_LONGS(MAX_UNITTEST_OVERLAYS)];
static int overlay_first_id = -1;

static long of_unittest_overlay_tracked(int id)
{
	if (WARN_ON(id >= MAX_UNITTEST_OVERLAYS))
		return 0;
	return overlay_id_bits[BIT_WORD(id)] & BIT_MASK(id);
}

static void of_unittest_track_overlay(int id)
{
	if (overlay_first_id < 0)
		overlay_first_id = id;
	id -= overlay_first_id;

	if (WARN_ON(id >= MAX_UNITTEST_OVERLAYS))
		return;
	overlay_id_bits[BIT_WORD(id)] |= BIT_MASK(id);
}

static void of_unittest_untrack_overlay(int id)
{
	if (overlay_first_id < 0)
		return;
	id -= overlay_first_id;
	if (WARN_ON(id >= MAX_UNITTEST_OVERLAYS))
		return;
	overlay_id_bits[BIT_WORD(id)] &= ~BIT_MASK(id);
}

static void of_unittest_destroy_tracked_overlays(void)
{
	int id, ret, defers, ovcs_id;

	if (overlay_first_id < 0)
		return;

	/* try until no defers */
	do {
		defers = 0;
		/* remove in reverse order */
		for (id = MAX_UNITTEST_OVERLAYS - 1; id >= 0; id--) {
			if (!of_unittest_overlay_tracked(id))
				continue;

			ovcs_id = id + overlay_first_id;
			ret = of_overlay_remove(&ovcs_id);
			if (ret == -ENODEV) {
				pr_warn("%s: no overlay to destroy for #%d\n",
					__func__, id + overlay_first_id);
				continue;
			}
			if (ret != 0) {
				defers++;
				pr_warn("%s: overlay destroy failed for #%d\n",
					__func__, id + overlay_first_id);
				continue;
			}

			of_unittest_untrack_overlay(id);
		}
	} while (defers > 0);
}

static int __init of_unittest_apply_overlay(int overlay_nr, int *overlay_id)
{
	const char *overlay_name;

	overlay_name = overlay_name_from_nr(overlay_nr);

	if (!overlay_data_apply(overlay_name, overlay_id)) {
		unittest(0, "could not apply overlay \"%s\"\n",
				overlay_name);
		return -EFAULT;
	}
	of_unittest_track_overlay(*overlay_id);

	return 0;
}

/* apply an overlay while checking before and after states */
static int __init of_unittest_apply_overlay_check(int overlay_nr,
		int unittest_nr, int before, int after,
		enum overlay_type ovtype)
{
	int ret, ovcs_id;

	/* unittest device must not be in before state */
	if (of_unittest_device_exists(unittest_nr, ovtype) != before) {
		unittest(0, "%s with device @\"%s\" %s\n",
				overlay_name_from_nr(overlay_nr),
				unittest_path(unittest_nr, ovtype),
				!before ? "enabled" : "disabled");
		return -EINVAL;
	}

	ovcs_id = 0;
	ret = of_unittest_apply_overlay(overlay_nr, &ovcs_id);
	if (ret != 0) {
		/* of_unittest_apply_overlay already called unittest() */
		return ret;
	}

	/* unittest device must be to set to after state */
	if (of_unittest_device_exists(unittest_nr, ovtype) != after) {
		unittest(0, "%s failed to create @\"%s\" %s\n",
				overlay_name_from_nr(overlay_nr),
				unittest_path(unittest_nr, ovtype),
				!after ? "enabled" : "disabled");
		return -EINVAL;
	}

	return 0;
}

/* apply an overlay and then revert it while checking before, after states */
static int __init of_unittest_apply_revert_overlay_check(int overlay_nr,
		int unittest_nr, int before, int after,
		enum overlay_type ovtype)
{
	int ret, ovcs_id, save_id;

	/* unittest device must be in before state */
	if (of_unittest_device_exists(unittest_nr, ovtype) != before) {
		unittest(0, "%s with device @\"%s\" %s\n",
				overlay_name_from_nr(overlay_nr),
				unittest_path(unittest_nr, ovtype),
				!before ? "enabled" : "disabled");
		return -EINVAL;
	}

	/* apply the overlay */
	ovcs_id = 0;
	ret = of_unittest_apply_overlay(overlay_nr, &ovcs_id);
	if (ret != 0) {
		/* of_unittest_apply_overlay already called unittest() */
		return ret;
	}

	/* unittest device must be in after state */
	if (of_unittest_device_exists(unittest_nr, ovtype) != after) {
		unittest(0, "%s failed to create @\"%s\" %s\n",
				overlay_name_from_nr(overlay_nr),
				unittest_path(unittest_nr, ovtype),
				!after ? "enabled" : "disabled");
		return -EINVAL;
	}

	save_id = ovcs_id;
	ret = of_overlay_remove(&ovcs_id);
	if (ret != 0) {
		unittest(0, "%s failed to be destroyed @\"%s\"\n",
				overlay_name_from_nr(overlay_nr),
				unittest_path(unittest_nr, ovtype));
		return ret;
	}
	of_unittest_untrack_overlay(save_id);

	/* unittest device must be again in before state */
	if (of_unittest_device_exists(unittest_nr, PDEV_OVERLAY) != before) {
		unittest(0, "%s with device @\"%s\" %s\n",
				overlay_name_from_nr(overlay_nr),
				unittest_path(unittest_nr, ovtype),
				!before ? "enabled" : "disabled");
		return -EINVAL;
	}

	return 0;
}

/* test activation of device */
static void __init of_unittest_overlay_0(void)
{
	int ret;

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest0/status");

	/* device should enable */
	ret = of_unittest_apply_overlay_check(0, 0, 0, 1, PDEV_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest0/status");

	if (ret)
		return;

	unittest(1, "overlay test %d passed\n", 0);
}

/* test deactivation of device */
static void __init of_unittest_overlay_1(void)
{
	int ret;

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest1/status");

	/* device should disable */
	ret = of_unittest_apply_overlay_check(1, 1, 1, 0, PDEV_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest1/status");

	if (ret)
		return;

	unittest(1, "overlay test %d passed\n", 1);

}

/* test activation of device */
static void __init of_unittest_overlay_2(void)
{
	int ret;

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest2/status");

	/* device should enable */
	ret = of_unittest_apply_overlay_check(2, 2, 0, 1, PDEV_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest2/status");

	if (ret)
		return;
	unittest(1, "overlay test %d passed\n", 2);
}

/* test deactivation of device */
static void __init of_unittest_overlay_3(void)
{
	int ret;

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest3/status");

	/* device should disable */
	ret = of_unittest_apply_overlay_check(3, 3, 1, 0, PDEV_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest3/status");

	if (ret)
		return;

	unittest(1, "overlay test %d passed\n", 3);
}

/* test activation of a full device node */
static void __init of_unittest_overlay_4(void)
{
	/* device should disable */
	if (of_unittest_apply_overlay_check(4, 4, 0, 1, PDEV_OVERLAY))
		return;

	unittest(1, "overlay test %d passed\n", 4);
}

/* test overlay apply/revert sequence */
static void __init of_unittest_overlay_5(void)
{
	int ret;

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest5/status");

	/* device should disable */
	ret = of_unittest_apply_revert_overlay_check(5, 5, 0, 1, PDEV_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest5/status");

	if (ret)
		return;

	unittest(1, "overlay test %d passed\n", 5);
}

/* test overlay application in sequence */
static void __init of_unittest_overlay_6(void)
{
	int i, ov_id[2], ovcs_id;
	int overlay_nr = 6, unittest_nr = 6;
	int before = 0, after = 1;
	const char *overlay_name;

	int ret;

	/* unittest device must be in before state */
	for (i = 0; i < 2; i++) {
		if (of_unittest_device_exists(unittest_nr + i, PDEV_OVERLAY)
				!= before) {
			unittest(0, "%s with device @\"%s\" %s\n",
					overlay_name_from_nr(overlay_nr + i),
					unittest_path(unittest_nr + i,
						PDEV_OVERLAY),
					!before ? "enabled" : "disabled");
			return;
		}
	}

	/* apply the overlays */

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest6/status");

	overlay_name = overlay_name_from_nr(overlay_nr + 0);

	ret = overlay_data_apply(overlay_name, &ovcs_id);

	if (!ret) {
		unittest(0, "could not apply overlay \"%s\"\n", overlay_name);
			return;
	}
	ov_id[0] = ovcs_id;
	of_unittest_track_overlay(ov_id[0]);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest6/status");

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest7/status");

	overlay_name = overlay_name_from_nr(overlay_nr + 1);

	ret = overlay_data_apply(overlay_name, &ovcs_id);

	if (!ret) {
		unittest(0, "could not apply overlay \"%s\"\n", overlay_name);
			return;
	}
	ov_id[1] = ovcs_id;
	of_unittest_track_overlay(ov_id[1]);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest7/status");


	for (i = 0; i < 2; i++) {
		/* unittest device must be in after state */
		if (of_unittest_device_exists(unittest_nr + i, PDEV_OVERLAY)
				!= after) {
			unittest(0, "overlay @\"%s\" failed @\"%s\" %s\n",
					overlay_name_from_nr(overlay_nr + i),
					unittest_path(unittest_nr + i,
						PDEV_OVERLAY),
					!after ? "enabled" : "disabled");
			return;
		}
	}

	for (i = 1; i >= 0; i--) {
		ovcs_id = ov_id[i];
		if (of_overlay_remove(&ovcs_id)) {
			unittest(0, "%s failed destroy @\"%s\"\n",
					overlay_name_from_nr(overlay_nr + i),
					unittest_path(unittest_nr + i,
						PDEV_OVERLAY));
			return;
		}
		of_unittest_untrack_overlay(ov_id[i]);
	}

	for (i = 0; i < 2; i++) {
		/* unittest device must be again in before state */
		if (of_unittest_device_exists(unittest_nr + i, PDEV_OVERLAY)
				!= before) {
			unittest(0, "%s with device @\"%s\" %s\n",
					overlay_name_from_nr(overlay_nr + i),
					unittest_path(unittest_nr + i,
						PDEV_OVERLAY),
					!before ? "enabled" : "disabled");
			return;
		}
	}

	unittest(1, "overlay test %d passed\n", 6);

}

/* test overlay application in sequence */
static void __init of_unittest_overlay_8(void)
{
	int i, ov_id[2], ovcs_id;
	int overlay_nr = 8, unittest_nr = 8;
	const char *overlay_name;
	int ret;

	/* we don't care about device state in this test */

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest8/status");

	overlay_name = overlay_name_from_nr(overlay_nr + 0);

	ret = overlay_data_apply(overlay_name, &ovcs_id);
	if (!ret)
		unittest(0, "could not apply overlay \"%s\"\n", overlay_name);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest8/status");

	if (!ret)
		return;

	ov_id[0] = ovcs_id;
	of_unittest_track_overlay(ov_id[0]);

	overlay_name = overlay_name_from_nr(overlay_nr + 1);

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest8/property-foo");

	/* apply the overlays */
	ret = overlay_data_apply(overlay_name, &ovcs_id);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/test-unittest8/property-foo");

	if (!ret) {
		unittest(0, "could not apply overlay \"%s\"\n", overlay_name);
		return;
	}

	ov_id[1] = ovcs_id;
	of_unittest_track_overlay(ov_id[1]);

	/* now try to remove first overlay (it should fail) */
	ovcs_id = ov_id[0];

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: node_overlaps_later_cs: #6 overlaps with #7 @/testcase-data/overlay-node/test-bus/test-unittest8");

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: overlay #6 is not topmost");

	ret = of_overlay_remove(&ovcs_id);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: overlay #6 is not topmost");

	EXPECT_END(KERN_INFO,
		   "OF: overlay: node_overlaps_later_cs: #6 overlaps with #7 @/testcase-data/overlay-node/test-bus/test-unittest8");

	if (!ret) {
		unittest(0, "%s was destroyed @\"%s\"\n",
				overlay_name_from_nr(overlay_nr + 0),
				unittest_path(unittest_nr,
					PDEV_OVERLAY));
		return;
	}

	/* removing them in order should work */
	for (i = 1; i >= 0; i--) {
		ovcs_id = ov_id[i];
		if (of_overlay_remove(&ovcs_id)) {
			unittest(0, "%s not destroyed @\"%s\"\n",
					overlay_name_from_nr(overlay_nr + i),
					unittest_path(unittest_nr,
						PDEV_OVERLAY));
			return;
		}
		of_unittest_untrack_overlay(ov_id[i]);
	}

	unittest(1, "overlay test %d passed\n", 8);
}

/* test insertion of a bus with parent devices */
static void __init of_unittest_overlay_10(void)
{
	int ret;
	char *child_path;

	/* device should disable */
	ret = of_unittest_apply_overlay_check(10, 10, 0, 1, PDEV_OVERLAY);

	if (unittest(ret == 0,
			"overlay test %d failed; overlay application\n", 10))
		return;

	child_path = kasprintf(GFP_KERNEL, "%s/test-unittest101",
			unittest_path(10, PDEV_OVERLAY));
	if (unittest(child_path, "overlay test %d failed; kasprintf\n", 10))
		return;

	ret = of_path_device_type_exists(child_path, PDEV_OVERLAY);
	kfree(child_path);

	unittest(ret, "overlay test %d failed; no child device\n", 10);
}

/* test insertion of a bus with parent devices (and revert) */
static void __init of_unittest_overlay_11(void)
{
	int ret;

	/* device should disable */
	ret = of_unittest_apply_revert_overlay_check(11, 11, 0, 1,
			PDEV_OVERLAY);

	unittest(ret == 0, "overlay test %d failed; overlay apply\n", 11);
}

#if IS_BUILTIN(CONFIG_I2C) && IS_ENABLED(CONFIG_OF_OVERLAY)

struct unittest_i2c_bus_data {
	struct platform_device	*pdev;
	struct i2c_adapter	adap;
};

static int unittest_i2c_master_xfer(struct i2c_adapter *adap,
		struct i2c_msg *msgs, int num)
{
	struct unittest_i2c_bus_data *std = i2c_get_adapdata(adap);

	(void)std;

	return num;
}

static u32 unittest_i2c_functionality(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL;
}

static const struct i2c_algorithm unittest_i2c_algo = {
	.master_xfer	= unittest_i2c_master_xfer,
	.functionality	= unittest_i2c_functionality,
};

static int unittest_i2c_bus_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct unittest_i2c_bus_data *std;
	struct i2c_adapter *adap;
	int ret;

	if (np == NULL) {
		dev_err(dev, "No OF data for device\n");
		return -EINVAL;

	}

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);

	std = devm_kzalloc(dev, sizeof(*std), GFP_KERNEL);
	if (!std)
		return -ENOMEM;

	/* link them together */
	std->pdev = pdev;
	platform_set_drvdata(pdev, std);

	adap = &std->adap;
	i2c_set_adapdata(adap, std);
	adap->nr = -1;
	strlcpy(adap->name, pdev->name, sizeof(adap->name));
	adap->class = I2C_CLASS_DEPRECATED;
	adap->algo = &unittest_i2c_algo;
	adap->dev.parent = dev;
	adap->dev.of_node = dev->of_node;
	adap->timeout = 5 * HZ;
	adap->retries = 3;

	ret = i2c_add_numbered_adapter(adap);
	if (ret != 0) {
		dev_err(dev, "Failed to add I2C adapter\n");
		return ret;
	}

	return 0;
}

static int unittest_i2c_bus_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct unittest_i2c_bus_data *std = platform_get_drvdata(pdev);

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);
	i2c_del_adapter(&std->adap);

	return 0;
}

static const struct of_device_id unittest_i2c_bus_match[] = {
	{ .compatible = "unittest-i2c-bus", },
	{},
};

static struct platform_driver unittest_i2c_bus_driver = {
	.probe			= unittest_i2c_bus_probe,
	.remove			= unittest_i2c_bus_remove,
	.driver = {
		.name		= "unittest-i2c-bus",
		.of_match_table	= of_match_ptr(unittest_i2c_bus_match),
	},
};

static int unittest_i2c_dev_probe(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	struct device *dev = &client->dev;
	struct device_node *np = client->dev.of_node;

	if (!np) {
		dev_err(dev, "No OF node\n");
		return -EINVAL;
	}

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);

	return 0;
};

static int unittest_i2c_dev_remove(struct i2c_client *client)
{
	struct device *dev = &client->dev;
	struct device_node *np = client->dev.of_node;

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);
	return 0;
}

static const struct i2c_device_id unittest_i2c_dev_id[] = {
	{ .name = "unittest-i2c-dev" },
	{ }
};

static struct i2c_driver unittest_i2c_dev_driver = {
	.driver = {
		.name = "unittest-i2c-dev",
	},
	.probe = unittest_i2c_dev_probe,
	.remove = unittest_i2c_dev_remove,
	.id_table = unittest_i2c_dev_id,
};

#if IS_BUILTIN(CONFIG_I2C_MUX)

static int unittest_i2c_mux_select_chan(struct i2c_mux_core *muxc, u32 chan)
{
	return 0;
}

static int unittest_i2c_mux_probe(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	int i, nchans;
	struct device *dev = &client->dev;
	struct i2c_adapter *adap = client->adapter;
	struct device_node *np = client->dev.of_node, *child;
	struct i2c_mux_core *muxc;
	u32 reg, max_reg;

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);

	if (!np) {
		dev_err(dev, "No OF node\n");
		return -EINVAL;
	}

	max_reg = (u32)-1;
	for_each_child_of_node(np, child) {
		if (of_property_read_u32(child, "reg", &reg))
			continue;
		if (max_reg == (u32)-1 || reg > max_reg)
			max_reg = reg;
	}
	nchans = max_reg == (u32)-1 ? 0 : max_reg + 1;
	if (nchans == 0) {
		dev_err(dev, "No channels\n");
		return -EINVAL;
	}

	muxc = i2c_mux_alloc(adap, dev, nchans, 0, 0,
			     unittest_i2c_mux_select_chan, NULL);
	if (!muxc)
		return -ENOMEM;
	for (i = 0; i < nchans; i++) {
		if (i2c_mux_add_adapter(muxc, 0, i, 0)) {
			dev_err(dev, "Failed to register mux #%d\n", i);
			i2c_mux_del_adapters(muxc);
			return -ENODEV;
		}
	}

	i2c_set_clientdata(client, muxc);

	return 0;
};

static int unittest_i2c_mux_remove(struct i2c_client *client)
{
	struct device *dev = &client->dev;
	struct device_node *np = client->dev.of_node;
	struct i2c_mux_core *muxc = i2c_get_clientdata(client);

	dev_dbg(dev, "%s for node @%pOF\n", __func__, np);
	i2c_mux_del_adapters(muxc);
	return 0;
}

static const struct i2c_device_id unittest_i2c_mux_id[] = {
	{ .name = "unittest-i2c-mux" },
	{ }
};

static struct i2c_driver unittest_i2c_mux_driver = {
	.driver = {
		.name = "unittest-i2c-mux",
	},
	.probe = unittest_i2c_mux_probe,
	.remove = unittest_i2c_mux_remove,
	.id_table = unittest_i2c_mux_id,
};

#endif

static int of_unittest_overlay_i2c_init(void)
{
	int ret;

	ret = i2c_add_driver(&unittest_i2c_dev_driver);
	if (unittest(ret == 0,
			"could not register unittest i2c device driver\n"))
		return ret;

	ret = platform_driver_register(&unittest_i2c_bus_driver);

	if (unittest(ret == 0,
			"could not register unittest i2c bus driver\n"))
		return ret;

#if IS_BUILTIN(CONFIG_I2C_MUX)

	EXPECT_BEGIN(KERN_INFO,
		     "i2c i2c-1: Added multiplexed i2c bus 2");

	ret = i2c_add_driver(&unittest_i2c_mux_driver);

	EXPECT_END(KERN_INFO,
		   "i2c i2c-1: Added multiplexed i2c bus 2");

	if (unittest(ret == 0,
			"could not register unittest i2c mux driver\n"))
		return ret;
#endif

	return 0;
}

static void of_unittest_overlay_i2c_cleanup(void)
{
#if IS_BUILTIN(CONFIG_I2C_MUX)
	i2c_del_driver(&unittest_i2c_mux_driver);
#endif
	platform_driver_unregister(&unittest_i2c_bus_driver);
	i2c_del_driver(&unittest_i2c_dev_driver);
}

static void __init of_unittest_overlay_i2c_12(void)
{
	int ret;

	/* device should enable */
	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/i2c-test-bus/test-unittest12/status");

	ret = of_unittest_apply_overlay_check(12, 12, 0, 1, I2C_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/i2c-test-bus/test-unittest12/status");

	if (ret)
		return;

	unittest(1, "overlay test %d passed\n", 12);
}

/* test deactivation of device */
static void __init of_unittest_overlay_i2c_13(void)
{
	int ret;

	EXPECT_BEGIN(KERN_INFO,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/i2c-test-bus/test-unittest13/status");

	/* device should disable */
	ret = of_unittest_apply_overlay_check(13, 13, 1, 0, I2C_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data/overlay-node/test-bus/i2c-test-bus/test-unittest13/status");

	if (ret)
		return;

	unittest(1, "overlay test %d passed\n", 13);
}

/* just check for i2c mux existence */
static void of_unittest_overlay_i2c_14(void)
{
}

static void __init of_unittest_overlay_i2c_15(void)
{
	int ret;

	/* device should enable */
	EXPECT_BEGIN(KERN_INFO,
		     "i2c i2c-1: Added multiplexed i2c bus 3");

	ret = of_unittest_apply_overlay_check(15, 15, 0, 1, I2C_OVERLAY);

	EXPECT_END(KERN_INFO,
		   "i2c i2c-1: Added multiplexed i2c bus 3");

	if (ret)
		return;

	unittest(1, "overlay test %d passed\n", 15);
}

#else

static inline void of_unittest_overlay_i2c_14(void) { }
static inline void of_unittest_overlay_i2c_15(void) { }

#endif

static void __init of_unittest_overlay(void)
{
	struct device_node *bus_np = NULL;

	if (platform_driver_register(&unittest_driver)) {
		unittest(0, "could not register unittest driver\n");
		goto out;
	}

	bus_np = of_find_node_by_path(bus_path);
	if (bus_np == NULL) {
		unittest(0, "could not find bus_path \"%s\"\n", bus_path);
		goto out;
	}

	if (of_platform_default_populate(bus_np, NULL, NULL)) {
		unittest(0, "could not populate bus @ \"%s\"\n", bus_path);
		goto out;
	}

	if (!of_unittest_device_exists(100, PDEV_OVERLAY)) {
		unittest(0, "could not find unittest0 @ \"%s\"\n",
				unittest_path(100, PDEV_OVERLAY));
		goto out;
	}

	if (of_unittest_device_exists(101, PDEV_OVERLAY)) {
		unittest(0, "unittest1 @ \"%s\" should not exist\n",
				unittest_path(101, PDEV_OVERLAY));
		goto out;
	}

	unittest(1, "basic infrastructure of overlays passed");

	/* tests in sequence */
	of_unittest_overlay_0();
	of_unittest_overlay_1();
	of_unittest_overlay_2();
	of_unittest_overlay_3();
	of_unittest_overlay_4();
	of_unittest_overlay_5();
	of_unittest_overlay_6();
	of_unittest_overlay_8();

	of_unittest_overlay_10();
	of_unittest_overlay_11();

#if IS_BUILTIN(CONFIG_I2C)
	if (unittest(of_unittest_overlay_i2c_init() == 0, "i2c init failed\n"))
		goto out;

	of_unittest_overlay_i2c_12();
	of_unittest_overlay_i2c_13();
	of_unittest_overlay_i2c_14();
	of_unittest_overlay_i2c_15();

	of_unittest_overlay_i2c_cleanup();
#endif

	of_unittest_overlay_gpio();

	of_unittest_destroy_tracked_overlays();

out:
	of_node_put(bus_np);
}

#else
static inline void __init of_unittest_overlay(void) { }
#endif

#ifdef CONFIG_OF_OVERLAY

/*
 * __dtb_ot_begin[] and __dtb_ot_end[] are created by cmd_dt_S_dtb
 * in scripts/Makefile.lib
 */

#define OVERLAY_INFO_EXTERN(name) \
	extern uint8_t __dtb_##name##_begin[]; \
	extern uint8_t __dtb_##name##_end[]

#define OVERLAY_INFO(overlay_name, expected)             \
{	.dtb_begin       = __dtb_##overlay_name##_begin, \
	.dtb_end         = __dtb_##overlay_name##_end,   \
	.expected_result = expected,                     \
	.name            = #overlay_name,                \
}

struct overlay_info {
	uint8_t		*dtb_begin;
	uint8_t		*dtb_end;
	int		expected_result;
	int		overlay_id;
	char		*name;
};

OVERLAY_INFO_EXTERN(overlay_base);
OVERLAY_INFO_EXTERN(overlay);
OVERLAY_INFO_EXTERN(overlay_0);
OVERLAY_INFO_EXTERN(overlay_1);
OVERLAY_INFO_EXTERN(overlay_2);
OVERLAY_INFO_EXTERN(overlay_3);
OVERLAY_INFO_EXTERN(overlay_4);
OVERLAY_INFO_EXTERN(overlay_5);
OVERLAY_INFO_EXTERN(overlay_6);
OVERLAY_INFO_EXTERN(overlay_7);
OVERLAY_INFO_EXTERN(overlay_8);
OVERLAY_INFO_EXTERN(overlay_9);
OVERLAY_INFO_EXTERN(overlay_10);
OVERLAY_INFO_EXTERN(overlay_11);
OVERLAY_INFO_EXTERN(overlay_12);
OVERLAY_INFO_EXTERN(overlay_13);
OVERLAY_INFO_EXTERN(overlay_15);
OVERLAY_INFO_EXTERN(overlay_gpio_01);
OVERLAY_INFO_EXTERN(overlay_gpio_02a);
OVERLAY_INFO_EXTERN(overlay_gpio_02b);
OVERLAY_INFO_EXTERN(overlay_gpio_03);
OVERLAY_INFO_EXTERN(overlay_gpio_04a);
OVERLAY_INFO_EXTERN(overlay_gpio_04b);
OVERLAY_INFO_EXTERN(overlay_bad_add_dup_node);
OVERLAY_INFO_EXTERN(overlay_bad_add_dup_prop);
OVERLAY_INFO_EXTERN(overlay_bad_phandle);
OVERLAY_INFO_EXTERN(overlay_bad_symbol);

/* entries found by name */
static struct overlay_info overlays[] = {
	OVERLAY_INFO(overlay_base, -9999),
	OVERLAY_INFO(overlay, 0),
	OVERLAY_INFO(overlay_0, 0),
	OVERLAY_INFO(overlay_1, 0),
	OVERLAY_INFO(overlay_2, 0),
	OVERLAY_INFO(overlay_3, 0),
	OVERLAY_INFO(overlay_4, 0),
	OVERLAY_INFO(overlay_5, 0),
	OVERLAY_INFO(overlay_6, 0),
	OVERLAY_INFO(overlay_7, 0),
	OVERLAY_INFO(overlay_8, 0),
	OVERLAY_INFO(overlay_9, 0),
	OVERLAY_INFO(overlay_10, 0),
	OVERLAY_INFO(overlay_11, 0),
	OVERLAY_INFO(overlay_12, 0),
	OVERLAY_INFO(overlay_13, 0),
	OVERLAY_INFO(overlay_15, 0),
	OVERLAY_INFO(overlay_gpio_01, 0),
	OVERLAY_INFO(overlay_gpio_02a, 0),
	OVERLAY_INFO(overlay_gpio_02b, 0),
	OVERLAY_INFO(overlay_gpio_03, 0),
	OVERLAY_INFO(overlay_gpio_04a, 0),
	OVERLAY_INFO(overlay_gpio_04b, 0),
	OVERLAY_INFO(overlay_bad_add_dup_node, -EINVAL),
	OVERLAY_INFO(overlay_bad_add_dup_prop, -EINVAL),
	OVERLAY_INFO(overlay_bad_phandle, -EINVAL),
	OVERLAY_INFO(overlay_bad_symbol, -EINVAL),
	/* end marker */
	{.dtb_begin = NULL, .dtb_end = NULL, .expected_result = 0, .name = NULL}
};

static struct device_node *overlay_base_root;

static void * __init dt_alloc_memory(u64 size, u64 align)
{
	void *ptr = memblock_alloc(size, align);

	if (!ptr)
		panic("%s: Failed to allocate %llu bytes align=0x%llx\n",
		      __func__, size, align);

	return ptr;
}

/*
 * Create base device tree for the overlay unittest.
 *
 * This is called from very early boot code.
 *
 * Do as much as possible the same way as done in __unflatten_device_tree
 * and other early boot steps for the normal FDT so that the overlay base
 * unflattened tree will have the same characteristics as the real tree
 * (such as having memory allocated by the early allocator).  The goal
 * is to test "the real thing" as much as possible, and test "test setup
 * code" as little as possible.
 *
 * Have to stop before resolving phandles, because that uses kmalloc.
 */
void __init unittest_unflatten_overlay_base(void)
{
	struct overlay_info *info;
	u32 data_size;
	void *new_fdt;
	u32 size;
	int found = 0;
	const char *overlay_name = "overlay_base";

	for (info = overlays; info && info->name; info++) {
		if (!strcmp(overlay_name, info->name)) {
			found = 1;
			break;
		}
	}
	if (!found) {
		pr_err("no overlay data for %s\n", overlay_name);
		return;
	}

	info = &overlays[0];

	if (info->expected_result != -9999) {
		pr_err("No dtb 'overlay_base' to attach\n");
		return;
	}

	data_size = info->dtb_end - info->dtb_begin;
	if (!data_size) {
		pr_err("No dtb 'overlay_base' to attach\n");
		return;
	}

	size = fdt_totalsize(info->dtb_begin);
	if (size != data_size) {
		pr_err("dtb 'overlay_base' header totalsize != actual size");
		return;
	}

	new_fdt = dt_alloc_memory(size, roundup_pow_of_two(FDT_V17_SIZE));
	if (!new_fdt) {
		pr_err("alloc for dtb 'overlay_base' failed");
		return;
	}

	memcpy(new_fdt, info->dtb_begin, size);

	__unflatten_device_tree(new_fdt, NULL, &overlay_base_root,
				dt_alloc_memory, true);
}

/*
 * The purpose of of_unittest_overlay_data_add is to add an
 * overlay in the normal fashion.  This is a test of the whole
 * picture, instead of testing individual elements.
 *
 * A secondary purpose is to be able to verify that the contents of
 * /proc/device-tree/ contains the updated structure and values from
 * the overlay.  That must be verified separately in user space.
 *
 * Return 0 on unexpected error.
 */
static int __init overlay_data_apply(const char *overlay_name, int *overlay_id)
{
	struct overlay_info *info;
	int found = 0;
	int ret;
	u32 size;

	for (info = overlays; info && info->name; info++) {
		if (!strcmp(overlay_name, info->name)) {
			found = 1;
			break;
		}
	}
	if (!found) {
		pr_err("no overlay data for %s\n", overlay_name);
		return 0;
	}

	size = info->dtb_end - info->dtb_begin;
	if (!size)
		pr_err("no overlay data for %s\n", overlay_name);

	ret = of_overlay_fdt_apply(info->dtb_begin, size, &info->overlay_id);
	if (overlay_id)
		*overlay_id = info->overlay_id;
	if (ret < 0)
		goto out;

	pr_debug("%s applied\n", overlay_name);

out:
	if (ret != info->expected_result)
		pr_err("of_overlay_fdt_apply() expected %d, ret=%d, %s\n",
		       info->expected_result, ret, overlay_name);

	return (ret == info->expected_result);
}

/*
 * The purpose of of_unittest_overlay_high_level is to add an overlay
 * in the normal fashion.  This is a test of the whole picture,
 * instead of individual elements.
 *
 * The first part of the function is _not_ normal overlay usage; it is
 * finishing splicing the base overlay device tree into the live tree.
 */
static __init void of_unittest_overlay_high_level(void)
{
	struct device_node *last_sibling;
	struct device_node *np;
	struct device_node *of_symbols;
	struct device_node *overlay_base_symbols;
	struct device_node **pprev;
	struct property *prop;
	int ret;

	if (!overlay_base_root) {
		unittest(0, "overlay_base_root not initialized\n");
		return;
	}

	/*
	 * Could not fixup phandles in unittest_unflatten_overlay_base()
	 * because kmalloc() was not yet available.
	 */
	of_overlay_mutex_lock();
	of_resolve_phandles(overlay_base_root);
	of_overlay_mutex_unlock();


	/*
	 * do not allow overlay_base to duplicate any node already in
	 * tree, this greatly simplifies the code
	 */

	/*
	 * remove overlay_base_root node "__local_fixups", after
	 * being used by of_resolve_phandles()
	 */
	pprev = &overlay_base_root->child;
	for (np = overlay_base_root->child; np; np = np->sibling) {
		if (of_node_name_eq(np, "__local_fixups__")) {
			*pprev = np->sibling;
			break;
		}
		pprev = &np->sibling;
	}

	/* remove overlay_base_root node "__symbols__" if in live tree */
	of_symbols = of_get_child_by_name(of_root, "__symbols__");
	if (of_symbols) {
		/* will have to graft properties from node into live tree */
		pprev = &overlay_base_root->child;
		for (np = overlay_base_root->child; np; np = np->sibling) {
			if (of_node_name_eq(np, "__symbols__")) {
				overlay_base_symbols = np;
				*pprev = np->sibling;
				break;
			}
			pprev = &np->sibling;
		}
	}

	for_each_child_of_node(overlay_base_root, np) {
		struct device_node *base_child;
		for_each_child_of_node(of_root, base_child) {
			if (!strcmp(np->full_name, base_child->full_name)) {
				unittest(0, "illegal node name in overlay_base %pOFn",
					 np);
				of_node_put(np);
				of_node_put(base_child);
				return;
			}
		}
	}

	/*
	 * overlay 'overlay_base' is not allowed to have root
	 * properties, so only need to splice nodes into main device tree.
	 *
	 * root node of *overlay_base_root will not be freed, it is lost
	 * memory.
	 */

	for (np = overlay_base_root->child; np; np = np->sibling)
		np->parent = of_root;

	mutex_lock(&of_mutex);

	for (last_sibling = np = of_root->child; np; np = np->sibling)
		last_sibling = np;

	if (last_sibling)
		last_sibling->sibling = overlay_base_root->child;
	else
		of_root->child = overlay_base_root->child;

	for_each_of_allnodes_from(overlay_base_root, np)
		__of_attach_node_sysfs(np);

	if (of_symbols) {
		struct property *new_prop;
		for_each_property_of_node(overlay_base_symbols, prop) {

			new_prop = __of_prop_dup(prop, GFP_KERNEL);
			if (!new_prop) {
				unittest(0, "__of_prop_dup() of '%s' from overlay_base node __symbols__",
					 prop->name);
				goto err_unlock;
			}
			if (__of_add_property(of_symbols, new_prop)) {
				kfree(new_prop->name);
				kfree(new_prop->value);
				kfree(new_prop);
				/* "name" auto-generated by unflatten */
				if (!strcmp(prop->name, "name"))
					continue;
				unittest(0, "duplicate property '%s' in overlay_base node __symbols__",
					 prop->name);
				goto err_unlock;
			}
			if (__of_add_property_sysfs(of_symbols, new_prop)) {
				unittest(0, "unable to add property '%s' in overlay_base node __symbols__ to sysfs",
					 prop->name);
				goto err_unlock;
			}
		}
	}

	mutex_unlock(&of_mutex);


	/* now do the normal overlay usage test */

	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/substation@100/status");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/fairway-1/status");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/fairway-1/ride@100/track@30/incline-up");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/fairway-1/ride@100/track@40/incline-up");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/lights@40000/status");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/lights@40000/color");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/lights@40000/rate");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/hvac_2");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/ride_200");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/ride_200_left");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/ride_200_right");

	ret = overlay_data_apply("overlay", NULL);

	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/ride_200_right");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/ride_200_left");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/ride_200");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /__symbols__/hvac_2");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/lights@40000/rate");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/lights@40000/color");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/lights@40000/status");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/fairway-1/ride@100/track@40/incline-up");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/fairway-1/ride@100/track@30/incline-up");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/fairway-1/status");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: WARNING: memory leak will occur if overlay removed, property: /testcase-data-2/substation@100/status");

	unittest(ret, "Adding overlay 'overlay' failed\n");

	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add and/or delete node /testcase-data-2/substation@100/motor-1/controller");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add, update, and/or delete property /testcase-data-2/substation@100/motor-1/controller/name");

	unittest(overlay_data_apply("overlay_bad_add_dup_node", NULL),
		 "Adding overlay 'overlay_bad_add_dup_node' failed\n");

	EXPECT_END(KERN_ERR,
		   "OF: overlay: ERROR: multiple fragments add, update, and/or delete property /testcase-data-2/substation@100/motor-1/controller/name");
	EXPECT_END(KERN_ERR,
		   "OF: overlay: ERROR: multiple fragments add and/or delete node /testcase-data-2/substation@100/motor-1/controller");

	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add and/or delete node /testcase-data-2/substation@100/motor-1/electric");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add, update, and/or delete property /testcase-data-2/substation@100/motor-1/electric/rpm_avail");
	EXPECT_BEGIN(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add, update, and/or delete property /testcase-data-2/substation@100/motor-1/electric/name");

	unittest(overlay_data_apply("overlay_bad_add_dup_prop", NULL),
		 "Adding overlay 'overlay_bad_add_dup_prop' failed\n");

	EXPECT_END(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add, update, and/or delete property /testcase-data-2/substation@100/motor-1/electric/name");
	EXPECT_END(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add, update, and/or delete property /testcase-data-2/substation@100/motor-1/electric/rpm_avail");
	EXPECT_END(KERN_ERR,
		     "OF: overlay: ERROR: multiple fragments add and/or delete node /testcase-data-2/substation@100/motor-1/electric");

	unittest(overlay_data_apply("overlay_bad_phandle", NULL),
		 "Adding overlay 'overlay_bad_phandle' failed\n");

	unittest(overlay_data_apply("overlay_bad_symbol", NULL),
		 "Adding overlay 'overlay_bad_symbol' failed\n");

	return;

err_unlock:
	mutex_unlock(&of_mutex);
}

#else

static inline __init void of_unittest_overlay_high_level(void) {}

#endif

static int __init of_unittest(void)
{
	struct device_node *np;
	int res;

	pr_info("start of unittest - you will see error messages\n");

	/* adding data for unittest */

	if (IS_ENABLED(CONFIG_UML))
		unittest_unflatten_overlay_base();

	res = unittest_data_add();
	if (res)
		return res;
	if (!of_aliases)
		of_aliases = of_find_node_by_path("/aliases");

	np = of_find_node_by_path("/testcase-data/phandle-tests/consumer-a");
	if (!np) {
		pr_info("No testcase data in device tree; not running tests\n");
		return 0;
	}
	of_node_put(np);

	of_unittest_check_tree_linkage();
	of_unittest_check_phandles();
	of_unittest_find_node_by_name();
	of_unittest_dynamic();
	of_unittest_parse_phandle_with_args();
	of_unittest_parse_phandle_with_args_map();
	of_unittest_printf();
	of_unittest_property_string();
	of_unittest_property_copy();
	of_unittest_changeset();
	of_unittest_parse_interrupts();
	of_unittest_parse_interrupts_extended();
	of_unittest_dma_get_max_cpu_address();
	of_unittest_parse_dma_ranges();
	of_unittest_pci_dma_ranges();
	of_unittest_match_node();
	of_unittest_platform_populate();
	of_unittest_overlay();

	/* Double check linkage after removing testcase data */
	of_unittest_check_tree_linkage();

	of_unittest_overlay_high_level();

	pr_info("end of unittest - %i passed, %i failed\n",
		unittest_results.passed, unittest_results.failed);

	return 0;
}
late_initcall(of_unittest);
