//
//
// Copyright 2018 gRPC authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//

#include <memory>

#include <gtest/gtest.h>

#include "absl/types/span.h"

#include <grpc/support/alloc.h>
#include <grpc/support/log.h>

#include "src/core/tsi/alts/crypt/gsec.h"
#include "test/core/tsi/alts/crypt/gsec_test_util.h"
#include "test/core/util/test_config.h"

const size_t kTestMinTagLengthForCorruption = 8;
const size_t kTestNumCrypters = 3;
const size_t kTestMaxSlices = 5;
const size_t kTestMaxLength = 1024;
const size_t kTestNumEncryptions = 100;

// Struct for pre-generated test vector
typedef struct gsec_aead_test_vector {
  uint8_t* nonce;
  uint8_t* aad;
  uint8_t* key;
  uint8_t* plaintext;
  uint8_t* ciphertext_and_tag;
  size_t nonce_length;
  size_t aad_length;
  size_t key_length;
  size_t plaintext_length;
  size_t ciphertext_and_tag_length;
} gsec_aead_test_vector;

static void gsec_randomly_slice(uint8_t* input, size_t input_length,
                                struct iovec** output, size_t* output_length) {
  if (input_length == 0) {
    *output = nullptr;
    *output_length = 0;
    return;
  }
  *output_length = gsec_test_bias_random_uint32(kTestMaxSlices) + 1;
  *output =
      static_cast<struct iovec*>(malloc(*output_length * sizeof(**output)));
  size_t i;
  for (i = 0; i < *output_length - 1; i++) {
    size_t slice_length =
        gsec_test_bias_random_uint32(static_cast<uint32_t>(input_length));
    struct iovec slice = {input, slice_length};
    (*output)[i] = slice;
    input += slice_length;
    input_length -= slice_length;
  }
  struct iovec slice = {input, input_length};
  (*output)[*output_length - 1] = slice;
}

static void gsec_assert_ok(grpc_status_code status, const char* error_detail) {
  char empty_string[] = "";
  if (error_detail == nullptr) {
    error_detail = empty_string;
  }
  if (status != GRPC_STATUS_OK) {
    fprintf(stderr, "Status is not ok: %s\n", error_detail);
  }
  ASSERT_EQ(status, GRPC_STATUS_OK);
}

static void gsec_test_random_encrypt_decrypt(gsec_aead_crypter* crypter,
                                             size_t aad_length,
                                             size_t message_length) {
  ASSERT_NE(crypter, nullptr);
  size_t nonce_length, tag_length;
  uint8_t *nonce, *aad, *message;
  gsec_aead_crypter_nonce_length(crypter, &nonce_length,
                                 /*error_details=*/nullptr);
  gsec_aead_crypter_tag_length(crypter, &tag_length, /*error_details=*/nullptr);

  gsec_test_random_array(&nonce, nonce_length);
  gsec_test_random_array(&aad, aad_length);
  gsec_test_random_array(&message, message_length);

  // Test encryption
  size_t ciphertext_and_tag_length, ciphertext_bytes_written = 0;
  gsec_aead_crypter_max_ciphertext_and_tag_length(crypter, message_length,
                                                  &ciphertext_and_tag_length,
                                                  /*error_details=*/nullptr);

  uint8_t* ciphertext_and_tag =
      static_cast<uint8_t*>(gpr_malloc(ciphertext_and_tag_length));

  char* error_buffer = nullptr;
  gsec_assert_ok(
      gsec_aead_crypter_encrypt(crypter, nonce, nonce_length, aad, aad_length,
                                message, message_length, ciphertext_and_tag,
                                ciphertext_and_tag_length,
                                &ciphertext_bytes_written, &error_buffer),
      error_buffer);
  ASSERT_EQ(message_length + tag_length, ciphertext_and_tag_length);
  ASSERT_EQ(ciphertext_bytes_written, ciphertext_and_tag_length);

  // Test decryption
  size_t plaintext_length, plaintext_bytes_written = 0;
  gsec_aead_crypter_max_plaintext_length(crypter, ciphertext_bytes_written,
                                         &plaintext_length,
                                         /*error_details=*/nullptr);
  uint8_t* plaintext = static_cast<uint8_t*>(gpr_malloc(plaintext_length));
  grpc_status_code status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length, aad, aad_length, ciphertext_and_tag,
      ciphertext_bytes_written, plaintext, plaintext_length,
      &plaintext_bytes_written, /*error_details=*/nullptr);

  ASSERT_EQ(status, GRPC_STATUS_OK);
  ASSERT_EQ(message_length, plaintext_bytes_written);
  if (message_length != 0) {
    ASSERT_EQ(memcmp(message, plaintext, message_length), 0);
  }

  ///
  /// The returned plaintext will be zeroed if there was an authentication
  /// error.
  ///
  uint8_t* zero_message = static_cast<uint8_t*>(gpr_zalloc(plaintext_length));
  if (tag_length >= kTestMinTagLengthForCorruption) {
    char* error_message;
    // Corrupt nonce
    if (nonce_length > 0) {
      plaintext_bytes_written = 0;
      uint8_t* corrupt_nonce;
      gsec_test_copy_and_alter_random_byte(nonce, &corrupt_nonce, nonce_length);
      status = gsec_aead_crypter_decrypt(
          crypter, corrupt_nonce, nonce_length, aad, aad_length,
          ciphertext_and_tag, ciphertext_bytes_written, plaintext,
          plaintext_length, &plaintext_bytes_written, &error_message);

      ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
          status, GRPC_STATUS_FAILED_PRECONDITION, "Checking tag failed.",
          error_message));
      ASSERT_EQ(plaintext_bytes_written, 0);
      if (plaintext_length != 0) {
        ASSERT_EQ(memcmp(zero_message, plaintext, plaintext_length), 0);
      }
      gpr_free(corrupt_nonce);
      gpr_free(error_message);
    }

    // Corrupt ciphertext_and_tag
    plaintext_bytes_written = 0;
    uint8_t* corrupt_ciphertext_and_tag;
    gsec_test_copy_and_alter_random_byte(ciphertext_and_tag,
                                         &corrupt_ciphertext_and_tag,
                                         ciphertext_and_tag_length);
    status = gsec_aead_crypter_decrypt(
        crypter, nonce, nonce_length, aad, aad_length,
        corrupt_ciphertext_and_tag, ciphertext_bytes_written, plaintext,
        plaintext_length, &plaintext_bytes_written, &error_message);

    ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
        status, GRPC_STATUS_FAILED_PRECONDITION, error_message,
        "Checking tag failed"));
    ASSERT_EQ(plaintext_bytes_written, 0);
    if (plaintext_length != 0) {
      ASSERT_EQ(memcmp(zero_message, plaintext, plaintext_length), 0);
    }
    gpr_free(error_message);
    gpr_free(corrupt_ciphertext_and_tag);

    // Corrupt start of ciphertext_and_tag
    plaintext_bytes_written = 0;
    gsec_test_copy(ciphertext_and_tag, &corrupt_ciphertext_and_tag,
                   ciphertext_and_tag_length);
    (*corrupt_ciphertext_and_tag)++;
    status = gsec_aead_crypter_decrypt(
        crypter, nonce, nonce_length, aad, aad_length,
        corrupt_ciphertext_and_tag, ciphertext_bytes_written, plaintext,
        plaintext_length, &plaintext_bytes_written, &error_message);
    ASSERT_EQ(plaintext_bytes_written, 0);
    if (plaintext_length != 0) {
      ASSERT_EQ(memcmp(zero_message, plaintext, plaintext_length), 0);
    }
    ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
        status, GRPC_STATUS_FAILED_PRECONDITION, error_message,
        "Checking tag failed"));
    gpr_free(error_message);
    gpr_free(corrupt_ciphertext_and_tag);

    // Corrupt end of ciphertext_and_tag
    plaintext_bytes_written = 0;
    gsec_test_copy(ciphertext_and_tag, &corrupt_ciphertext_and_tag,
                   ciphertext_and_tag_length);
    (*(corrupt_ciphertext_and_tag + ciphertext_and_tag_length - 1))++;

    status = gsec_aead_crypter_decrypt(
        crypter, nonce, nonce_length, aad, aad_length,
        corrupt_ciphertext_and_tag, ciphertext_bytes_written, plaintext,
        plaintext_length, &plaintext_bytes_written, &error_message);

    ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
        status, GRPC_STATUS_FAILED_PRECONDITION, error_message,
        "Checking tag failed"));
    ASSERT_EQ(plaintext_bytes_written, 0);
    if (plaintext_length != 0) {
      ASSERT_EQ(memcmp(zero_message, plaintext, plaintext_length), 0);
    }
    gpr_free(error_message);
    gpr_free(corrupt_ciphertext_and_tag);
  }

  gpr_free(zero_message);
  gpr_free(nonce);
  gpr_free(aad);
  gpr_free(message);
  gpr_free(plaintext);
  gpr_free(ciphertext_and_tag);
}

static void gsec_test_encrypt_decrypt(gsec_aead_crypter* crypter) {
  ASSERT_NE(crypter, nullptr);
  size_t aad_length, message_length;
  aad_length = gsec_test_bias_random_uint32(kTestMaxLength);
  message_length = gsec_test_bias_random_uint32(kTestMaxLength);
  gsec_test_random_encrypt_decrypt(crypter, aad_length, message_length);
  gsec_test_random_encrypt_decrypt(crypter, 0, message_length);
  gsec_test_random_encrypt_decrypt(crypter, aad_length, 0);
}

static void gsec_test_multiple_random_encrypt_decrypt(
    gsec_aead_crypter* crypter, size_t* aad_lengths, size_t* message_lengths,
    size_t count) {
  ASSERT_NE(crypter, nullptr);
  size_t nonce_length, tag_length;
  uint8_t **nonces, **aads, **messages;
  nonces = static_cast<uint8_t**>(gpr_malloc(sizeof(uint8_t*) * count));
  aads = static_cast<uint8_t**>(gpr_malloc(sizeof(uint8_t*) * count));
  messages = static_cast<uint8_t**>(gpr_malloc(sizeof(uint8_t*) * count));

  gsec_aead_crypter_nonce_length(crypter, &nonce_length,
                                 /*error_details=*/nullptr);
  gsec_aead_crypter_tag_length(crypter, &tag_length, /*error_details=*/nullptr);

  size_t ind;
  for (ind = 0; ind < count; ind++) {
    size_t aad_length = (aad_lengths == nullptr) ? 0 : aad_lengths[ind];
    size_t message_length =
        (message_lengths == nullptr) ? 0 : message_lengths[ind];
    gsec_test_random_array(&(nonces[ind]), nonce_length);
    gsec_test_random_array(&(aads[ind]), aad_length);
    gsec_test_random_array(&(messages[ind]), message_length);
  }

  size_t* ciphertext_and_tag_lengths =
      static_cast<size_t*>(gpr_malloc(sizeof(size_t) * count));
  size_t* ciphertext_bytes_writtens =
      static_cast<size_t*>(gpr_malloc(sizeof(size_t) * count));
  size_t* plaintext_lengths =
      static_cast<size_t*>(gpr_malloc(sizeof(size_t) * count));
  size_t* plaintext_bytes_writtens =
      static_cast<size_t*>(gpr_malloc(sizeof(size_t) * count));
  uint8_t** ciphertext_and_tags =
      static_cast<uint8_t**>(gpr_malloc(sizeof(uint8_t*) * count));
  uint8_t** plaintexts =
      static_cast<uint8_t**>(gpr_malloc(sizeof(uint8_t*) * count));

  // Do encryption
  for (ind = 0; ind < count; ind++) {
    size_t aad_length = (aad_lengths == nullptr) ? 0 : aad_lengths[ind];
    size_t message_length =
        (message_lengths == nullptr) ? 0 : message_lengths[ind];
    gsec_aead_crypter_max_ciphertext_and_tag_length(
        crypter, message_length, &(ciphertext_and_tag_lengths[ind]),
        /*error_details=*/nullptr);
    ciphertext_and_tags[ind] =
        static_cast<uint8_t*>(gpr_malloc(ciphertext_and_tag_lengths[ind]));
    grpc_status_code status = gsec_aead_crypter_encrypt(
        crypter, nonces[ind], nonce_length, aads[ind], aad_length,
        messages[ind], message_length, ciphertext_and_tags[ind],
        ciphertext_and_tag_lengths[ind], &(ciphertext_bytes_writtens[ind]),
        /*error_details=*/nullptr);
    ASSERT_EQ(status, GRPC_STATUS_OK);
    ASSERT_EQ(message_length + tag_length, ciphertext_and_tag_lengths[ind]);
    ASSERT_EQ(ciphertext_bytes_writtens[ind], ciphertext_and_tag_lengths[ind]);
  }
  // Do Decryption
  for (ind = 0; ind < count; ind++) {
    size_t aad_length = (aad_lengths == nullptr) ? 0 : aad_lengths[ind];
    size_t message_length =
        (message_lengths == nullptr) ? 0 : message_lengths[ind];
    gsec_aead_crypter_max_plaintext_length(
        crypter, ciphertext_bytes_writtens[ind], &(plaintext_lengths[ind]),
        /*error_details=*/nullptr);
    plaintexts[ind] = static_cast<uint8_t*>(gpr_malloc(plaintext_lengths[ind]));
    grpc_status_code status = gsec_aead_crypter_decrypt(
        crypter, nonces[ind], nonce_length, aads[ind], aad_length,
        ciphertext_and_tags[ind], ciphertext_bytes_writtens[ind],
        plaintexts[ind], plaintext_lengths[ind],
        &(plaintext_bytes_writtens[ind]), /*error_details=*/nullptr);
    ASSERT_EQ(status, GRPC_STATUS_OK);
    ASSERT_EQ(message_length, plaintext_bytes_writtens[ind]);
    if (message_length != 0) {
      ASSERT_EQ(memcmp(messages[ind], plaintexts[ind], message_length), 0);
    }
  }

  // Slice the plaintext and encrypt with iovecs
  for (ind = 0; ind < count; ind++) {
    size_t aad_length = (aad_lengths == nullptr) ? 0 : aad_lengths[ind];
    struct iovec* aad_vecs = nullptr;
    size_t aad_vecs_length = 0;
    gsec_randomly_slice(aads[ind], aad_length, &aad_vecs, &aad_vecs_length);
    size_t message_length =
        (message_lengths == nullptr) ? 0 : message_lengths[ind];
    struct iovec* message_vecs = nullptr;
    size_t message_vecs_length = 0;
    gsec_randomly_slice(messages[ind], message_length, &message_vecs,
                        &message_vecs_length);

    size_t ciphertext_length = ciphertext_and_tag_lengths[ind];
    uint8_t* another_ciphertext =
        static_cast<uint8_t*>(malloc(ciphertext_length));
    struct iovec another_ciphertext_vec = {another_ciphertext,
                                           ciphertext_length};

    char* error_details = nullptr;
    size_t ciphertext_bytes_written = 0;
    gsec_assert_ok(
        gsec_aead_crypter_encrypt_iovec(
            crypter, nonces[ind], nonce_length, aad_vecs, aad_vecs_length,
            message_vecs, message_vecs_length, another_ciphertext_vec,
            &ciphertext_bytes_written, &error_details),
        error_details);
    ASSERT_EQ(memcmp(ciphertext_and_tags[ind], another_ciphertext_vec.iov_base,
                     ciphertext_length),
              0);
    free(another_ciphertext);
    free(aad_vecs);
    free(message_vecs);
  }

  // Slice the ciphertext and decrypt with iovecs
  for (ind = 0; ind < count; ind++) {
    size_t message_length =
        (message_lengths == nullptr) ? 0 : message_lengths[ind];
    message_length = message_length + 0;

    size_t aad_length = (aad_lengths == nullptr) ? 0 : aad_lengths[ind];

    struct iovec* aad_vecs = nullptr;
    size_t aad_vecs_length = 0;
    gsec_randomly_slice(aads[ind], aad_length, &aad_vecs, &aad_vecs_length);

    struct iovec* ciphertext_vecs = nullptr;
    size_t ciphertext_vecs_length = 0;
    gsec_randomly_slice(ciphertext_and_tags[ind],
                        ciphertext_bytes_writtens[ind], &ciphertext_vecs,
                        &ciphertext_vecs_length);

    size_t decrypted_length = plaintext_lengths[ind];
    uint8_t* decrypted = static_cast<uint8_t*>(malloc(decrypted_length));
    struct iovec decrypted_vec = {decrypted, decrypted_length};

    char* error_details = nullptr;
    gsec_assert_ok(gsec_aead_crypter_decrypt_iovec(
                       crypter, nonces[ind], nonce_length, aad_vecs,
                       aad_vecs_length, ciphertext_vecs, ciphertext_vecs_length,
                       decrypted_vec, &decrypted_length, &error_details),
                   error_details);
    ASSERT_EQ(decrypted_vec.iov_len, message_length);
    if (message_length != 0) {
      ASSERT_EQ(memcmp(decrypted_vec.iov_base, messages[ind], message_length),
                0);
    }
    free(decrypted);
    free(aad_vecs);
    free(ciphertext_vecs);
  }

  for (ind = 0; ind < count; ind++) {
    gpr_free(nonces[ind]);
    gpr_free(aads[ind]);
    gpr_free(messages[ind]);
    gpr_free(ciphertext_and_tags[ind]);
    gpr_free(plaintexts[ind]);
  }
  gpr_free(nonces);
  gpr_free(aads);
  gpr_free(messages);
  gpr_free(ciphertext_and_tag_lengths);
  gpr_free(ciphertext_bytes_writtens);
  gpr_free(plaintext_lengths);
  gpr_free(plaintext_bytes_writtens);
  gpr_free(ciphertext_and_tags);
  gpr_free(plaintexts);
}

static void gsec_test_multiple_encrypt_decrypt(gsec_aead_crypter* crypter) {
  ASSERT_NE(crypter, nullptr);
  size_t count = kTestNumEncryptions;
  size_t* aad_lengths =
      static_cast<size_t*>(gpr_malloc(sizeof(size_t) * count));
  size_t* message_lengths =
      static_cast<size_t*>(gpr_malloc(sizeof(size_t) * count));
  size_t ind;
  for (ind = 0; ind < count; ind++) {
    aad_lengths[ind] = gsec_test_bias_random_uint32(kTestMaxLength);
    message_lengths[ind] = gsec_test_bias_random_uint32(kTestMaxLength);
  }
  gsec_test_multiple_random_encrypt_decrypt(crypter, aad_lengths,
                                            message_lengths, count);
  gsec_test_multiple_random_encrypt_decrypt(crypter, aad_lengths, nullptr,
                                            count);
  gsec_test_multiple_random_encrypt_decrypt(crypter, nullptr, message_lengths,
                                            count);
  gpr_free(aad_lengths);
  gpr_free(message_lengths);
}

static void gsec_test_encryption_failure(gsec_aead_crypter* crypter) {
  ASSERT_NE(crypter, nullptr);
  size_t aad_length = kTestMaxLength;
  size_t message_length = kTestMaxLength;
  size_t nonce_length;

  char* error_message;
  uint8_t *nonce, *aad, *message;

  gsec_aead_crypter_nonce_length(crypter, &nonce_length,
                                 /*error_details=*/nullptr);
  gsec_test_random_array(&nonce, nonce_length);
  gsec_test_random_array(&aad, aad_length);
  gsec_test_random_array(&message, message_length);

  size_t ciphertext_and_tag_length, ciphertext_bytes_written = 0;
  gsec_aead_crypter_max_ciphertext_and_tag_length(crypter, message_length,
                                                  &ciphertext_and_tag_length,
                                                  /*error_details=*/nullptr);
  uint8_t* ciphertext_and_tag =
      static_cast<uint8_t*>(gpr_malloc(ciphertext_and_tag_length));

  // nullptr nonce
  grpc_status_code status = gsec_aead_crypter_encrypt(
      crypter, nullptr, nonce_length, aad, aad_length, message, message_length,
      ciphertext_and_tag, ciphertext_and_tag_length, &ciphertext_bytes_written,
      &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "Nonce buffer is nullptr."));
  gpr_free(error_message);

  // Big nonce
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length + 1, aad, aad_length, message,
      message_length, ciphertext_and_tag, ciphertext_and_tag_length,
      &ciphertext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "Nonce buffer has the wrong length."));
  gpr_free(error_message);

  // Small nonce
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length - 1, aad, aad_length, message,
      message_length, ciphertext_and_tag, ciphertext_and_tag_length,
      &ciphertext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "Nonce buffer has the wrong length."));
  gpr_free(error_message);

  // nullptr aad
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length, nullptr, aad_length, message,
      message_length, ciphertext_and_tag, ciphertext_and_tag_length,
      &ciphertext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message, "aad is nullptr."));
  gpr_free(error_message);

  // nullptr aad with zero length
  gsec_assert_ok(
      gsec_aead_crypter_encrypt(crypter, nonce, nonce_length, nullptr, 0,
                                message, message_length, ciphertext_and_tag,
                                ciphertext_and_tag_length,
                                &ciphertext_bytes_written, &error_message),
      error_message);

  // nullptr plaintext
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length, aad, aad_length, nullptr, message_length,
      ciphertext_and_tag, ciphertext_and_tag_length, &ciphertext_bytes_written,
      &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "plaintext is nullptr."));
  gpr_free(error_message);

  // nullptr ciphertext
  status = gsec_aead_crypter_encrypt(crypter, nonce, nonce_length, aad,
                                     aad_length, message, message_length,
                                     nullptr, ciphertext_and_tag_length,
                                     &ciphertext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "ciphertext is nullptr."));
  gpr_free(error_message);

  // Short ciphertext
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length, aad, aad_length, message, message_length,
      ciphertext_and_tag, ciphertext_and_tag_length - 1,
      &ciphertext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "ciphertext is too small to hold a tag."));
  gpr_free(error_message);

  // nullptr ciphertext_bytes_written
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length, aad, aad_length, message, message_length,
      ciphertext_and_tag, ciphertext_and_tag_length, nullptr, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "bytes_written is nullptr."));
  gpr_free(error_message);

  // nullptr plaintext/ciphertext encrypt with zero length
  gsec_assert_ok(gsec_aead_crypter_encrypt(
                     crypter, nonce, nonce_length, aad, aad_length, nullptr, 0,
                     ciphertext_and_tag, ciphertext_and_tag_length,
                     &ciphertext_bytes_written, &error_message),
                 error_message);

  // Success
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length, aad, aad_length, message, message_length,
      ciphertext_and_tag, ciphertext_and_tag_length, &ciphertext_bytes_written,
      &error_message);
  ASSERT_EQ(status, GRPC_STATUS_OK);

  gpr_free(message);
  gpr_free(aad);
  gpr_free(nonce);
  gpr_free(ciphertext_and_tag);
}

static void gsec_test_decryption_failure(gsec_aead_crypter* crypter) {
  ASSERT_NE(crypter, nullptr);
  size_t aad_length = kTestMaxLength;
  size_t message_length = kTestMaxLength;
  size_t nonce_length, tag_length;
  uint8_t *nonce, *aad, *message;

  gsec_aead_crypter_nonce_length(crypter, &nonce_length,
                                 /*error_details=*/nullptr);
  gsec_aead_crypter_tag_length(crypter, &tag_length, /*error_details=*/nullptr);
  gsec_test_random_array(&nonce, nonce_length);
  gsec_test_random_array(&aad, aad_length);
  gsec_test_random_array(&message, message_length);

  // Test encryption
  size_t ciphertext_and_tag_length, ciphertext_bytes_written = 0;
  gsec_aead_crypter_max_ciphertext_and_tag_length(crypter, message_length,
                                                  &ciphertext_and_tag_length,
                                                  /*error_details=*/nullptr);
  uint8_t* ciphertext_and_tag =
      static_cast<uint8_t*>(gpr_malloc(ciphertext_and_tag_length));

  grpc_status_code status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length, aad, aad_length, message, message_length,
      ciphertext_and_tag, ciphertext_and_tag_length, &ciphertext_bytes_written,
      /*error_details=*/nullptr);
  ASSERT_EQ(status, GRPC_STATUS_OK);
  ASSERT_EQ(ciphertext_bytes_written, ciphertext_and_tag_length);

  size_t plaintext_length, plaintext_bytes_written = 0;
  gsec_aead_crypter_max_plaintext_length(crypter, ciphertext_bytes_written,
                                         &plaintext_length,
                                         /*error_details=*/nullptr);
  uint8_t* plaintext = static_cast<uint8_t*>(gpr_malloc(plaintext_length));

  char* error_message;
  // nullptr nonce
  status = gsec_aead_crypter_decrypt(
      crypter, nullptr, nonce_length, aad, aad_length, ciphertext_and_tag,
      ciphertext_and_tag_length, plaintext, plaintext_length,
      &plaintext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "Nonce buffer is nullptr."));
  gpr_free(error_message);

  // Big nonce
  status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length + 1, aad, aad_length, ciphertext_and_tag,
      ciphertext_and_tag_length, plaintext, plaintext_length,
      &plaintext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "Nonce buffer has the wrong length."));
  gpr_free(error_message);

  // Small nonce
  status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length - 1, aad, aad_length, ciphertext_and_tag,
      ciphertext_and_tag_length, plaintext, plaintext_length,
      &plaintext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "Nonce buffer has the wrong length."));
  gpr_free(error_message);

  // nullptr aad
  status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length, nullptr, aad_length, ciphertext_and_tag,
      ciphertext_and_tag_length, plaintext, plaintext_length,
      &plaintext_bytes_written, &error_message);
  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message, "aad is nullptr."));
  gpr_free(error_message);

  // nullptr aad with zero length
  status = gsec_aead_crypter_encrypt(
      crypter, nonce, nonce_length, nullptr, 0, message, message_length,
      ciphertext_and_tag, ciphertext_and_tag_length, &ciphertext_bytes_written,
      &error_message);
  ASSERT_EQ(status, GRPC_STATUS_OK);

  status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length, nullptr, 0, ciphertext_and_tag,
      ciphertext_and_tag_length, plaintext, plaintext_length,
      &plaintext_bytes_written, &error_message);
  ASSERT_EQ(status, GRPC_STATUS_OK);

  // Small ciphertext
  if (tag_length > 0) {
    status = gsec_aead_crypter_decrypt(
        crypter, nonce, nonce_length, aad, aad_length, ciphertext_and_tag,
        tag_length - 1, plaintext, plaintext_length, &plaintext_bytes_written,
        &error_message);

    ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
        status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
        "ciphertext is too small to hold a tag."));
    gpr_free(error_message);
  }

  // nullptr ciphertext
  status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length, aad, aad_length, nullptr,
      ciphertext_and_tag_length, plaintext, plaintext_length,
      &plaintext_bytes_written, &error_message);

  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "ciphertext is nullptr."));
  gpr_free(error_message);

  // nullptr plaintext
  status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length, aad, aad_length, ciphertext_and_tag,
      ciphertext_and_tag_length, nullptr, plaintext_length,
      &plaintext_bytes_written, &error_message);

  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "plaintext is nullptr, but plaintext_length is positive."));
  gpr_free(error_message);

  // Short plaintext
  status = gsec_aead_crypter_decrypt(
      crypter, nonce, nonce_length, aad, aad_length, ciphertext_and_tag,
      ciphertext_and_tag_length, plaintext, plaintext_length - 1,
      &plaintext_bytes_written, &error_message);

  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "Not enough plaintext buffer to hold encrypted ciphertext."));
  gpr_free(error_message);

  // nullptr plaintext_bytes_written
  status = gsec_aead_crypter_decrypt(crypter, nonce, nonce_length, aad,
                                     aad_length, ciphertext_and_tag,
                                     ciphertext_and_tag_length, plaintext,
                                     plaintext_length, nullptr, &error_message);

  ASSERT_TRUE(gsec_test_expect_compare_code_and_substr(
      status, GRPC_STATUS_INVALID_ARGUMENT, error_message,
      "bytes_written is nullptr."));
  gpr_free(error_message);

  gpr_free(message);
  gpr_free(plaintext);
  gpr_free(ciphertext_and_tag);
  gpr_free(aad);
  gpr_free(nonce);
}

static void gsec_test_encrypt_decrypt_test_vector(
    gsec_aead_crypter* crypter, gsec_aead_test_vector* test_vector) {
  ASSERT_NE(crypter, nullptr);
  // Test byte-based encryption interface.
  size_t ciphertext_and_tag_length, ciphertext_bytes_written = 0;
  gsec_aead_crypter_max_ciphertext_and_tag_length(
      crypter, test_vector->plaintext_length, &ciphertext_and_tag_length,
      /*error_details=*/nullptr);
  uint8_t* ciphertext_and_tag_bytes =
      static_cast<uint8_t*>(gpr_malloc(ciphertext_and_tag_length));
  grpc_status_code status = gsec_aead_crypter_encrypt(
      crypter, test_vector->nonce, test_vector->nonce_length, test_vector->aad,
      test_vector->aad_length, test_vector->plaintext,
      test_vector->plaintext_length, ciphertext_and_tag_bytes,
      ciphertext_and_tag_length, &ciphertext_bytes_written,
      /*error_details=*/nullptr);

  ASSERT_EQ(status, GRPC_STATUS_OK);
  ASSERT_EQ(ciphertext_bytes_written, ciphertext_and_tag_length);
  ASSERT_EQ(memcmp(test_vector->ciphertext_and_tag, ciphertext_and_tag_bytes,
                   ciphertext_and_tag_length),
            0);

  // Test byte-based decryption interface
  size_t plaintext_length, plaintext_bytes_written = 0;
  gsec_aead_crypter_max_plaintext_length(crypter, ciphertext_and_tag_length,
                                         &plaintext_length,
                                         /*error_details=*/nullptr);
  uint8_t* plaintext_bytes =
      static_cast<uint8_t*>(gpr_malloc(plaintext_length));
  status = gsec_aead_crypter_decrypt(
      crypter, test_vector->nonce, test_vector->nonce_length, test_vector->aad,
      test_vector->aad_length, test_vector->ciphertext_and_tag,
      test_vector->ciphertext_and_tag_length, plaintext_bytes, plaintext_length,
      &plaintext_bytes_written, /*error_details=*/nullptr);
  ASSERT_EQ(status, GRPC_STATUS_OK);
  if (plaintext_bytes_written != 0) {
    ASSERT_EQ(memcmp(test_vector->plaintext, plaintext_bytes,
                     plaintext_bytes_written),
              0);
  }

  gpr_free(ciphertext_and_tag_bytes);
  gpr_free(plaintext_bytes);
}

static void gsec_test_get_crypter_from_test_vector(
    gsec_aead_crypter** crypter, gsec_aead_test_vector* test_vector,
    bool rekey = false) {
  size_t key_length = test_vector->key_length;
  ASSERT_TRUE(key_length == kAes128GcmKeyLength ||
              key_length == kAes256GcmKeyLength ||
              key_length == kAes128GcmRekeyKeyLength);
  size_t nonce_length = test_vector->nonce_length;
  ASSERT_EQ(nonce_length, kAesGcmNonceLength);
  size_t plaintext_length = test_vector->plaintext_length;
  size_t ciphertext_and_tag_length = test_vector->ciphertext_and_tag_length;
  ASSERT_EQ(ciphertext_and_tag_length, plaintext_length + kAesGcmTagLength);
  size_t tag_length = ciphertext_and_tag_length - plaintext_length;
  gsec_aes_gcm_aead_crypter_create(
      std::make_unique<grpc_core::GsecKey>(
          absl::MakeConstSpan(test_vector->key, key_length), rekey),
      nonce_length, tag_length, crypter, /*error_details=*/nullptr);
}

static void gsec_test_verify_crypter_on_test_vector(
    gsec_aead_test_vector* test_vector, bool rekey = false) {
  gsec_aead_crypter* crypter;
  gsec_test_get_crypter_from_test_vector(&crypter, test_vector, rekey);
  gsec_test_encrypt_decrypt_test_vector(crypter, test_vector);
  gsec_aead_crypter_destroy(crypter);
}

static void gsec_aead_malloc_test_vector(
    gsec_aead_test_vector** test_vector, const uint8_t* key, size_t key_length,
    const uint8_t* nonce, size_t nonce_length, const uint8_t* aad,
    size_t aad_length, const uint8_t* plaintext, size_t plaintext_length,
    const uint8_t* ciphertext_and_tag, size_t ciphertext_and_tag_length) {
  *test_vector = static_cast<gsec_aead_test_vector*>(
      gpr_malloc(sizeof(gsec_aead_test_vector)));
  (*test_vector)->key_length = key_length;
  (*test_vector)->nonce_length = nonce_length;
  (*test_vector)->aad_length = aad_length;
  (*test_vector)->plaintext_length = plaintext_length;
  (*test_vector)->ciphertext_and_tag_length = ciphertext_and_tag_length;
  gsec_test_copy(key, &((*test_vector)->key), key_length);
  gsec_test_copy(nonce, &((*test_vector)->nonce), nonce_length);
  gsec_test_copy(aad, &((*test_vector)->aad), aad_length);
  gsec_test_copy(plaintext, &((*test_vector)->plaintext), plaintext_length);
  gsec_test_copy(ciphertext_and_tag, &((*test_vector)->ciphertext_and_tag),
                 ciphertext_and_tag_length);
}

static void gsec_aead_free_test_vector(gsec_aead_test_vector* test_vector) {
  gpr_free(test_vector->key);
  gpr_free(test_vector->nonce);
  gpr_free(test_vector->aad);
  gpr_free(test_vector->plaintext);
  gpr_free(test_vector->ciphertext_and_tag);
  gpr_free(test_vector);
}

static void gsec_test_create_random_aes_gcm_crypter(gsec_aead_crypter** crypter,
                                                    size_t key_length,
                                                    size_t nonce_length,
                                                    size_t tag_length,
                                                    bool rekey) {
  uint8_t* key;
  gsec_test_random_array(&key, key_length);
  gsec_aes_gcm_aead_crypter_create(
      std::make_unique<grpc_core::GsecKey>(absl::MakeConstSpan(key, key_length),
                                           rekey),
      nonce_length, tag_length, crypter, /*error_details=*/nullptr);
  gpr_free(key);
}

static void gsec_test_get_random_aes_gcm_crypters(
    gsec_aead_crypter*** crypters) {
  *crypters = static_cast<gsec_aead_crypter**>(
      gpr_malloc(sizeof(gsec_aead_crypter*) * kTestNumCrypters));
  gsec_test_create_random_aes_gcm_crypter(
      &((*crypters)[0]), kAes128GcmKeyLength, kAesGcmNonceLength,
      kAesGcmTagLength, /*rekey=*/false);
  gsec_test_create_random_aes_gcm_crypter(
      &((*crypters)[1]), kAes256GcmKeyLength, kAesGcmNonceLength,
      kAesGcmTagLength, /*rekey=*/false);
  gsec_test_create_random_aes_gcm_crypter(
      &((*crypters)[2]), kAes128GcmRekeyKeyLength, kAesGcmNonceLength,
      kAesGcmTagLength, /*rekey=*/true);
}

TEST(AltsCryptTest, GsecKeyCreationIsRekey) {
  uint8_t* key;
  gsec_test_random_array(&key, kAes128GcmRekeyKeyLength);
  grpc_core::GsecKey gsec_key({key, kAes128GcmRekeyKeyLength},
                              /*is_rekey=*/true);
  EXPECT_TRUE(gsec_key.IsRekey());
  EXPECT_EQ(gsec_key.key().size(), kAes256GcmKeyLength);
  EXPECT_EQ(gsec_key.aead_key().size(), kAes128GcmKeyLength);
  EXPECT_EQ(gsec_key.kdf_counter().size(), 6);
  EXPECT_EQ(gsec_key.nonce_mask().size(), kAesGcmNonceLength);
  gpr_free(key);
}

TEST(AltsCryptTest, GsecKeyCreationIsNotRekey) {
  uint8_t* key;
  gsec_test_random_array(&key, kAes256GcmKeyLength);
  grpc_core::GsecKey gsec_key({key, kAes256GcmKeyLength},
                              /*is_rekey=*/false);
  EXPECT_FALSE(gsec_key.IsRekey());
  EXPECT_EQ(gsec_key.key().size(), kAes256GcmKeyLength);
  gpr_free(key);
}

TEST(AltsCryptTest, GsecTestDoGenericCrypterTests) {
  gsec_aead_crypter** crypters;
  gsec_test_get_random_aes_gcm_crypters(&crypters);
  size_t ind;
  for (ind = 0; ind < kTestNumCrypters; ind++) {
    gsec_test_encrypt_decrypt(crypters[ind]);
    gsec_test_multiple_encrypt_decrypt(crypters[ind]);
    gsec_test_encryption_failure(crypters[ind]);
    gsec_test_decryption_failure(crypters[ind]);
  }
  for (ind = 0; ind < kTestNumCrypters; ind++) {
    gsec_aead_crypter_destroy(crypters[ind]);
  }
  gpr_free(crypters);
}

TEST(AltsCryptTest, GsecTestDoVectorTestsRekeyNist) {
  // NIST vectors from:
  // http://csrc.nist.gov/groups/ST/toolkit/BCM/documents/proposedmodes/gcm/gcm-revised-spec.pdf
  //
  // IEEE vectors from:
  // http://www.ieee802.org/1/files/public/docs2011/bn-randall-test-vectors-0511-v1.pdf
  //
  // Key expanded by setting expandedKey = (key||(key ^ {0x01, .., 0x01})||key ^
  // {0x02,..,0x02}))[0:44].

  gsec_aead_test_vector vec;

  // Derived from NIST test vector 1
  uint8_t nonce_0[] = {0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                       0x0, 0x0, 0x0, 0x0, 0x0, 0x0};
  uint8_t aad_0[1] = {};
  uint8_t key_0[] = {0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2,
                     0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2};
  uint8_t plaintext_0[1] = {};
  uint8_t ciphertext_0[] = {0x85, 0xE8, 0x73, 0xE0, 0x2,  0xF6, 0xEB, 0xDC,
                            0x40, 0x60, 0x95, 0x4E, 0xB8, 0x67, 0x55, 0x8};
  vec = {nonce_0, aad_0, key_0, plaintext_0, ciphertext_0, 12, 0, 44, 0, 16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from NIST test vector 2
  uint8_t nonce_1[] = {0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                       0x0, 0x0, 0x0, 0x0, 0x0, 0x0};
  uint8_t aad_1[1] = {};
  uint8_t key_1[] = {0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
                     0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x2,
                     0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2, 0x2};
  uint8_t plaintext_1[] = {0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0};
  uint8_t ciphertext_1[] = {0x51, 0xE9, 0xA8, 0xCB, 0x23, 0xCA, 0x25, 0x12,
                            0xC8, 0x25, 0x6A, 0xFF, 0xF8, 0xE7, 0x2D, 0x68,
                            0x1A, 0xCA, 0x19, 0xA1, 0x14, 0x8A, 0xC1, 0x15,
                            0xE8, 0x3D, 0xF4, 0x88, 0x8C, 0xC0, 0xD,  0x11};
  vec = {nonce_1, aad_1, key_1, plaintext_1, ciphertext_1, 12, 0, 44, 16, 32};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from NIST test vector 3
  uint8_t nonce_2[] = {0xCA, 0xFE, 0xBA, 0xBE, 0xFA, 0xCE,
                       0xDB, 0xAD, 0xDE, 0xCA, 0xF8, 0x88};
  uint8_t aad_2[1] = {};
  uint8_t key_2[] = {0xFE, 0xFF, 0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D,
                     0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x8,  0xFF, 0xFE,
                     0xE8, 0x93, 0x87, 0x64, 0x72, 0x1D, 0x6C, 0x6B, 0x8E,
                     0x95, 0x66, 0x31, 0x82, 0x9,  0xFC, 0xFD, 0xEB, 0x90,
                     0x84, 0x67, 0x71, 0x1E, 0x6F, 0x68, 0x8D, 0x96};
  uint8_t plaintext_2[] = {
      0xD9, 0x31, 0x32, 0x25, 0xF8, 0x84, 0x6,  0xE5, 0xA5, 0x59, 0x9,
      0xC5, 0xAF, 0xF5, 0x26, 0x9A, 0x86, 0xA7, 0xA9, 0x53, 0x15, 0x34,
      0xF7, 0xDA, 0x2E, 0x4C, 0x30, 0x3D, 0x8A, 0x31, 0x8A, 0x72, 0x1C,
      0x3C, 0xC,  0x95, 0x95, 0x68, 0x9,  0x53, 0x2F, 0xCF, 0xE,  0x24,
      0x49, 0xA6, 0xB5, 0x25, 0xB1, 0x6A, 0xED, 0xF5, 0xAA, 0xD,  0xE6,
      0x57, 0xBA, 0x63, 0x7B, 0x39, 0x1A, 0xAF, 0xD2, 0x55};
  uint8_t ciphertext_2[] = {
      0x10, 0x18, 0xED, 0x5A, 0x14, 0x2,  0xA8, 0x65, 0x16, 0xD6, 0x57, 0x6D,
      0x70, 0xB2, 0xFF, 0xCC, 0xCA, 0x26, 0x1B, 0x94, 0xDF, 0x88, 0xB5, 0x8F,
      0x53, 0xB6, 0x4D, 0xFB, 0xA4, 0x35, 0xD1, 0x8B, 0x2F, 0x6E, 0x3B, 0x78,
      0x69, 0xF9, 0x35, 0x3D, 0x4A, 0xC8, 0xCF, 0x9,  0xAF, 0xB1, 0x66, 0x3D,
      0xAA, 0x7B, 0x40, 0x17, 0xE6, 0xFC, 0x2C, 0x17, 0x7C, 0xC,  0x8,  0x7C,
      0xD,  0xF1, 0x16, 0x21, 0x29, 0x95, 0x22, 0x13, 0xCE, 0xE1, 0xBC, 0x6E,
      0x9C, 0x84, 0x95, 0xDD, 0x70, 0x5E, 0x1F, 0x3D};
  vec = {nonce_2, aad_2, key_2, plaintext_2, ciphertext_2, 12, 0, 44, 64, 80};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from NIST test vector 4
  uint8_t nonce_3[] = {0xCA, 0xFE, 0xBA, 0xBE, 0xFA, 0xCE,
                       0xDB, 0xAD, 0xDE, 0xCA, 0xF8, 0x88};
  uint8_t aad_3[] = {0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE,
                     0xEF, 0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD,
                     0xBE, 0xEF, 0xAB, 0xAD, 0xDA, 0xD2};
  uint8_t key_3[] = {0xFE, 0xFF, 0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D,
                     0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x8,  0xFF, 0xFE,
                     0xE8, 0x93, 0x87, 0x64, 0x72, 0x1D, 0x6C, 0x6B, 0x8E,
                     0x95, 0x66, 0x31, 0x82, 0x9,  0xFC, 0xFD, 0xEB, 0x90,
                     0x84, 0x67, 0x71, 0x1E, 0x6F, 0x68, 0x8D, 0x96};
  uint8_t plaintext_3[] = {
      0xD9, 0x31, 0x32, 0x25, 0xF8, 0x84, 0x6,  0xE5, 0xA5, 0x59, 0x9,  0xC5,
      0xAF, 0xF5, 0x26, 0x9A, 0x86, 0xA7, 0xA9, 0x53, 0x15, 0x34, 0xF7, 0xDA,
      0x2E, 0x4C, 0x30, 0x3D, 0x8A, 0x31, 0x8A, 0x72, 0x1C, 0x3C, 0xC,  0x95,
      0x95, 0x68, 0x9,  0x53, 0x2F, 0xCF, 0xE,  0x24, 0x49, 0xA6, 0xB5, 0x25,
      0xB1, 0x6A, 0xED, 0xF5, 0xAA, 0xD,  0xE6, 0x57, 0xBA, 0x63, 0x7B, 0x39};
  uint8_t ciphertext_3[] = {
      0x10, 0x18, 0xED, 0x5A, 0x14, 0x2,  0xA8, 0x65, 0x16, 0xD6, 0x57,
      0x6D, 0x70, 0xB2, 0xFF, 0xCC, 0xCA, 0x26, 0x1B, 0x94, 0xDF, 0x88,
      0xB5, 0x8F, 0x53, 0xB6, 0x4D, 0xFB, 0xA4, 0x35, 0xD1, 0x8B, 0x2F,
      0x6E, 0x3B, 0x78, 0x69, 0xF9, 0x35, 0x3D, 0x4A, 0xC8, 0xCF, 0x9,
      0xAF, 0xB1, 0x66, 0x3D, 0xAA, 0x7B, 0x40, 0x17, 0xE6, 0xFC, 0x2C,
      0x17, 0x7C, 0xC,  0x8,  0x7C, 0x47, 0x64, 0x56, 0x5D, 0x7,  0x7E,
      0x91, 0x24, 0x0,  0x1D, 0xDB, 0x27, 0xFC, 0x8,  0x48, 0xC5};
  vec = {nonce_3, aad_3, key_3, plaintext_3, ciphertext_3, 12, 20, 44, 60, 76};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from adapted NIST test vector 4 for KDF counter boundary (flip
  // nonce bit 15)
  uint8_t nonce_4[] = {0xCA, 0x7E, 0xBA, 0xBE, 0xFA, 0xCE,
                       0xDB, 0xAD, 0xDE, 0xCA, 0xF8, 0x88};
  uint8_t aad_4[] = {0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE,
                     0xEF, 0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD,
                     0xBE, 0xEF, 0xAB, 0xAD, 0xDA, 0xD2};
  uint8_t key_4[] = {0xFE, 0xFF, 0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D,
                     0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x8,  0xFF, 0xFE,
                     0xE8, 0x93, 0x87, 0x64, 0x72, 0x1D, 0x6C, 0x6B, 0x8E,
                     0x95, 0x66, 0x31, 0x82, 0x9,  0xFC, 0xFD, 0xEB, 0x90,
                     0x84, 0x67, 0x71, 0x1E, 0x6F, 0x68, 0x8D, 0x96};
  uint8_t plaintext_4[] = {
      0xD9, 0x31, 0x32, 0x25, 0xF8, 0x84, 0x6,  0xE5, 0xA5, 0x59, 0x9,  0xC5,
      0xAF, 0xF5, 0x26, 0x9A, 0x86, 0xA7, 0xA9, 0x53, 0x15, 0x34, 0xF7, 0xDA,
      0x2E, 0x4C, 0x30, 0x3D, 0x8A, 0x31, 0x8A, 0x72, 0x1C, 0x3C, 0xC,  0x95,
      0x95, 0x68, 0x9,  0x53, 0x2F, 0xCF, 0xE,  0x24, 0x49, 0xA6, 0xB5, 0x25,
      0xB1, 0x6A, 0xED, 0xF5, 0xAA, 0xD,  0xE6, 0x57, 0xBA, 0x63, 0x7B, 0x39};
  uint8_t ciphertext_4[] = {
      0xE6, 0x50, 0xD3, 0xC0, 0xFB, 0x87, 0x93, 0x27, 0xF2, 0xD0, 0x32,
      0x87, 0xFA, 0x93, 0xCD, 0x7,  0x34, 0x2B, 0x13, 0x62, 0x15, 0xAD,
      0xBC, 0xA0, 0xC,  0x3B, 0xD5, 0x9,  0x9E, 0xC4, 0x18, 0x32, 0xB1,
      0xD1, 0x8E, 0x4,  0x23, 0xED, 0x26, 0xBB, 0x12, 0xC6, 0xCD, 0x9,
      0xDE, 0xBB, 0x29, 0x23, 0xA,  0x94, 0xC0, 0xCE, 0xE1, 0x59, 0x3,
      0x65, 0x6F, 0x85, 0xED, 0xB6, 0xFC, 0x50, 0x9B, 0x1B, 0x28, 0x21,
      0x63, 0x82, 0x17, 0x2E, 0xCB, 0xCC, 0x31, 0xE1, 0xE9, 0xB1};
  vec = {nonce_4, aad_4, key_4, plaintext_4, ciphertext_4, 12, 20, 44, 60, 76};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from adapted NIST test vector 4 for KDF counter boundary (flip
  // nonce bit 16)
  uint8_t nonce_5[] = {0xCA, 0xFE, 0xBB, 0xBE, 0xFA, 0xCE,
                       0xDB, 0xAD, 0xDE, 0xCA, 0xF8, 0x88};
  uint8_t aad_5[] = {0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE,
                     0xEF, 0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD,
                     0xBE, 0xEF, 0xAB, 0xAD, 0xDA, 0xD2};
  uint8_t key_5[] = {0xFE, 0xFF, 0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D,
                     0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x8,  0xFF, 0xFE,
                     0xE8, 0x93, 0x87, 0x64, 0x72, 0x1D, 0x6C, 0x6B, 0x8E,
                     0x95, 0x66, 0x31, 0x82, 0x9,  0xFC, 0xFD, 0xEB, 0x90,
                     0x84, 0x67, 0x71, 0x1E, 0x6F, 0x68, 0x8D, 0x96};
  uint8_t plaintext_5[] = {
      0xD9, 0x31, 0x32, 0x25, 0xF8, 0x84, 0x6,  0xE5, 0xA5, 0x59, 0x9,  0xC5,
      0xAF, 0xF5, 0x26, 0x9A, 0x86, 0xA7, 0xA9, 0x53, 0x15, 0x34, 0xF7, 0xDA,
      0x2E, 0x4C, 0x30, 0x3D, 0x8A, 0x31, 0x8A, 0x72, 0x1C, 0x3C, 0xC,  0x95,
      0x95, 0x68, 0x9,  0x53, 0x2F, 0xCF, 0xE,  0x24, 0x49, 0xA6, 0xB5, 0x25,
      0xB1, 0x6A, 0xED, 0xF5, 0xAA, 0xD,  0xE6, 0x57, 0xBA, 0x63, 0x7B, 0x39};
  uint8_t ciphertext_5[] = {
      0xC0, 0x12, 0x1E, 0x6C, 0x95, 0x4D, 0x7,  0x67, 0xF9, 0x66, 0x30,
      0xC3, 0x34, 0x50, 0x99, 0x97, 0x91, 0xB2, 0xDA, 0x2A, 0xD0, 0x5C,
      0x41, 0x90, 0x16, 0x9C, 0xCA, 0xD9, 0xAC, 0x86, 0xFF, 0x1C, 0x72,
      0x1E, 0x3D, 0x82, 0xF2, 0xAD, 0x22, 0xAB, 0x46, 0x3B, 0xAB, 0x4A,
      0x7,  0x54, 0xB7, 0xDD, 0x68, 0xCA, 0x4D, 0xE7, 0xEA, 0x25, 0x31,
      0xB6, 0x25, 0xED, 0xA0, 0x1F, 0x89, 0x31, 0x2B, 0x2A, 0xB9, 0x57,
      0xD5, 0xC7, 0xF8, 0x56, 0x8D, 0xD9, 0x5F, 0xCD, 0xCD, 0x1F};
  vec = {nonce_5, aad_5, key_5, plaintext_5, ciphertext_5, 12, 20, 44, 60, 76};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from adapted NIST test vector 4 for KDF counter boundary (flip
  // nonce bit 63)
  uint8_t nonce_6[] = {0xCA, 0xFE, 0xBA, 0xBE, 0xFA, 0xCE,
                       0xDB, 0x2D, 0xDE, 0xCA, 0xF8, 0x88};
  uint8_t aad_6[] = {0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE,
                     0xEF, 0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD,
                     0xBE, 0xEF, 0xAB, 0xAD, 0xDA, 0xD2};
  uint8_t key_6[] = {0xFE, 0xFF, 0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D,
                     0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x8,  0xFF, 0xFE,
                     0xE8, 0x93, 0x87, 0x64, 0x72, 0x1D, 0x6C, 0x6B, 0x8E,
                     0x95, 0x66, 0x31, 0x82, 0x9,  0xFC, 0xFD, 0xEB, 0x90,
                     0x84, 0x67, 0x71, 0x1E, 0x6F, 0x68, 0x8D, 0x96};
  uint8_t plaintext_6[] = {
      0xD9, 0x31, 0x32, 0x25, 0xF8, 0x84, 0x6,  0xE5, 0xA5, 0x59, 0x9,  0xC5,
      0xAF, 0xF5, 0x26, 0x9A, 0x86, 0xA7, 0xA9, 0x53, 0x15, 0x34, 0xF7, 0xDA,
      0x2E, 0x4C, 0x30, 0x3D, 0x8A, 0x31, 0x8A, 0x72, 0x1C, 0x3C, 0xC,  0x95,
      0x95, 0x68, 0x9,  0x53, 0x2F, 0xCF, 0xE,  0x24, 0x49, 0xA6, 0xB5, 0x25,
      0xB1, 0x6A, 0xED, 0xF5, 0xAA, 0xD,  0xE6, 0x57, 0xBA, 0x63, 0x7B, 0x39};
  uint8_t ciphertext_6[] = {
      0x8A, 0xF3, 0x7E, 0xA5, 0x68, 0x4A, 0x4D, 0x81, 0xD4, 0xFD, 0x81,
      0x72, 0x61, 0xFD, 0x97, 0x43, 0x9,  0x9E, 0x7E, 0x6A, 0x2,  0x5E,
      0xAA, 0xCF, 0x8E, 0x54, 0xB1, 0x24, 0xFB, 0x57, 0x43, 0x14, 0x9E,
      0x5,  0xCB, 0x89, 0xF4, 0xA4, 0x94, 0x67, 0xFE, 0x2E, 0x5E, 0x59,
      0x65, 0xF2, 0x9A, 0x19, 0xF9, 0x94, 0x16, 0xB0, 0x1,  0x6B, 0x54,
      0x58, 0x5D, 0x12, 0x55, 0x37, 0x83, 0xBA, 0x59, 0xE9, 0xF7, 0x82,
      0xE8, 0x2E, 0x9,  0x7C, 0x33, 0x6B, 0xF7, 0x98, 0x9F, 0x8};
  vec = {nonce_6, aad_6, key_6, plaintext_6, ciphertext_6, 12, 20, 44, 60, 76};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from adapted NIST test vector 4 for KDF counter boundary (flip
  // nonce bit 64)
  uint8_t nonce_7[] = {0xCA, 0xFE, 0xBA, 0xBE, 0xFA, 0xCE,
                       0xDB, 0xAD, 0xDF, 0xCA, 0xF8, 0x88};
  uint8_t aad_7[] = {0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE,
                     0xEF, 0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD,
                     0xBE, 0xEF, 0xAB, 0xAD, 0xDA, 0xD2};
  uint8_t key_7[] = {0xFE, 0xFF, 0xE9, 0x92, 0x86, 0x65, 0x73, 0x1C, 0x6D,
                     0x6A, 0x8F, 0x94, 0x67, 0x30, 0x83, 0x8,  0xFF, 0xFE,
                     0xE8, 0x93, 0x87, 0x64, 0x72, 0x1D, 0x6C, 0x6B, 0x8E,
                     0x95, 0x66, 0x31, 0x82, 0x9,  0xFC, 0xFD, 0xEB, 0x90,
                     0x84, 0x67, 0x71, 0x1E, 0x6F, 0x68, 0x8D, 0x96};
  uint8_t plaintext_7[] = {
      0xD9, 0x31, 0x32, 0x25, 0xF8, 0x84, 0x6,  0xE5, 0xA5, 0x59, 0x9,  0xC5,
      0xAF, 0xF5, 0x26, 0x9A, 0x86, 0xA7, 0xA9, 0x53, 0x15, 0x34, 0xF7, 0xDA,
      0x2E, 0x4C, 0x30, 0x3D, 0x8A, 0x31, 0x8A, 0x72, 0x1C, 0x3C, 0xC,  0x95,
      0x95, 0x68, 0x9,  0x53, 0x2F, 0xCF, 0xE,  0x24, 0x49, 0xA6, 0xB5, 0x25,
      0xB1, 0x6A, 0xED, 0xF5, 0xAA, 0xD,  0xE6, 0x57, 0xBA, 0x63, 0x7B, 0x39};
  uint8_t ciphertext_7[] = {
      0xFB, 0xD5, 0x28, 0x44, 0x8D, 0x3,  0x46, 0xBF, 0xA8, 0x78, 0x63,
      0x48, 0x64, 0xD4, 0x7,  0xA3, 0x5A, 0x3,  0x9D, 0xE9, 0xDB, 0x2F,
      0x1F, 0xEB, 0x8E, 0x96, 0x5B, 0x3A, 0xE9, 0x35, 0x6C, 0xE6, 0x28,
      0x94, 0x41, 0xD7, 0x7F, 0x8F, 0xD,  0xF2, 0x94, 0x89, 0x1F, 0x37,
      0xEA, 0x43, 0x8B, 0x22, 0x3E, 0x3B, 0xF2, 0xBD, 0xC5, 0x3D, 0x4C,
      0x5A, 0x74, 0xFB, 0x68, 0xB,  0xB3, 0x12, 0xA8, 0xDE, 0xC6, 0xF7,
      0x25, 0x2C, 0xBC, 0xD7, 0xF5, 0x79, 0x97, 0x50, 0xAD, 0x78};
  vec = {nonce_7, aad_7, key_7, plaintext_7, ciphertext_7, 12, 20, 44, 60, 76};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);
}

TEST(AltsCryptTest, GsecTestDoVectorTestsRekeyIeee) {
  // IEEE vectors from:
  // http://www.ieee802.org/1/files/public/docs2011/bn-randall-test-vectors-0511-v1.pdf
  //
  // Key expanded by setting expandedKey = (key||(key ^ {0x01, .., 0x01})||key ^
  // {0x02,..,0x02}))[0:44].

  gsec_aead_test_vector vec;

  // Derived from IEEE 2.1.1 54-byte auth
  uint8_t nonce_8[] = {0x12, 0x15, 0x35, 0x24, 0xC0, 0x89,
                       0x5E, 0x81, 0xB2, 0xC2, 0x84, 0x65};
  uint8_t aad_8[] = {0xD6, 0x9,  0xB1, 0xF0, 0x56, 0x63, 0x7A, 0xD,  0x46, 0xDF,
                     0x99, 0x8D, 0x88, 0xE5, 0x22, 0x2A, 0xB2, 0xC2, 0x84, 0x65,
                     0x12, 0x15, 0x35, 0x24, 0xC0, 0x89, 0x5E, 0x81, 0x8,  0x0,
                     0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                     0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22,
                     0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
                     0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x0,  0x1};
  uint8_t key_8[] = {0xAD, 0x7A, 0x2B, 0xD0, 0x3E, 0xAC, 0x83, 0x5A, 0x6F,
                     0x62, 0xF,  0xDC, 0xB5, 0x6,  0xB3, 0x45, 0xAC, 0x7B,
                     0x2A, 0xD1, 0x3F, 0xAD, 0x82, 0x5B, 0x6E, 0x63, 0xE,
                     0xDD, 0xB4, 0x7,  0xB2, 0x44, 0xAF, 0x78, 0x29, 0xD2,
                     0x3C, 0xAE, 0x81, 0x58, 0x6D, 0x60, 0xD,  0xDE};
  uint8_t plaintext_8[1] = {};
  uint8_t ciphertext_8[] = {0x3E, 0xA0, 0xB5, 0x84, 0xF3, 0xC8, 0x5E, 0x93,
                            0xF9, 0x32, 0xE,  0xA5, 0x91, 0x69, 0x9E, 0xFB};
  vec = {nonce_8, aad_8, key_8, plaintext_8, ciphertext_8, 12, 70, 44, 0, 16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.1.2 54-byte auth
  uint8_t nonce_9[] = {0x12, 0x15, 0x35, 0x24, 0xC0, 0x89,
                       0x5E, 0x81, 0xB2, 0xC2, 0x84, 0x65};
  uint8_t aad_9[] = {0xD6, 0x9,  0xB1, 0xF0, 0x56, 0x63, 0x7A, 0xD,  0x46, 0xDF,
                     0x99, 0x8D, 0x88, 0xE5, 0x22, 0x2A, 0xB2, 0xC2, 0x84, 0x65,
                     0x12, 0x15, 0x35, 0x24, 0xC0, 0x89, 0x5E, 0x81, 0x8,  0x0,
                     0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                     0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22,
                     0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
                     0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x0,  0x1};
  uint8_t key_9[] = {0xE3, 0xC0, 0x8A, 0x8F, 0x6,  0xC6, 0xE3, 0xAD, 0x95,
                     0xA7, 0x5,  0x57, 0xB2, 0x3F, 0x75, 0x48, 0x3C, 0xE3,
                     0x30, 0x21, 0xA9, 0xC7, 0x2B, 0x70, 0x25, 0x66, 0x62,
                     0x4,  0xC6, 0x9C, 0xB,  0x72, 0xE1, 0xC2, 0x88, 0x8D,
                     0x4,  0xC4, 0xE1, 0xAF, 0x97, 0xA5, 0x7,  0x55};
  uint8_t plaintext_9[1] = {};
  uint8_t ciphertext_9[] = {0x29, 0x4E, 0x2,  0x8B, 0xF1, 0xFE, 0x6F, 0x14,
                            0xC4, 0xE8, 0xF7, 0x30, 0x5C, 0x93, 0x3E, 0xB5};
  vec = {nonce_9, aad_9, key_9, plaintext_9, ciphertext_9, 12, 70, 44, 0, 16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.2.1 60-byte crypt
  uint8_t nonce_10[] = {0x12, 0x15, 0x35, 0x24, 0xC0, 0x89,
                        0x5E, 0x81, 0xB2, 0xC2, 0x84, 0x65};
  uint8_t aad_10[] = {0xD6, 0x9,  0xB1, 0xF0, 0x56, 0x63, 0x7A,
                      0xD,  0x46, 0xDF, 0x99, 0x8D, 0x88, 0xE5,
                      0x2E, 0x0,  0xB2, 0xC2, 0x84, 0x65, 0x12,
                      0x15, 0x35, 0x24, 0xC0, 0x89, 0x5E, 0x81};
  uint8_t key_10[] = {0xAD, 0x7A, 0x2B, 0xD0, 0x3E, 0xAC, 0x83, 0x5A, 0x6F,
                      0x62, 0xF,  0xDC, 0xB5, 0x6,  0xB3, 0x45, 0xAC, 0x7B,
                      0x2A, 0xD1, 0x3F, 0xAD, 0x82, 0x5B, 0x6E, 0x63, 0xE,
                      0xDD, 0xB4, 0x7,  0xB2, 0x44, 0xAF, 0x78, 0x29, 0xD2,
                      0x3C, 0xAE, 0x81, 0x58, 0x6D, 0x60, 0xD,  0xDE};
  uint8_t plaintext_10[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
      0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30,
      0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x0,  0x2};
  uint8_t ciphertext_10[] = {
      0xDB, 0x3D, 0x25, 0x71, 0x9C, 0x6B, 0xA,  0x3C, 0xA6, 0x14, 0x5C,
      0x15, 0x9D, 0x5C, 0x6E, 0xD9, 0xAF, 0xF9, 0xC6, 0xE0, 0xB7, 0x9F,
      0x17, 0x1,  0x9E, 0xA9, 0x23, 0xB8, 0x66, 0x5D, 0xDF, 0x52, 0x13,
      0x7A, 0xD6, 0x11, 0xF0, 0xD1, 0xBF, 0x41, 0x7A, 0x7C, 0xA8, 0x5E,
      0x45, 0xAF, 0xE1, 0x6,  0xFF, 0x9C, 0x75, 0x69, 0xD3, 0x35, 0xD0,
      0x86, 0xAE, 0x6C, 0x3,  0xF0, 0x9,  0x87, 0xCC, 0xD6};
  vec = {nonce_10, aad_10, key_10, plaintext_10, ciphertext_10,
         12,       28,     44,     48,           64};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.2.2 60-byte crypt
  uint8_t nonce_11[] = {0x12, 0x15, 0x35, 0x24, 0xC0, 0x89,
                        0x5E, 0x81, 0xB2, 0xC2, 0x84, 0x65};
  uint8_t aad_11[] = {0xD6, 0x9,  0xB1, 0xF0, 0x56, 0x63, 0x7A,
                      0xD,  0x46, 0xDF, 0x99, 0x8D, 0x88, 0xE5,
                      0x2E, 0x0,  0xB2, 0xC2, 0x84, 0x65, 0x12,
                      0x15, 0x35, 0x24, 0xC0, 0x89, 0x5E, 0x81};
  uint8_t key_11[] = {0xE3, 0xC0, 0x8A, 0x8F, 0x6,  0xC6, 0xE3, 0xAD, 0x95,
                      0xA7, 0x5,  0x57, 0xB2, 0x3F, 0x75, 0x48, 0x3C, 0xE3,
                      0x30, 0x21, 0xA9, 0xC7, 0x2B, 0x70, 0x25, 0x66, 0x62,
                      0x4,  0xC6, 0x9C, 0xB,  0x72, 0xE1, 0xC2, 0x88, 0x8D,
                      0x4,  0xC4, 0xE1, 0xAF, 0x97, 0xA5, 0x7,  0x55};
  uint8_t plaintext_11[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
      0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30,
      0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x0,  0x2};
  uint8_t ciphertext_11[] = {
      0x16, 0x41, 0xF2, 0x8E, 0xC1, 0x3A, 0xFC, 0xC8, 0xF7, 0x90, 0x33,
      0x89, 0x78, 0x72, 0x1,  0x5,  0x16, 0x44, 0x91, 0x49, 0x33, 0xE9,
      0x20, 0x2B, 0xB9, 0xD0, 0x6A, 0xA0, 0x20, 0xC2, 0xA6, 0x7E, 0xF5,
      0x1D, 0xFE, 0x7B, 0xC0, 0xA,  0x85, 0x6C, 0x55, 0xB8, 0xF8, 0x13,
      0x3E, 0x77, 0xF6, 0x59, 0x13, 0x25, 0x2,  0xBA, 0xD6, 0x3F, 0x57,
      0x13, 0xD5, 0x7D, 0xC,  0x11, 0xE0, 0xF8, 0x71, 0xED};
  vec = {nonce_11, aad_11, key_11, plaintext_11, ciphertext_11,
         12,       28,     44,     48,           64};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.3.1 60-byte auth
  uint8_t nonce_12[] = {0xF0, 0x76, 0x1E, 0x8D, 0xCD, 0x3D,
                        0x0,  0x1,  0x76, 0xD4, 0x57, 0xED};
  uint8_t aad_12[] = {
      0xE2, 0x1,  0x6,  0xD7, 0xCD, 0xD,  0xF0, 0x76, 0x1E, 0x8D, 0xCD, 0x3D,
      0x88, 0xE5, 0x40, 0x0,  0x76, 0xD4, 0x57, 0xED, 0x8,  0x0,  0xF,  0x10,
      0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C,
      0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
      0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x0,  0x3};
  uint8_t key_12[] = {0x7,  0x1B, 0x11, 0x3B, 0xC,  0xA7, 0x43, 0xFE, 0xCC,
                      0xCF, 0x3D, 0x5,  0x1F, 0x73, 0x73, 0x82, 0x6,  0x1A,
                      0x10, 0x3A, 0xD,  0xA6, 0x42, 0xFF, 0xCD, 0xCE, 0x3C,
                      0x4,  0x1E, 0x72, 0x72, 0x83, 0x5,  0x19, 0x13, 0x39,
                      0xE,  0xA5, 0x41, 0xFC, 0xCE, 0xCD, 0x3F, 0x7};
  uint8_t plaintext_12[1] = {};
  uint8_t ciphertext_12[] = {0x58, 0x83, 0x7A, 0x10, 0x56, 0x2B, 0xF,  0x1F,
                             0x8E, 0xDB, 0xE5, 0x8C, 0xA5, 0x58, 0x11, 0xD3};
  vec = {nonce_12, aad_12, key_12, plaintext_12, ciphertext_12, 12, 68,
         44,       0,      16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.3.2 60-byte auth
  uint8_t nonce_13[] = {0xF0, 0x76, 0x1E, 0x8D, 0xCD, 0x3D,
                        0x0,  0x1,  0x76, 0xD4, 0x57, 0xED};
  uint8_t aad_13[] = {
      0xE2, 0x1,  0x6,  0xD7, 0xCD, 0xD,  0xF0, 0x76, 0x1E, 0x8D, 0xCD, 0x3D,
      0x88, 0xE5, 0x40, 0x0,  0x76, 0xD4, 0x57, 0xED, 0x8,  0x0,  0xF,  0x10,
      0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C,
      0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
      0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x0,  0x3};
  uint8_t key_13[] = {0x69, 0x1D, 0x3E, 0xE9, 0x9,  0xD7, 0xF5, 0x41, 0x67,
                      0xFD, 0x1C, 0xA0, 0xB5, 0xD7, 0x69, 0x8,  0x1F, 0x2B,
                      0xDE, 0x1A, 0xEE, 0x65, 0x5F, 0xDB, 0xAB, 0x80, 0xBD,
                      0x52, 0x95, 0xAE, 0x6B, 0xE7, 0x6B, 0x1F, 0x3C, 0xEB,
                      0xB,  0xD5, 0xF7, 0x43, 0x65, 0xFF, 0x1E, 0xA2};
  uint8_t plaintext_13[1] = {};
  uint8_t ciphertext_13[] = {0xC2, 0x72, 0x2F, 0xF6, 0xCA, 0x29, 0xA2, 0x57,
                             0x71, 0x8A, 0x52, 0x9D, 0x1F, 0xC,  0x6A, 0x3B};
  vec = {nonce_13, aad_13, key_13, plaintext_13, ciphertext_13, 12, 68,
         44,       0,      16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.4.1 54-byte crypt
  uint8_t nonce_14[] = {0xF0, 0x76, 0x1E, 0x8D, 0xCD, 0x3D,
                        0x0,  0x1,  0x76, 0xD4, 0x57, 0xED};
  uint8_t aad_14[] = {0xE2, 0x1,  0x6,  0xD7, 0xCD, 0xD,  0xF0,
                      0x76, 0x1E, 0x8D, 0xCD, 0x3D, 0x88, 0xE5,
                      0x4C, 0x2A, 0x76, 0xD4, 0x57, 0xED};
  uint8_t key_14[] = {0x7,  0x1B, 0x11, 0x3B, 0xC,  0xA7, 0x43, 0xFE, 0xCC,
                      0xCF, 0x3D, 0x5,  0x1F, 0x73, 0x73, 0x82, 0x6,  0x1A,
                      0x10, 0x3A, 0xD,  0xA6, 0x42, 0xFF, 0xCD, 0xCE, 0x3C,
                      0x4,  0x1E, 0x72, 0x72, 0x83, 0x5,  0x19, 0x13, 0x39,
                      0xE,  0xA5, 0x41, 0xFC, 0xCE, 0xCD, 0x3F, 0x7};
  uint8_t plaintext_14[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D,
      0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x0,  0x4};
  uint8_t ciphertext_14[] = {
      0xFD, 0x96, 0xB7, 0x15, 0xB9, 0x3A, 0x13, 0x34, 0x6A, 0xF5, 0x1E, 0x8A,
      0xCD, 0xF7, 0x92, 0xCD, 0xC7, 0xB2, 0x68, 0x6F, 0x85, 0x74, 0xC7, 0xE,
      0x6B, 0xC,  0xBF, 0x16, 0x29, 0x1D, 0xED, 0x42, 0x7A, 0xD7, 0x3F, 0xEC,
      0x48, 0xCD, 0x29, 0x8E, 0x5,  0x28, 0xA1, 0xF4, 0xC6, 0x44, 0xA9, 0x49,
      0xFC, 0x31, 0xDC, 0x92, 0x79, 0x70, 0x6D, 0xDB, 0xA3, 0x3F};
  vec = {nonce_14, aad_14, key_14, plaintext_14, ciphertext_14,
         12,       20,     44,     42,           58};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.4.2 54-byte crypt
  uint8_t nonce_15[] = {0xF0, 0x76, 0x1E, 0x8D, 0xCD, 0x3D,
                        0x0,  0x1,  0x76, 0xD4, 0x57, 0xED};
  uint8_t aad_15[] = {0xE2, 0x1,  0x6,  0xD7, 0xCD, 0xD,  0xF0,
                      0x76, 0x1E, 0x8D, 0xCD, 0x3D, 0x88, 0xE5,
                      0x4C, 0x2A, 0x76, 0xD4, 0x57, 0xED};
  uint8_t key_15[] = {0x69, 0x1D, 0x3E, 0xE9, 0x9,  0xD7, 0xF5, 0x41, 0x67,
                      0xFD, 0x1C, 0xA0, 0xB5, 0xD7, 0x69, 0x8,  0x1F, 0x2B,
                      0xDE, 0x1A, 0xEE, 0x65, 0x5F, 0xDB, 0xAB, 0x80, 0xBD,
                      0x52, 0x95, 0xAE, 0x6B, 0xE7, 0x6B, 0x1F, 0x3C, 0xEB,
                      0xB,  0xD5, 0xF7, 0x43, 0x65, 0xFF, 0x1E, 0xA2};
  uint8_t plaintext_15[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D,
      0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x0,  0x4};
  uint8_t ciphertext_15[] = {
      0xB6, 0x8F, 0x63, 0x0,  0xC2, 0xE9, 0xAE, 0x83, 0x3B, 0xDC, 0x7,  0xE,
      0x24, 0x2,  0x1A, 0x34, 0x77, 0x11, 0x8E, 0x78, 0xCC, 0xF8, 0x4E, 0x11,
      0xA4, 0x85, 0xD8, 0x61, 0x47, 0x6C, 0x30, 0xF,  0x17, 0x53, 0x53, 0xD5,
      0xCD, 0xF9, 0x20, 0x8,  0xA4, 0xF8, 0x78, 0xE6, 0xCC, 0x35, 0x77, 0x76,
      0x80, 0x85, 0xC5, 0xA,  0xE,  0x98, 0xFD, 0xA6, 0xCB, 0xB8};
  vec = {nonce_15, aad_15, key_15, plaintext_15, ciphertext_15,
         12,       20,     44,     42,           58};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.5.1 65-byte auth
  uint8_t nonce_16[] = {0x7C, 0xFD, 0xE9, 0xF9, 0xE3, 0x37,
                        0x24, 0xC6, 0x89, 0x32, 0xD6, 0x12};
  uint8_t aad_16[] = {
      0x84, 0xC5, 0xD5, 0x13, 0xD2, 0xAA, 0xF6, 0xE5, 0xBB, 0xD2, 0x72, 0x77,
      0x88, 0xE5, 0x23, 0x0,  0x89, 0x32, 0xD6, 0x12, 0x7C, 0xFD, 0xE9, 0xF9,
      0xE3, 0x37, 0x24, 0xC6, 0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14,
      0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
      0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x0,  0x5};
  uint8_t key_16[] = {0x1,  0x3F, 0xE0, 0xB,  0x5F, 0x11, 0xBE, 0x7F, 0x86,
                      0x6D, 0xC,  0xBB, 0xC5, 0x5A, 0x7A, 0x90, 0x0,  0x3E,
                      0xE1, 0xA,  0x5E, 0x10, 0xBF, 0x7E, 0x87, 0x6C, 0xD,
                      0xBA, 0xC4, 0x5B, 0x7B, 0x91, 0x3,  0x3D, 0xE2, 0x9,
                      0x5D, 0x13, 0xBC, 0x7D, 0x84, 0x6F, 0xE,  0xB9};
  uint8_t plaintext_16[1] = {};
  uint8_t ciphertext_16[] = {0xCC, 0xA2, 0xE,  0xEC, 0xDA, 0x62, 0x83, 0xF0,
                             0x9B, 0xB3, 0x54, 0x3D, 0xD9, 0x9E, 0xDB, 0x9B};
  vec = {nonce_16, aad_16, key_16, plaintext_16, ciphertext_16, 12, 81,
         44,       0,      16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.5.2 65-byte auth
  uint8_t nonce_17[] = {0x7C, 0xFD, 0xE9, 0xF9, 0xE3, 0x37,
                        0x24, 0xC6, 0x89, 0x32, 0xD6, 0x12};
  uint8_t aad_17[] = {
      0x84, 0xC5, 0xD5, 0x13, 0xD2, 0xAA, 0xF6, 0xE5, 0xBB, 0xD2, 0x72, 0x77,
      0x88, 0xE5, 0x23, 0x0,  0x89, 0x32, 0xD6, 0x12, 0x7C, 0xFD, 0xE9, 0xF9,
      0xE3, 0x37, 0x24, 0xC6, 0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14,
      0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
      0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x0,  0x5};
  uint8_t key_17[] = {0x83, 0xC0, 0x93, 0xB5, 0x8D, 0xE7, 0xFF, 0xE1, 0xC0,
                      0xDA, 0x92, 0x6A, 0xC4, 0x3F, 0xB3, 0x60, 0x9A, 0xC1,
                      0xC8, 0xF,  0xEE, 0x1B, 0x62, 0x44, 0x97, 0xEF, 0x94,
                      0x2E, 0x2F, 0x79, 0xA8, 0x23, 0x81, 0xC2, 0x91, 0xB7,
                      0x8F, 0xE5, 0xFD, 0xE3, 0xC2, 0xD8, 0x90, 0x68};
  uint8_t plaintext_17[1] = {};
  uint8_t ciphertext_17[] = {0xB2, 0x32, 0xCC, 0x1D, 0xA5, 0x11, 0x7B, 0xF1,
                             0x50, 0x3,  0x73, 0x4F, 0xA5, 0x99, 0xD2, 0x71};
  vec = {nonce_17, aad_17, key_17, plaintext_17, ciphertext_17, 12, 81,
         44,       0,      16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE  2.6.1 61-byte crypt
  uint8_t nonce_18[] = {0x7C, 0xFD, 0xE9, 0xF9, 0xE3, 0x37,
                        0x24, 0xC6, 0x89, 0x32, 0xD6, 0x12};
  uint8_t aad_18[] = {0x84, 0xC5, 0xD5, 0x13, 0xD2, 0xAA, 0xF6,
                      0xE5, 0xBB, 0xD2, 0x72, 0x77, 0x88, 0xE5,
                      0x2F, 0x0,  0x89, 0x32, 0xD6, 0x12, 0x7C,
                      0xFD, 0xE9, 0xF9, 0xE3, 0x37, 0x24, 0xC6};
  uint8_t key_18[] = {0x1,  0x3F, 0xE0, 0xB,  0x5F, 0x11, 0xBE, 0x7F, 0x86,
                      0x6D, 0xC,  0xBB, 0xC5, 0x5A, 0x7A, 0x90, 0x0,  0x3E,
                      0xE1, 0xA,  0x5E, 0x10, 0xBF, 0x7E, 0x87, 0x6C, 0xD,
                      0xBA, 0xC4, 0x5B, 0x7B, 0x91, 0x3,  0x3D, 0xE2, 0x9,
                      0x5D, 0x13, 0xBC, 0x7D, 0x84, 0x6F, 0xE,  0xB9};
  uint8_t plaintext_18[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
      0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A,
      0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x0,  0x6};
  uint8_t ciphertext_18[] = {
      0xFF, 0x19, 0x10, 0xD3, 0x5A, 0xD7, 0xE5, 0x65, 0x78, 0x90, 0xC7,
      0xC5, 0x60, 0x14, 0x6F, 0xD0, 0x38, 0x70, 0x7F, 0x20, 0x4B, 0x66,
      0xED, 0xBC, 0x3D, 0x16, 0x1F, 0x8A, 0xCE, 0x24, 0x4B, 0x98, 0x59,
      0x21, 0x2,  0x3C, 0x43, 0x6E, 0x3A, 0x1C, 0x35, 0x32, 0xEC, 0xD5,
      0xD0, 0x9A, 0x5,  0x6D, 0x70, 0xBE, 0x58, 0x3F, 0xD,  0x10, 0x82,
      0x9D, 0x93, 0x87, 0xD0, 0x7D, 0x33, 0xD8, 0x72, 0xE4, 0x90};
  vec = {nonce_18, aad_18, key_18, plaintext_18, ciphertext_18,
         12,       28,     44,     49,           65};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.6.2 61-byte crypt
  uint8_t nonce_19[] = {0x7C, 0xFD, 0xE9, 0xF9, 0xE3, 0x37,
                        0x24, 0xC6, 0x89, 0x32, 0xD6, 0x12};
  uint8_t aad_19[] = {0x84, 0xC5, 0xD5, 0x13, 0xD2, 0xAA, 0xF6,
                      0xE5, 0xBB, 0xD2, 0x72, 0x77, 0x88, 0xE5,
                      0x2F, 0x0,  0x89, 0x32, 0xD6, 0x12, 0x7C,
                      0xFD, 0xE9, 0xF9, 0xE3, 0x37, 0x24, 0xC6};
  uint8_t key_19[] = {0x83, 0xC0, 0x93, 0xB5, 0x8D, 0xE7, 0xFF, 0xE1, 0xC0,
                      0xDA, 0x92, 0x6A, 0xC4, 0x3F, 0xB3, 0x60, 0x9A, 0xC1,
                      0xC8, 0xF,  0xEE, 0x1B, 0x62, 0x44, 0x97, 0xEF, 0x94,
                      0x2E, 0x2F, 0x79, 0xA8, 0x23, 0x81, 0xC2, 0x91, 0xB7,
                      0x8F, 0xE5, 0xFD, 0xE3, 0xC2, 0xD8, 0x90, 0x68};
  uint8_t plaintext_19[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
      0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A,
      0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x0,  0x6};
  uint8_t ciphertext_19[] = {
      0xD,  0xB4, 0xCF, 0x95, 0x6B, 0x5F, 0x97, 0xEC, 0xA4, 0xEA, 0xB8,
      0x2A, 0x69, 0x55, 0x30, 0x7F, 0x9A, 0xE0, 0x2A, 0x32, 0xDD, 0x7D,
      0x93, 0xF8, 0x3D, 0x66, 0xAD, 0x4,  0xE1, 0xCF, 0xDC, 0x51, 0x82,
      0xAD, 0x12, 0xAB, 0xDE, 0xA5, 0xBB, 0xB6, 0x19, 0xA1, 0xBD, 0x5F,
      0xB9, 0xA5, 0x73, 0x59, 0xF,  0xBA, 0x90, 0x8E, 0x9C, 0x7A, 0x46,
      0xC1, 0xF7, 0xBA, 0x9,  0x5,  0xD1, 0xB5, 0x5F, 0xFD, 0xA4};
  vec = {nonce_19, aad_19, key_19, plaintext_19, ciphertext_19,
         12,       28,     44,     49,           65};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.7.1 79-byte crypt
  uint8_t nonce_20[] = {0x7A, 0xE8, 0xE2, 0xCA, 0x4E, 0xC5,
                        0x0,  0x1,  0x2E, 0x58, 0x49, 0x5C};
  uint8_t aad_20[] = {
      0x68, 0xF2, 0xE7, 0x76, 0x96, 0xCE, 0x7A, 0xE8, 0xE2, 0xCA, 0x4E,
      0xC5, 0x88, 0xE5, 0x41, 0x0,  0x2E, 0x58, 0x49, 0x5C, 0x8,  0x0,
      0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
      0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A,
      0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45,
      0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x0,  0x7};
  uint8_t key_20[] = {0x88, 0xEE, 0x8,  0x7F, 0xD9, 0x5D, 0xA9, 0xFB, 0xF6,
                      0x72, 0x5A, 0xA9, 0xD7, 0x57, 0xB0, 0xCD, 0x89, 0xEF,
                      0x9,  0x7E, 0xD8, 0x5C, 0xA8, 0xFA, 0xF7, 0x73, 0x5B,
                      0xA8, 0xD6, 0x56, 0xB1, 0xCC, 0x8A, 0xEC, 0xA,  0x7D,
                      0xDB, 0x5F, 0xAB, 0xF9, 0xF4, 0x70, 0x58, 0xAB};
  uint8_t plaintext_20[1] = {};
  uint8_t ciphertext_20[] = {0x81, 0x3F, 0xE,  0x63, 0xF,  0x96, 0xFB, 0x2D,
                             0x3,  0xF,  0x58, 0xD8, 0x3F, 0x5C, 0xDF, 0xD0};
  vec = {nonce_20, aad_20, key_20, plaintext_20, ciphertext_20, 12, 87,
         44,       0,      16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.7.2 79-byte crypt
  uint8_t nonce_21[] = {0x7A, 0xE8, 0xE2, 0xCA, 0x4E, 0xC5,
                        0x0,  0x1,  0x2E, 0x58, 0x49, 0x5C};
  uint8_t aad_21[] = {
      0x68, 0xF2, 0xE7, 0x76, 0x96, 0xCE, 0x7A, 0xE8, 0xE2, 0xCA, 0x4E,
      0xC5, 0x88, 0xE5, 0x41, 0x0,  0x2E, 0x58, 0x49, 0x5C, 0x8,  0x0,
      0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
      0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A,
      0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45,
      0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x0,  0x7};
  uint8_t key_21[] = {0x4C, 0x97, 0x3D, 0xBC, 0x73, 0x64, 0x62, 0x16, 0x74,
                      0xF8, 0xB5, 0xB8, 0x9E, 0x5C, 0x15, 0x51, 0x1F, 0xCE,
                      0xD9, 0x21, 0x64, 0x90, 0xFB, 0x1C, 0x1A, 0x2C, 0xAA,
                      0xF,  0xFE, 0x4,  0x7,  0xE5, 0x4E, 0x95, 0x3F, 0xBE,
                      0x71, 0x66, 0x60, 0x14, 0x76, 0xFA, 0xB7, 0xBA};
  uint8_t plaintext_21[1] = {};
  uint8_t ciphertext_21[] = {0x77, 0xE5, 0xA4, 0x4C, 0x21, 0xEB, 0x7, 0x18,
                             0x8A, 0xAC, 0xBD, 0x74, 0xD1, 0x98, 0xE, 0x97};
  vec = {nonce_21, aad_21, key_21, plaintext_21, ciphertext_21, 12, 87,
         44,       0,      16};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.8.1 61-byte crypt
  uint8_t nonce_22[] = {0x7A, 0xE8, 0xE2, 0xCA, 0x4E, 0xC5,
                        0x0,  0x1,  0x2E, 0x58, 0x49, 0x5C};
  uint8_t aad_22[] = {0x68, 0xF2, 0xE7, 0x76, 0x96, 0xCE, 0x7A,
                      0xE8, 0xE2, 0xCA, 0x4E, 0xC5, 0x88, 0xE5,
                      0x4D, 0x0,  0x2E, 0x58, 0x49, 0x5C};
  uint8_t key_22[] = {0x88, 0xEE, 0x8,  0x7F, 0xD9, 0x5D, 0xA9, 0xFB, 0xF6,
                      0x72, 0x5A, 0xA9, 0xD7, 0x57, 0xB0, 0xCD, 0x89, 0xEF,
                      0x9,  0x7E, 0xD8, 0x5C, 0xA8, 0xFA, 0xF7, 0x73, 0x5B,
                      0xA8, 0xD6, 0x56, 0xB1, 0xCC, 0x8A, 0xEC, 0xA,  0x7D,
                      0xDB, 0x5F, 0xAB, 0xF9, 0xF4, 0x70, 0x58, 0xAB};
  uint8_t plaintext_22[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D,
      0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43,
      0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x0,  0x8};
  uint8_t ciphertext_22[] = {
      0x95, 0x8E, 0xC3, 0xF6, 0xD6, 0xA,  0xFE, 0xDA, 0x99, 0xEF, 0xD8, 0x88,
      0xF1, 0x75, 0xE5, 0xFC, 0xD4, 0xC8, 0x7B, 0x9B, 0xCC, 0x5C, 0x2F, 0x54,
      0x26, 0x25, 0x3A, 0x8B, 0x50, 0x62, 0x96, 0xC8, 0xC4, 0x33, 0x9,  0xAB,
      0x2A, 0xDB, 0x59, 0x39, 0x46, 0x25, 0x41, 0xD9, 0x5E, 0x80, 0x81, 0x1E,
      0x4,  0xE7, 0x6,  0xB1, 0x49, 0x8F, 0x2C, 0x40, 0x7C, 0x7F, 0xB2, 0x34,
      0xF8, 0xCC, 0x1,  0xA6, 0x47, 0x55, 0xE,  0xE6, 0xB5, 0x57, 0xB3, 0x5A,
      0x7E, 0x39, 0x45, 0x38, 0x18, 0x21, 0xF4};
  vec = {nonce_22, aad_22, key_22, plaintext_22, ciphertext_22,
         12,       20,     44,     63,           79};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);

  // Derived from IEEE 2.8.2 61-byte crypt
  uint8_t nonce_23[] = {0x7A, 0xE8, 0xE2, 0xCA, 0x4E, 0xC5,
                        0x0,  0x1,  0x2E, 0x58, 0x49, 0x5C};
  uint8_t aad_23[] = {0x68, 0xF2, 0xE7, 0x76, 0x96, 0xCE, 0x7A,
                      0xE8, 0xE2, 0xCA, 0x4E, 0xC5, 0x88, 0xE5,
                      0x4D, 0x0,  0x2E, 0x58, 0x49, 0x5C};
  uint8_t key_23[] = {0x4C, 0x97, 0x3D, 0xBC, 0x73, 0x64, 0x62, 0x16, 0x74,
                      0xF8, 0xB5, 0xB8, 0x9E, 0x5C, 0x15, 0x51, 0x1F, 0xCE,
                      0xD9, 0x21, 0x64, 0x90, 0xFB, 0x1C, 0x1A, 0x2C, 0xAA,
                      0xF,  0xFE, 0x4,  0x7,  0xE5, 0x4E, 0x95, 0x3F, 0xBE,
                      0x71, 0x66, 0x60, 0x14, 0x76, 0xFA, 0xB7, 0xBA};
  uint8_t plaintext_23[] = {
      0x8,  0x0,  0xF,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D,
      0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43,
      0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x0,  0x8};
  uint8_t ciphertext_23[] = {
      0xB4, 0x4D, 0x7,  0x20, 0x11, 0xCD, 0x36, 0xD2, 0x72, 0xA9, 0xB7, 0xA9,
      0x8D, 0xB9, 0xAA, 0x90, 0xCB, 0xC5, 0xC6, 0x7B, 0x93, 0xDD, 0xCE, 0x67,
      0xC8, 0x54, 0x50, 0x32, 0x14, 0xE2, 0xE8, 0x96, 0xEC, 0x7E, 0x9D, 0xB6,
      0x49, 0xED, 0x4B, 0xCF, 0x6F, 0x85, 0xA,  0xAC, 0x2,  0x23, 0xD0, 0xCF,
      0x92, 0xC8, 0x3D, 0xB8, 0x7,  0x95, 0xC3, 0xA1, 0x7E, 0xCC, 0x12, 0x48,
      0xBB, 0x0,  0x59, 0x17, 0x12, 0xB1, 0xAE, 0x71, 0xE2, 0x68, 0x16, 0x41,
      0x96, 0x25, 0x21, 0x62, 0x81, 0xB,  0x0};
  vec = {nonce_23, aad_23, key_23, plaintext_23, ciphertext_23,
         12,       20,     44,     63,           79};
  gsec_test_verify_crypter_on_test_vector(&vec, /*rekey=*/true);
}

TEST(AltsCryptTest, GsecTestDoVectorTestsNist) {
  ///
  /// From:
  /// http://csrc.nist.gov/groups/ST/toolkit/BCM/documents/proposedmodes/gcm/
  /// gcm-revised-spec.pdf
  ///

  // Test vector 1
  gsec_aead_test_vector* test_vector_1;
  const uint8_t test_vector_1_key[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                       0x00, 0x00, 0x00, 0x00};
  const uint8_t test_vector_1_nonce[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                         0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  const uint8_t test_vector_1_aad[1] = {};
  const uint8_t test_vector_1_plaintext[1] = {};
  const uint8_t test_vector_1_ciphertext_and_tag[] = {
      0x58, 0xe2, 0xfc, 0xce, 0xfa, 0x7e, 0x30, 0x61,
      0x36, 0x7f, 0x1d, 0x57, 0xa4, 0xe7, 0x45, 0x5a};
  gsec_aead_malloc_test_vector(
      &test_vector_1, test_vector_1_key,
      sizeof(test_vector_1_key) / sizeof(uint8_t), test_vector_1_nonce,
      sizeof(test_vector_1_nonce) / sizeof(uint8_t), test_vector_1_aad, 0,
      test_vector_1_plaintext, 0, test_vector_1_ciphertext_and_tag,
      sizeof(test_vector_1_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_1);
  gsec_aead_free_test_vector(test_vector_1);

  // Test vector 2
  gsec_aead_test_vector* test_vector_2;
  const uint8_t test_vector_2_key[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                       0x00, 0x00, 0x00, 0x00};
  const uint8_t test_vector_2_nonce[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                         0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  const uint8_t test_vector_2_aad[1] = {};
  const uint8_t test_vector_2_plaintext[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                             0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                             0x00, 0x00, 0x00, 0x00};
  const uint8_t test_vector_2_ciphertext_and_tag[] = {
      0x03, 0x88, 0xda, 0xce, 0x60, 0xb6, 0xa3, 0x92, 0xf3, 0x28, 0xc2,
      0xb9, 0x71, 0xb2, 0xfe, 0x78, 0xab, 0x6e, 0x47, 0xd4, 0x2c, 0xec,
      0x13, 0xbd, 0xf5, 0x3a, 0x67, 0xb2, 0x12, 0x57, 0xbd, 0xdf};
  gsec_aead_malloc_test_vector(
      &test_vector_2, test_vector_2_key,
      sizeof(test_vector_2_key) / sizeof(uint8_t), test_vector_2_nonce,
      sizeof(test_vector_2_nonce) / sizeof(uint8_t), test_vector_2_aad, 0,
      test_vector_2_plaintext,
      sizeof(test_vector_2_plaintext) / sizeof(uint8_t),
      test_vector_2_ciphertext_and_tag,
      sizeof(test_vector_2_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_2);
  gsec_aead_free_test_vector(test_vector_2);

  // Test vector 3
  gsec_aead_test_vector* test_vector_3;
  const uint8_t test_vector_3_key[] = {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65,
                                       0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94,
                                       0x67, 0x30, 0x83, 0x08};
  const uint8_t test_vector_3_nonce[] = {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce,
                                         0xdb, 0xad, 0xde, 0xca, 0xf8, 0x88};
  const uint8_t test_vector_3_aad[1] = {};
  const uint8_t test_vector_3_plaintext[] = {
      0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09,
      0xc5, 0xaf, 0xf5, 0x26, 0x9a, 0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34,
      0xf7, 0xda, 0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72, 0x1c,
      0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24,
      0x49, 0xa6, 0xb5, 0x25, 0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6,
      0x57, 0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55};
  const uint8_t test_vector_3_ciphertext_and_tag[] = {
      0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24, 0x4b, 0x72, 0x21, 0xb7,
      0x84, 0xd0, 0xd4, 0x9c, 0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
      0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e, 0x21, 0xd5, 0x14, 0xb2,
      0x54, 0x66, 0x93, 0x1c, 0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
      0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97, 0x3d, 0x58, 0xe0, 0x91,
      0x47, 0x3f, 0x59, 0x85, 0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6,
      0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4};
  gsec_aead_malloc_test_vector(
      &test_vector_3, test_vector_3_key,
      sizeof(test_vector_3_key) / sizeof(uint8_t), test_vector_3_nonce,
      sizeof(test_vector_3_nonce) / sizeof(uint8_t), test_vector_3_aad, 0,
      test_vector_3_plaintext,
      sizeof(test_vector_3_plaintext) / sizeof(uint8_t),
      test_vector_3_ciphertext_and_tag,
      sizeof(test_vector_3_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_3);
  gsec_aead_free_test_vector(test_vector_3);

  // Test vector 4
  gsec_aead_test_vector* test_vector_4;
  const uint8_t test_vector_4_key[] = {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65,
                                       0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94,
                                       0x67, 0x30, 0x83, 0x08};
  const uint8_t test_vector_4_nonce[] = {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce,
                                         0xdb, 0xad, 0xde, 0xca, 0xf8, 0x88};
  const uint8_t test_vector_4_aad[] = {0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe,
                                       0xef, 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad,
                                       0xbe, 0xef, 0xab, 0xad, 0xda, 0xd2};
  const uint8_t test_vector_4_plaintext[] = {
      0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5,
      0xaf, 0xf5, 0x26, 0x9a, 0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
      0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72, 0x1c, 0x3c, 0x0c, 0x95,
      0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
      0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39};
  const uint8_t test_vector_4_ciphertext_and_tag[] = {
      0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24, 0x4b, 0x72, 0x21,
      0xb7, 0x84, 0xd0, 0xd4, 0x9c, 0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02,
      0xa4, 0xe0, 0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e, 0x21,
      0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c, 0x7d, 0x8f, 0x6a, 0x5a,
      0xac, 0x84, 0xaa, 0x05, 0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac,
      0x97, 0x3d, 0x58, 0xe0, 0x91, 0x5b, 0xc9, 0x4f, 0xbc, 0x32, 0x21,
      0xa5, 0xdb, 0x94, 0xfa, 0xe9, 0x5a, 0xe7, 0x12, 0x1a, 0x47};
  gsec_aead_malloc_test_vector(
      &test_vector_4, test_vector_4_key,
      sizeof(test_vector_4_key) / sizeof(uint8_t), test_vector_4_nonce,
      sizeof(test_vector_4_nonce) / sizeof(uint8_t), test_vector_4_aad,
      sizeof(test_vector_4_aad) / sizeof(uint8_t), test_vector_4_plaintext,
      sizeof(test_vector_4_plaintext) / sizeof(uint8_t),
      test_vector_4_ciphertext_and_tag,
      sizeof(test_vector_4_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_4);
  gsec_aead_free_test_vector(test_vector_4);
}

TEST(AltsCryptTest, GsecTestDoVectorTestsIeee) {
  ///
  /// From:
  /// http://www.ieee802.org/1/files/public/docs2011/
  /// bn-randall-test-vectors-0511-v1.pdf
  ///

  // 2.1.1 54-byte auth
  gsec_aead_test_vector* test_vector_5;
  const uint8_t test_vector_5_key[] = {0xad, 0x7a, 0x2b, 0xd0, 0x3e, 0xac,
                                       0x83, 0x5a, 0x6f, 0x62, 0x0f, 0xdc,
                                       0xb5, 0x06, 0xb3, 0x45};
  const uint8_t test_vector_5_nonce[] = {0x12, 0x15, 0x35, 0x24, 0xc0, 0x89,
                                         0x5e, 0x81, 0xb2, 0xc2, 0x84, 0x65};
  const uint8_t test_vector_5_aad[] = {
      0xd6, 0x09, 0xb1, 0xf0, 0x56, 0x63, 0x7a, 0x0d, 0x46, 0xdf, 0x99, 0x8d,
      0x88, 0xe5, 0x22, 0x2a, 0xb2, 0xc2, 0x84, 0x65, 0x12, 0x15, 0x35, 0x24,
      0xc0, 0x89, 0x5e, 0x81, 0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
      0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c,
      0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x00, 0x01};
  const uint8_t test_vector_5_plaintext[1] = {};
  const uint8_t test_vector_5_ciphertext_and_tag[] = {
      0xf0, 0x94, 0x78, 0xa9, 0xb0, 0x90, 0x07, 0xd0,
      0x6f, 0x46, 0xe9, 0xb6, 0xa1, 0xda, 0x25, 0xdd};
  gsec_aead_malloc_test_vector(
      &test_vector_5, test_vector_5_key,
      sizeof(test_vector_5_key) / sizeof(uint8_t), test_vector_5_nonce,
      sizeof(test_vector_5_nonce) / sizeof(uint8_t), test_vector_5_aad,
      sizeof(test_vector_5_aad) / sizeof(uint8_t), test_vector_5_plaintext, 0,
      test_vector_5_ciphertext_and_tag,
      sizeof(test_vector_5_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_5);
  gsec_aead_free_test_vector(test_vector_5);

  // 2.1.2 54-byte auth
  gsec_aead_test_vector* test_vector_6;
  const uint8_t test_vector_6_key[] = {
      0xe3, 0xc0, 0x8a, 0x8f, 0x06, 0xc6, 0xe3, 0xad, 0x95, 0xa7, 0x05,
      0x57, 0xb2, 0x3f, 0x75, 0x48, 0x3c, 0xe3, 0x30, 0x21, 0xa9, 0xc7,
      0x2b, 0x70, 0x25, 0x66, 0x62, 0x04, 0xc6, 0x9c, 0x0b, 0x72};

  const uint8_t test_vector_6_nonce[] = {0x12, 0x15, 0x35, 0x24, 0xc0, 0x89,
                                         0x5e, 0x81, 0xb2, 0xc2, 0x84, 0x65};
  const uint8_t test_vector_6_aad[] = {
      0xd6, 0x09, 0xb1, 0xf0, 0x56, 0x63, 0x7a, 0x0d, 0x46, 0xdf, 0x99, 0x8d,
      0x88, 0xe5, 0x22, 0x2a, 0xb2, 0xc2, 0x84, 0x65, 0x12, 0x15, 0x35, 0x24,
      0xc0, 0x89, 0x5e, 0x81, 0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
      0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c,
      0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x00, 0x01};
  const uint8_t test_vector_6_plaintext[1] = {};
  const uint8_t test_vector_6_ciphertext_and_tag[] = {
      0x2f, 0x0b, 0xc5, 0xaf, 0x40, 0x9e, 0x06, 0xd6,
      0x09, 0xea, 0x8b, 0x7d, 0x0f, 0xa5, 0xea, 0x50};
  gsec_aead_malloc_test_vector(
      &test_vector_6, test_vector_6_key,
      sizeof(test_vector_6_key) / sizeof(uint8_t), test_vector_6_nonce,
      sizeof(test_vector_6_nonce) / sizeof(uint8_t), test_vector_6_aad,
      sizeof(test_vector_6_aad) / sizeof(uint8_t), test_vector_6_plaintext, 0,
      test_vector_6_ciphertext_and_tag,
      sizeof(test_vector_6_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_6);
  gsec_aead_free_test_vector(test_vector_6);

  // 2.2.1 60-byte crypt
  gsec_aead_test_vector* test_vector_7;
  const uint8_t test_vector_7_key[] = {0xad, 0x7a, 0x2b, 0xd0, 0x3e, 0xac,
                                       0x83, 0x5a, 0x6f, 0x62, 0x0f, 0xdc,
                                       0xb5, 0x06, 0xb3, 0x45};

  const uint8_t test_vector_7_nonce[] = {0x12, 0x15, 0x35, 0x24, 0xc0, 0x89,
                                         0x5e, 0x81, 0xb2, 0xc2, 0x84, 0x65};
  const uint8_t test_vector_7_aad[] = {
      0xd6, 0x09, 0xb1, 0xf0, 0x56, 0x63, 0x7a, 0x0d, 0x46, 0xdf,
      0x99, 0x8d, 0x88, 0xe5, 0x2e, 0x00, 0xb2, 0xc2, 0x84, 0x65,
      0x12, 0x15, 0x35, 0x24, 0xc0, 0x89, 0x5e, 0x81};
  const uint8_t test_vector_7_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
      0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
      0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x00, 0x02};
  const uint8_t test_vector_7_ciphertext_and_tag[] = {
      0x70, 0x1a, 0xfa, 0x1c, 0xc0, 0x39, 0xc0, 0xd7, 0x65, 0x12, 0x8a,
      0x66, 0x5d, 0xab, 0x69, 0x24, 0x38, 0x99, 0xbf, 0x73, 0x18, 0xcc,
      0xdc, 0x81, 0xc9, 0x93, 0x1d, 0xa1, 0x7f, 0xbe, 0x8e, 0xdd, 0x7d,
      0x17, 0xcb, 0x8b, 0x4c, 0x26, 0xfc, 0x81, 0xe3, 0x28, 0x4f, 0x2b,
      0x7f, 0xba, 0x71, 0x3d, 0x4f, 0x8d, 0x55, 0xe7, 0xd3, 0xf0, 0x6f,
      0xd5, 0xa1, 0x3c, 0x0c, 0x29, 0xb9, 0xd5, 0xb8, 0x80};
  gsec_aead_malloc_test_vector(
      &test_vector_7, test_vector_7_key,
      sizeof(test_vector_7_key) / sizeof(uint8_t), test_vector_7_nonce,
      sizeof(test_vector_7_nonce) / sizeof(uint8_t), test_vector_7_aad,
      sizeof(test_vector_7_aad) / sizeof(uint8_t), test_vector_7_plaintext,
      sizeof(test_vector_7_plaintext) / sizeof(uint8_t),
      test_vector_7_ciphertext_and_tag,
      sizeof(test_vector_7_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_7);
  gsec_aead_free_test_vector(test_vector_7);

  // 2.2.2 60-byte crypt
  gsec_aead_test_vector* test_vector_8;
  const uint8_t test_vector_8_key[] = {
      0xe3, 0xc0, 0x8a, 0x8f, 0x06, 0xc6, 0xe3, 0xad, 0x95, 0xa7, 0x05,
      0x57, 0xb2, 0x3f, 0x75, 0x48, 0x3c, 0xe3, 0x30, 0x21, 0xa9, 0xc7,
      0x2b, 0x70, 0x25, 0x66, 0x62, 0x04, 0xc6, 0x9c, 0x0b, 0x72};
  const uint8_t test_vector_8_nonce[] = {0x12, 0x15, 0x35, 0x24, 0xc0, 0x89,
                                         0x5e, 0x81, 0xb2, 0xc2, 0x84, 0x65};
  const uint8_t test_vector_8_aad[] = {
      0xd6, 0x09, 0xb1, 0xf0, 0x56, 0x63, 0x7a, 0x0d, 0x46, 0xdf,
      0x99, 0x8d, 0x88, 0xe5, 0x2e, 0x00, 0xb2, 0xc2, 0x84, 0x65,
      0x12, 0x15, 0x35, 0x24, 0xc0, 0x89, 0x5e, 0x81};
  const uint8_t test_vector_8_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
      0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
      0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x00, 0x02};
  const uint8_t test_vector_8_ciphertext_and_tag[] = {
      0xe2, 0x00, 0x6e, 0xb4, 0x2f, 0x52, 0x77, 0x02, 0x2d, 0x9b, 0x19,
      0x92, 0x5b, 0xc4, 0x19, 0xd7, 0xa5, 0x92, 0x66, 0x6c, 0x92, 0x5f,
      0xe2, 0xef, 0x71, 0x8e, 0xb4, 0xe3, 0x08, 0xef, 0xea, 0xa7, 0xc5,
      0x27, 0x3b, 0x39, 0x41, 0x18, 0x86, 0x0a, 0x5b, 0xe2, 0xa9, 0x7f,
      0x56, 0xab, 0x78, 0x36, 0x5c, 0xa5, 0x97, 0xcd, 0xbb, 0x3e, 0xdb,
      0x8d, 0x1a, 0x11, 0x51, 0xea, 0x0a, 0xf7, 0xb4, 0x36};
  gsec_aead_malloc_test_vector(
      &test_vector_8, test_vector_8_key,
      sizeof(test_vector_8_key) / sizeof(uint8_t), test_vector_8_nonce,
      sizeof(test_vector_8_nonce) / sizeof(uint8_t), test_vector_8_aad,
      sizeof(test_vector_8_aad) / sizeof(uint8_t), test_vector_8_plaintext,
      sizeof(test_vector_8_plaintext) / sizeof(uint8_t),
      test_vector_8_ciphertext_and_tag,
      sizeof(test_vector_8_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_8);
  gsec_aead_free_test_vector(test_vector_8);

  // 2.3.1 60-byte auth
  gsec_aead_test_vector* test_vector_9;
  const uint8_t test_vector_9_key[] = {0x07, 0x1b, 0x11, 0x3b, 0x0c, 0xa7,
                                       0x43, 0xfe, 0xcc, 0xcf, 0x3d, 0x05,
                                       0x1f, 0x73, 0x73, 0x82};
  const uint8_t test_vector_9_nonce[] = {0xf0, 0x76, 0x1e, 0x8d, 0xcd, 0x3d,
                                         0x00, 0x01, 0x76, 0xd4, 0x57, 0xed};
  const uint8_t test_vector_9_aad[] = {
      0xe2, 0x01, 0x06, 0xd7, 0xcd, 0x0d, 0xf0, 0x76, 0x1e, 0x8d, 0xcd, 0x3d,
      0x88, 0xe5, 0x40, 0x00, 0x76, 0xd4, 0x57, 0xed, 0x08, 0x00, 0x0f, 0x10,
      0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c,
      0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
      0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x00, 0x03};
  const uint8_t test_vector_9_plaintext[1] = {};
  const uint8_t test_vector_9_ciphertext_and_tag[] = {
      0x0c, 0x01, 0x7b, 0xc7, 0x3b, 0x22, 0x7d, 0xfc,
      0xc9, 0xba, 0xfa, 0x1c, 0x41, 0xac, 0xc3, 0x53};
  gsec_aead_malloc_test_vector(
      &test_vector_9, test_vector_9_key,
      sizeof(test_vector_9_key) / sizeof(uint8_t), test_vector_9_nonce,
      sizeof(test_vector_9_nonce) / sizeof(uint8_t), test_vector_9_aad,
      sizeof(test_vector_9_aad) / sizeof(uint8_t), test_vector_9_plaintext, 0,
      test_vector_9_ciphertext_and_tag,
      sizeof(test_vector_9_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_9);
  gsec_aead_free_test_vector(test_vector_9);

  // 2.3.2 60-byte auth
  gsec_aead_test_vector* test_vector_10;
  const uint8_t test_vector_10_key[] = {
      0x69, 0x1d, 0x3e, 0xe9, 0x09, 0xd7, 0xf5, 0x41, 0x67, 0xfd, 0x1c,
      0xa0, 0xb5, 0xd7, 0x69, 0x08, 0x1f, 0x2b, 0xde, 0x1a, 0xee, 0x65,
      0x5f, 0xdb, 0xab, 0x80, 0xbd, 0x52, 0x95, 0xae, 0x6b, 0xe7};
  const uint8_t test_vector_10_nonce[] = {0xf0, 0x76, 0x1e, 0x8d, 0xcd, 0x3d,
                                          0x00, 0x01, 0x76, 0xd4, 0x57, 0xed};
  const uint8_t test_vector_10_aad[] = {
      0xe2, 0x01, 0x06, 0xd7, 0xcd, 0x0d, 0xf0, 0x76, 0x1e, 0x8d, 0xcd, 0x3d,
      0x88, 0xe5, 0x40, 0x00, 0x76, 0xd4, 0x57, 0xed, 0x08, 0x00, 0x0f, 0x10,
      0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c,
      0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
      0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x00, 0x03};
  const uint8_t test_vector_10_plaintext[1] = {};
  const uint8_t test_vector_10_ciphertext_and_tag[] = {
      0x35, 0x21, 0x7c, 0x77, 0x4b, 0xbc, 0x31, 0xb6,
      0x31, 0x66, 0xbc, 0xf9, 0xd4, 0xab, 0xed, 0x07};
  gsec_aead_malloc_test_vector(
      &test_vector_10, test_vector_10_key,
      sizeof(test_vector_10_key) / sizeof(uint8_t), test_vector_10_nonce,
      sizeof(test_vector_10_nonce) / sizeof(uint8_t), test_vector_10_aad,
      sizeof(test_vector_10_aad) / sizeof(uint8_t), test_vector_10_plaintext, 0,
      test_vector_10_ciphertext_and_tag,
      sizeof(test_vector_10_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_10);
  gsec_aead_free_test_vector(test_vector_10);

  // 2.4.1 54-byte crypt
  gsec_aead_test_vector* test_vector_11;
  const uint8_t test_vector_11_key[] = {0x07, 0x1b, 0x11, 0x3b, 0x0c, 0xa7,
                                        0x43, 0xfe, 0xcc, 0xcf, 0x3d, 0x05,
                                        0x1f, 0x73, 0x73, 0x82};
  const uint8_t test_vector_11_nonce[] = {0xf0, 0x76, 0x1e, 0x8d, 0xcd, 0x3d,
                                          0x00, 0x01, 0x76, 0xd4, 0x57, 0xed};
  const uint8_t test_vector_11_aad[] = {
      0xe2, 0x01, 0x06, 0xd7, 0xcd, 0x0d, 0xf0, 0x76, 0x1e, 0x8d,
      0xcd, 0x3d, 0x88, 0xe5, 0x4c, 0x2a, 0x76, 0xd4, 0x57, 0xed};
  const uint8_t test_vector_11_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
      0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x00, 0x04};
  const uint8_t test_vector_11_ciphertext_and_tag[] = {
      0x13, 0xb4, 0xc7, 0x2b, 0x38, 0x9d, 0xc5, 0x01, 0x8e, 0x72, 0xa1, 0x71,
      0xdd, 0x85, 0xa5, 0xd3, 0x75, 0x22, 0x74, 0xd3, 0xa0, 0x19, 0xfb, 0xca,
      0xed, 0x09, 0xa4, 0x25, 0xcd, 0x9b, 0x2e, 0x1c, 0x9b, 0x72, 0xee, 0xe7,
      0xc9, 0xde, 0x7d, 0x52, 0xb3, 0xf3, 0xd6, 0xa5, 0x28, 0x4f, 0x4a, 0x6d,
      0x3f, 0xe2, 0x2a, 0x5d, 0x6c, 0x2b, 0x96, 0x04, 0x94, 0xc3};
  gsec_aead_malloc_test_vector(
      &test_vector_11, test_vector_11_key,
      sizeof(test_vector_11_key) / sizeof(uint8_t), test_vector_11_nonce,
      sizeof(test_vector_11_nonce) / sizeof(uint8_t), test_vector_11_aad,
      sizeof(test_vector_11_aad) / sizeof(uint8_t), test_vector_11_plaintext,
      sizeof(test_vector_11_plaintext) / sizeof(uint8_t),
      test_vector_11_ciphertext_and_tag,
      sizeof(test_vector_11_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_11);
  gsec_aead_free_test_vector(test_vector_11);

  // 2.4.2 54-byte crypt
  gsec_aead_test_vector* test_vector_12;
  const uint8_t test_vector_12_key[] = {
      0x69, 0x1d, 0x3e, 0xe9, 0x09, 0xd7, 0xf5, 0x41, 0x67, 0xfd, 0x1c,
      0xa0, 0xb5, 0xd7, 0x69, 0x08, 0x1f, 0x2b, 0xde, 0x1a, 0xee, 0x65,
      0x5f, 0xdb, 0xab, 0x80, 0xbd, 0x52, 0x95, 0xae, 0x6b, 0xe7};
  const uint8_t test_vector_12_nonce[] = {0xf0, 0x76, 0x1e, 0x8d, 0xcd, 0x3d,
                                          0x00, 0x01, 0x76, 0xd4, 0x57, 0xed};
  const uint8_t test_vector_12_aad[] = {
      0xe2, 0x01, 0x06, 0xd7, 0xcd, 0x0d, 0xf0, 0x76, 0x1e, 0x8d,
      0xcd, 0x3d, 0x88, 0xe5, 0x4c, 0x2a, 0x76, 0xd4, 0x57, 0xed};
  const uint8_t test_vector_12_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
      0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x00, 0x04};
  const uint8_t test_vector_12_ciphertext_and_tag[] = {
      0xc1, 0x62, 0x3f, 0x55, 0x73, 0x0c, 0x93, 0x53, 0x30, 0x97, 0xad, 0xda,
      0xd2, 0x56, 0x64, 0x96, 0x61, 0x25, 0x35, 0x2b, 0x43, 0xad, 0xac, 0xbd,
      0x61, 0xc5, 0xef, 0x3a, 0xc9, 0x0b, 0x5b, 0xee, 0x92, 0x9c, 0xe4, 0x63,
      0x0e, 0xa7, 0x9f, 0x6c, 0xe5, 0x19, 0x12, 0xaf, 0x39, 0xc2, 0xd1, 0xfd,
      0xc2, 0x05, 0x1f, 0x8b, 0x7b, 0x3c, 0x9d, 0x39, 0x7e, 0xf2};
  gsec_aead_malloc_test_vector(
      &test_vector_12, test_vector_12_key,
      sizeof(test_vector_12_key) / sizeof(uint8_t), test_vector_12_nonce,
      sizeof(test_vector_12_nonce) / sizeof(uint8_t), test_vector_12_aad,
      sizeof(test_vector_12_aad) / sizeof(uint8_t), test_vector_12_plaintext,
      sizeof(test_vector_12_plaintext) / sizeof(uint8_t),
      test_vector_12_ciphertext_and_tag,
      sizeof(test_vector_12_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_12);
  gsec_aead_free_test_vector(test_vector_12);

  // 2.5.1 65-byte auth
  gsec_aead_test_vector* test_vector_13;
  const uint8_t test_vector_13_key[] = {0x01, 0x3f, 0xe0, 0x0b, 0x5f, 0x11,
                                        0xbe, 0x7f, 0x86, 0x6d, 0x0c, 0xbb,
                                        0xc5, 0x5a, 0x7a, 0x90};
  const uint8_t test_vector_13_nonce[] = {0x7c, 0xfd, 0xe9, 0xf9, 0xe3, 0x37,
                                          0x24, 0xc6, 0x89, 0x32, 0xd6, 0x12};
  const uint8_t test_vector_13_aad[] = {
      0x84, 0xc5, 0xd5, 0x13, 0xd2, 0xaa, 0xf6, 0xe5, 0xbb, 0xd2, 0x72, 0x77,
      0x88, 0xe5, 0x23, 0x00, 0x89, 0x32, 0xd6, 0x12, 0x7c, 0xfd, 0xe9, 0xf9,
      0xe3, 0x37, 0x24, 0xc6, 0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
      0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c,
      0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x00, 0x05};
  const uint8_t test_vector_13_plaintext[1] = {};
  const uint8_t test_vector_13_ciphertext_and_tag[] = {
      0x21, 0x78, 0x67, 0xe5, 0x0c, 0x2d, 0xad, 0x74,
      0xc2, 0x8c, 0x3b, 0x50, 0xab, 0xdf, 0x69, 0x5a};
  gsec_aead_malloc_test_vector(
      &test_vector_13, test_vector_13_key,
      sizeof(test_vector_13_key) / sizeof(uint8_t), test_vector_13_nonce,
      sizeof(test_vector_13_nonce) / sizeof(uint8_t), test_vector_13_aad,
      sizeof(test_vector_13_aad) / sizeof(uint8_t), test_vector_13_plaintext, 0,
      test_vector_13_ciphertext_and_tag,
      sizeof(test_vector_13_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_13);
  gsec_aead_free_test_vector(test_vector_13);

  // 2.5.2 65-byte auth
  gsec_aead_test_vector* test_vector_14;
  const uint8_t test_vector_14_key[] = {
      0x83, 0xc0, 0x93, 0xb5, 0x8d, 0xe7, 0xff, 0xe1, 0xc0, 0xda, 0x92,
      0x6a, 0xc4, 0x3f, 0xb3, 0x60, 0x9a, 0xc1, 0xc8, 0x0f, 0xee, 0x1b,
      0x62, 0x44, 0x97, 0xef, 0x94, 0x2e, 0x2f, 0x79, 0xa8, 0x23};
  const uint8_t test_vector_14_nonce[] = {0x7c, 0xfd, 0xe9, 0xf9, 0xe3, 0x37,
                                          0x24, 0xc6, 0x89, 0x32, 0xd6, 0x12};
  const uint8_t test_vector_14_aad[] = {
      0x84, 0xc5, 0xd5, 0x13, 0xd2, 0xaa, 0xf6, 0xe5, 0xbb, 0xd2, 0x72, 0x77,
      0x88, 0xe5, 0x23, 0x00, 0x89, 0x32, 0xd6, 0x12, 0x7c, 0xfd, 0xe9, 0xf9,
      0xe3, 0x37, 0x24, 0xc6, 0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14,
      0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c,
      0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x00, 0x05};
  const uint8_t test_vector_14_plaintext[1] = {};
  const uint8_t test_vector_14_ciphertext_and_tag[] = {
      0x6e, 0xe1, 0x60, 0xe8, 0xfa, 0xec, 0xa4, 0xb3,
      0x6c, 0x86, 0xb2, 0x34, 0x92, 0x0c, 0xa9, 0x75};
  gsec_aead_malloc_test_vector(
      &test_vector_14, test_vector_14_key,
      sizeof(test_vector_14_key) / sizeof(uint8_t), test_vector_14_nonce,
      sizeof(test_vector_14_nonce) / sizeof(uint8_t), test_vector_14_aad,
      sizeof(test_vector_14_aad) / sizeof(uint8_t), test_vector_14_plaintext, 0,
      test_vector_14_ciphertext_and_tag,
      sizeof(test_vector_14_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_14);
  gsec_aead_free_test_vector(test_vector_14);

  // 2.6.1 61-byte crypt
  gsec_aead_test_vector* test_vector_15;
  const uint8_t test_vector_15_key[] = {0x01, 0x3f, 0xe0, 0x0b, 0x5f, 0x11,
                                        0xbe, 0x7f, 0x86, 0x6d, 0x0c, 0xbb,
                                        0xc5, 0x5a, 0x7a, 0x90};
  const uint8_t test_vector_15_nonce[] = {0x7c, 0xfd, 0xe9, 0xf9, 0xe3, 0x37,
                                          0x24, 0xc6, 0x89, 0x32, 0xd6, 0x12};
  const uint8_t test_vector_15_aad[] = {
      0x84, 0xc5, 0xd5, 0x13, 0xd2, 0xaa, 0xf6, 0xe5, 0xbb, 0xd2,
      0x72, 0x77, 0x88, 0xe5, 0x2f, 0x00, 0x89, 0x32, 0xd6, 0x12,
      0x7c, 0xfd, 0xe9, 0xf9, 0xe3, 0x37, 0x24, 0xc6};
  const uint8_t test_vector_15_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
      0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a,
      0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x00, 0x06};
  const uint8_t test_vector_15_ciphertext_and_tag[] = {
      0x3a, 0x4d, 0xe6, 0xfa, 0x32, 0x19, 0x10, 0x14, 0xdb, 0xb3, 0x03,
      0xd9, 0x2e, 0xe3, 0xa9, 0xe8, 0xa1, 0xb5, 0x99, 0xc1, 0x4d, 0x22,
      0xfb, 0x08, 0x00, 0x96, 0xe1, 0x38, 0x11, 0x81, 0x6a, 0x3c, 0x9c,
      0x9b, 0xcf, 0x7c, 0x1b, 0x9b, 0x96, 0xda, 0x80, 0x92, 0x04, 0xe2,
      0x9d, 0x0e, 0x2a, 0x76, 0x42, 0xbf, 0xd3, 0x10, 0xa4, 0x83, 0x7c,
      0x81, 0x6c, 0xcf, 0xa5, 0xac, 0x23, 0xab, 0x00, 0x39, 0x88};
  gsec_aead_malloc_test_vector(
      &test_vector_15, test_vector_15_key,
      sizeof(test_vector_15_key) / sizeof(uint8_t), test_vector_15_nonce,
      sizeof(test_vector_15_nonce) / sizeof(uint8_t), test_vector_15_aad,
      sizeof(test_vector_15_aad) / sizeof(uint8_t), test_vector_15_plaintext,
      sizeof(test_vector_15_plaintext) / sizeof(uint8_t),
      test_vector_15_ciphertext_and_tag,
      sizeof(test_vector_15_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_15);
  gsec_aead_free_test_vector(test_vector_15);

  // 2.6.2 61-byte crypt
  gsec_aead_test_vector* test_vector_16;
  const uint8_t test_vector_16_key[] = {
      0x83, 0xc0, 0x93, 0xb5, 0x8d, 0xe7, 0xff, 0xe1, 0xc0, 0xda, 0x92,
      0x6a, 0xc4, 0x3f, 0xb3, 0x60, 0x9a, 0xc1, 0xc8, 0x0f, 0xee, 0x1b,
      0x62, 0x44, 0x97, 0xef, 0x94, 0x2e, 0x2f, 0x79, 0xa8, 0x23};
  const uint8_t test_vector_16_nonce[] = {0x7c, 0xfd, 0xe9, 0xf9, 0xe3, 0x37,
                                          0x24, 0xc6, 0x89, 0x32, 0xd6, 0x12};
  const uint8_t test_vector_16_aad[] = {
      0x84, 0xc5, 0xd5, 0x13, 0xd2, 0xaa, 0xf6, 0xe5, 0xbb, 0xd2,
      0x72, 0x77, 0x88, 0xe5, 0x2f, 0x00, 0x89, 0x32, 0xd6, 0x12,
      0x7c, 0xfd, 0xe9, 0xf9, 0xe3, 0x37, 0x24, 0xc6};
  const uint8_t test_vector_16_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
      0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
      0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a,
      0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34,
      0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x00, 0x06};
  const uint8_t test_vector_16_ciphertext_and_tag[] = {
      0x11, 0x02, 0x22, 0xff, 0x80, 0x50, 0xcb, 0xec, 0xe6, 0x6a, 0x81,
      0x3a, 0xd0, 0x9a, 0x73, 0xed, 0x7a, 0x9a, 0x08, 0x9c, 0x10, 0x6b,
      0x95, 0x93, 0x89, 0x16, 0x8e, 0xd6, 0xe8, 0x69, 0x8e, 0xa9, 0x02,
      0xeb, 0x12, 0x77, 0xdb, 0xec, 0x2e, 0x68, 0xe4, 0x73, 0x15, 0x5a,
      0x15, 0xa7, 0xda, 0xee, 0xd4, 0xa1, 0x0f, 0x4e, 0x05, 0x13, 0x9c,
      0x23, 0xdf, 0x00, 0xb3, 0xaa, 0xdc, 0x71, 0xf0, 0x59, 0x6a};
  gsec_aead_malloc_test_vector(
      &test_vector_16, test_vector_16_key,
      sizeof(test_vector_16_key) / sizeof(uint8_t), test_vector_16_nonce,
      sizeof(test_vector_16_nonce) / sizeof(uint8_t), test_vector_16_aad,
      sizeof(test_vector_16_aad) / sizeof(uint8_t), test_vector_16_plaintext,
      sizeof(test_vector_16_plaintext) / sizeof(uint8_t),
      test_vector_16_ciphertext_and_tag,
      sizeof(test_vector_16_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_16);
  gsec_aead_free_test_vector(test_vector_16);

  // 2.7.1 79-byte crypt
  gsec_aead_test_vector* test_vector_17;
  const uint8_t test_vector_17_key[] = {0x88, 0xee, 0x08, 0x7f, 0xd9, 0x5d,
                                        0xa9, 0xfb, 0xf6, 0x72, 0x5a, 0xa9,
                                        0xd7, 0x57, 0xb0, 0xcd};
  const uint8_t test_vector_17_nonce[] = {0x7a, 0xe8, 0xe2, 0xca, 0x4e, 0xc5,
                                          0x00, 0x01, 0x2e, 0x58, 0x49, 0x5c};
  const uint8_t test_vector_17_aad[] = {
      0x68, 0xf2, 0xe7, 0x76, 0x96, 0xce, 0x7a, 0xe8, 0xe2, 0xca, 0x4e,
      0xc5, 0x88, 0xe5, 0x41, 0x00, 0x2e, 0x58, 0x49, 0x5c, 0x08, 0x00,
      0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
      0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a,
      0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45,
      0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x00, 0x07};
  const uint8_t test_vector_17_plaintext[1] = {};
  const uint8_t test_vector_17_ciphertext_and_tag[] = {
      0x07, 0x92, 0x2b, 0x8e, 0xbc, 0xf1, 0x0b, 0xb2,
      0x29, 0x75, 0x88, 0xca, 0x4c, 0x61, 0x45, 0x23};
  gsec_aead_malloc_test_vector(
      &test_vector_17, test_vector_17_key,
      sizeof(test_vector_17_key) / sizeof(uint8_t), test_vector_17_nonce,
      sizeof(test_vector_17_nonce) / sizeof(uint8_t), test_vector_17_aad,
      sizeof(test_vector_17_aad) / sizeof(uint8_t), test_vector_17_plaintext, 0,
      test_vector_17_ciphertext_and_tag,
      sizeof(test_vector_17_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_17);
  gsec_aead_free_test_vector(test_vector_17);

  // 2.7.2 79-byte crypt
  gsec_aead_test_vector* test_vector_18;
  const uint8_t test_vector_18_key[] = {
      0x4c, 0x97, 0x3d, 0xbc, 0x73, 0x64, 0x62, 0x16, 0x74, 0xf8, 0xb5,
      0xb8, 0x9e, 0x5c, 0x15, 0x51, 0x1f, 0xce, 0xd9, 0x21, 0x64, 0x90,
      0xfb, 0x1c, 0x1a, 0x2c, 0xaa, 0x0f, 0xfe, 0x04, 0x07, 0xe5};
  const uint8_t test_vector_18_nonce[] = {0x7a, 0xe8, 0xe2, 0xca, 0x4e, 0xc5,
                                          0x00, 0x01, 0x2e, 0x58, 0x49, 0x5c};
  const uint8_t test_vector_18_aad[] = {
      0x68, 0xf2, 0xe7, 0x76, 0x96, 0xce, 0x7a, 0xe8, 0xe2, 0xca, 0x4e,
      0xc5, 0x88, 0xe5, 0x41, 0x00, 0x2e, 0x58, 0x49, 0x5c, 0x08, 0x00,
      0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
      0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24,
      0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
      0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a,
      0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45,
      0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x00, 0x07};
  const uint8_t test_vector_18_plaintext[1] = {};
  const uint8_t test_vector_18_ciphertext_and_tag[] = {
      0x00, 0xbd, 0xa1, 0xb7, 0xe8, 0x76, 0x08, 0xbc,
      0xbf, 0x47, 0x0f, 0x12, 0x15, 0x7f, 0x4c, 0x07};
  gsec_aead_malloc_test_vector(
      &test_vector_18, test_vector_18_key,
      sizeof(test_vector_18_key) / sizeof(uint8_t), test_vector_18_nonce,
      sizeof(test_vector_18_nonce) / sizeof(uint8_t), test_vector_18_aad,
      sizeof(test_vector_18_aad) / sizeof(uint8_t), test_vector_18_plaintext, 0,
      test_vector_18_ciphertext_and_tag,
      sizeof(test_vector_18_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_18);
  gsec_aead_free_test_vector(test_vector_18);

  // 2.8.1 61-byte crypt
  gsec_aead_test_vector* test_vector_19;
  const uint8_t test_vector_19_key[] = {0x88, 0xee, 0x08, 0x7f, 0xd9, 0x5d,
                                        0xa9, 0xfb, 0xf6, 0x72, 0x5a, 0xa9,
                                        0xd7, 0x57, 0xb0, 0xcd};
  const uint8_t test_vector_19_nonce[] = {0x7a, 0xe8, 0xe2, 0xca, 0x4e, 0xc5,
                                          0x00, 0x01, 0x2e, 0x58, 0x49, 0x5c};
  const uint8_t test_vector_19_aad[] = {
      0x68, 0xf2, 0xe7, 0x76, 0x96, 0xce, 0x7a, 0xe8, 0xe2, 0xca,
      0x4e, 0xc5, 0x88, 0xe5, 0x4d, 0x00, 0x2e, 0x58, 0x49, 0x5c};
  const uint8_t test_vector_19_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
      0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
      0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x00, 0x08};
  const uint8_t test_vector_19_ciphertext_and_tag[] = {
      0xc3, 0x1f, 0x53, 0xd9, 0x9e, 0x56, 0x87, 0xf7, 0x36, 0x51, 0x19, 0xb8,
      0x32, 0xd2, 0xaa, 0xe7, 0x07, 0x41, 0xd5, 0x93, 0xf1, 0xf9, 0xe2, 0xab,
      0x34, 0x55, 0x77, 0x9b, 0x07, 0x8e, 0xb8, 0xfe, 0xac, 0xdf, 0xec, 0x1f,
      0x8e, 0x3e, 0x52, 0x77, 0xf8, 0x18, 0x0b, 0x43, 0x36, 0x1f, 0x65, 0x12,
      0xad, 0xb1, 0x6d, 0x2e, 0x38, 0x54, 0x8a, 0x2c, 0x71, 0x9d, 0xba, 0x72,
      0x28, 0xd8, 0x40, 0x88, 0xf8, 0x75, 0x7a, 0xdb, 0x8a, 0xa7, 0x88, 0xd8,
      0xf6, 0x5a, 0xd6, 0x68, 0xbe, 0x70, 0xe7};
  gsec_aead_malloc_test_vector(
      &test_vector_19, test_vector_19_key,
      sizeof(test_vector_19_key) / sizeof(uint8_t), test_vector_19_nonce,
      sizeof(test_vector_19_nonce) / sizeof(uint8_t), test_vector_19_aad,
      sizeof(test_vector_19_aad) / sizeof(uint8_t), test_vector_19_plaintext,
      sizeof(test_vector_19_plaintext) / sizeof(uint8_t),
      test_vector_19_ciphertext_and_tag,
      sizeof(test_vector_19_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_19);
  gsec_aead_free_test_vector(test_vector_19);

  // 2.8.2 61-byte crypt
  gsec_aead_test_vector* test_vector_20;
  const uint8_t test_vector_20_key[] = {
      0x4c, 0x97, 0x3d, 0xbc, 0x73, 0x64, 0x62, 0x16, 0x74, 0xf8, 0xb5,
      0xb8, 0x9e, 0x5c, 0x15, 0x51, 0x1f, 0xce, 0xd9, 0x21, 0x64, 0x90,
      0xfb, 0x1c, 0x1a, 0x2c, 0xaa, 0x0f, 0xfe, 0x04, 0x07, 0xe5};
  const uint8_t test_vector_20_nonce[] = {0x7a, 0xe8, 0xe2, 0xca, 0x4e, 0xc5,
                                          0x00, 0x01, 0x2e, 0x58, 0x49, 0x5c};
  const uint8_t test_vector_20_aad[] = {
      0x68, 0xf2, 0xe7, 0x76, 0x96, 0xce, 0x7a, 0xe8, 0xe2, 0xca,
      0x4e, 0xc5, 0x88, 0xe5, 0x4d, 0x00, 0x2e, 0x58, 0x49, 0x5c};
  const uint8_t test_vector_20_plaintext[] = {
      0x08, 0x00, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
      0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22,
      0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
      0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
      0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43,
      0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x00, 0x08};
  const uint8_t test_vector_20_ciphertext_and_tag[] = {
      0xba, 0x8a, 0xe3, 0x1b, 0xc5, 0x06, 0x48, 0x6d, 0x68, 0x73, 0xe4, 0xfc,
      0xe4, 0x60, 0xe7, 0xdc, 0x57, 0x59, 0x1f, 0xf0, 0x06, 0x11, 0xf3, 0x1c,
      0x38, 0x34, 0xfe, 0x1c, 0x04, 0xad, 0x80, 0xb6, 0x68, 0x03, 0xaf, 0xcf,
      0x5b, 0x27, 0xe6, 0x33, 0x3f, 0xa6, 0x7c, 0x99, 0xda, 0x47, 0xc2, 0xf0,
      0xce, 0xd6, 0x8d, 0x53, 0x1b, 0xd7, 0x41, 0xa9, 0x43, 0xcf, 0xf7, 0xa6,
      0x71, 0x3b, 0xd0, 0x26, 0x11, 0xcd, 0x7d, 0xaa, 0x01, 0xd6, 0x1c, 0x5c,
      0x88, 0x6d, 0xc1, 0xa8, 0x17, 0x01, 0x07};
  gsec_aead_malloc_test_vector(
      &test_vector_20, test_vector_20_key,
      sizeof(test_vector_20_key) / sizeof(uint8_t), test_vector_20_nonce,
      sizeof(test_vector_20_nonce) / sizeof(uint8_t), test_vector_20_aad,
      sizeof(test_vector_20_aad) / sizeof(uint8_t), test_vector_20_plaintext,
      sizeof(test_vector_20_plaintext) / sizeof(uint8_t),
      test_vector_20_ciphertext_and_tag,
      sizeof(test_vector_20_ciphertext_and_tag) / sizeof(uint8_t));
  gsec_test_verify_crypter_on_test_vector(test_vector_20);
  gsec_aead_free_test_vector(test_vector_20);
}

int main(int argc, char** argv) {
  grpc::testing::TestEnvironment env(&argc, argv);
  ::testing::InitGoogleTest(&argc, argv);
  grpc::testing::TestGrpcScope grpc_scope;
  return RUN_ALL_TESTS();
}
