/*
 * SPI device debug utility (using spidev driver)
 *
 * Copyright (C) 2016 Renesas Electronics Europe GmbH
 * Copyright (C) 2016 Renesas Electronics Corporation
 * Copyright (C) 2016 Cogent Embedded, Inc.
 *
 * based on:
 * Copyright (c) 2007  MontaVista Software, Inc.
 * Copyright (c) 2007  Anton Vorontsov <avorontsov@ru.mvista.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 */

#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>

#define ARRAY_SIZE(a) (sizeof(a) / sizeof((a)[0]))

static void pabort(const char *s)
{
	perror(s);
	abort();
}

static const char *device = "/dev/spidev2.0";
static uint8_t mode;
static uint8_t bits = 8;
static uint32_t speed = 6000000;
static uint16_t delay;
static unsigned long long reg_addr;
static unsigned long long reg_val;

enum Option {
	OptGetRegister = 'g',
	OptSetRegister = 's',
	OptSetDevice = 'd',
//	OptGetDriverInfo = 'D',
//<------>OptHelp = 'h',

	OptVerbose,
	OptLast = 256
};

static char options[OptLast];

static __u32 spidev_read(int fd, int addr)
{
	struct spi_ioc_transfer xfer[1];
	unsigned char buf[32], *bp;
	int status;
	int len = 8;

	memset(xfer, 0, sizeof xfer);
	memset(buf, 0, sizeof buf);

	buf[0] = (addr >> 16) & 0xff;
	buf[1] = (addr >> 8) & 0xff;
	buf[2] = (addr) & 0xff;
	buf[3] = 0; // READ

	xfer[0].tx_buf = (unsigned long)buf;
	xfer[0].len = len;

	status = ioctl(fd, SPI_IOC_MESSAGE(1), xfer);
	if (status < 0) {
		perror("SPI_IOC_MESSAGE");
		return -1;
	}

	xfer[0].rx_buf = (unsigned long)buf;
	xfer[0].len = len;

	status = ioctl(fd, SPI_IOC_MESSAGE(1), xfer);
	if (status < 0) {
		perror("SPI_IOC_MESSAGE");
		return -1;
	}

	return ((buf[4] << 24) | (buf[5] << 16) | (buf[6] << 8) | buf[7]);
}

static void spidev_write(int fd, int addr, int val)
{
	struct spi_ioc_transfer xfer[1];
	unsigned char buf[32], *bp;
	int status;
	int len = 8;

	memset(xfer, 0, sizeof xfer);
	memset(buf, 0, sizeof buf);

	buf[0] = (addr >> 16) & 0xff;
	buf[1] = (addr >> 8) & 0xff;
	buf[2] = (addr) & 0xff;
	buf[3] = 0x80; // WRITE
	buf[4] = (val >> 24) & 0xff;
	buf[5] = (val >> 16) & 0xff;
	buf[6] = (val >> 8) & 0xff;
	buf[7] = val & 0xff;

	xfer[0].tx_buf = (unsigned long)buf;
	xfer[0].len = len;

	status = ioctl(fd, SPI_IOC_MESSAGE(1), xfer);
	if (status < 0) {
		perror("SPI_IOC_MESSAGE");
		return;
	}
}

static const char *binary(unsigned long long val)
{
	static char bin[80];
	char *p = bin;
	int i, j;
	int bits = 64;

	if ((val & 0xffffffff00000000LL) == 0) {
		if ((val & 0xffff0000) == 0) {
			if ((val & 0xff00) == 0)
				bits = 8;
			else
				bits= 16;
		} else
			bits = 32;
	}

	for (i = bits - 1; i >= 0; i -= 8) {
		for (j = i; j >= i - 7; j--) {
			if (val & (1LL << j))
				*p++ = '1';
			else
				*p++ = '0';
		}
		*p++ = ' ';
	}
	p[-1] = 0;
	return bin;
}

static void print_usage(const char *prog)
{
	printf("Usage: %s [-DSbdlHOLC3]\n", prog);
	puts("  -d --device   device to use (default /dev/spidev2.0)\n"
	     "  -S --speed    max speed (Hz)\n"
	     "  -d --delay    delay (usec)\n"
	     "  -b --bpw      bits per word \n"
	     "  -l --loop     loopback\n"
	     "  -H --cpha     clock phase\n"
	     "  -O --cpol     clock polarity\n"
	     "  -L --lsb      least significant bit first\n"
	     "  -C --cs-high  chip select active high\n"
	     "  -3 --3wire    SI/SO signals shared\n"
	     "  -g --get-register=<addr> Get the addressed register\n"
	     "  -s --set-register=<addr> Set the addressed register with the commandline arguments\n"
	    );
	exit(1);
}

static void parse_opts(int argc, char *argv[])
{
	while (1) {
		static const struct option lopts[] = {
			{ "device", required_argument, 0, OptSetDevice },
			{ "speed", required_argument, 0, 'S' },
			{ "delay", required_argument, 0, 'D' },
			{ "bpw", required_argument, 0, 'b' },
			{ "loop", 0, 0, 'l' },
			{ "cpha", 0, 0, 'H' },
			{ "cpol", 0, 0, 'O' },
			{ "lsb", 0, 0, 'L' },
			{ "cs-high", 0, 0, 'C' },
			{ "3wire", 0, 0, '3' },
			{ "no-cs", 0, 0, 'N' },
			{ "ready", 0, 0, 'R' },
			{ "get-register", required_argument, 0, OptGetRegister},
			{ "set-register", required_argument, 0, OptSetRegister},
			{ NULL, 0, 0, 0 },
		};
		int c;

		c = getopt_long(argc, argv, "D:S:d:b:lHOLC3NRg:s:", lopts, NULL);

		if (c == -1)
			break;

		options[(int)c] = 1;

		switch (c) {
		case OptSetDevice:
			device = optarg;
			break;
		case 'S':
			speed = atoi(optarg);
			break;
		case 'D':
			delay = atoi(optarg);
			break;
		case 'b':
			bits = atoi(optarg);
			break;
		case 'l':
			mode |= SPI_LOOP;
			break;
		case 'H':
			mode |= SPI_CPHA;
			break;
		case 'O':
			mode |= SPI_CPOL;
			break;
		case 'L':
			mode |= SPI_LSB_FIRST;
			break;
		case 'C':
			mode |= SPI_CS_HIGH;
			break;
		case '3':
			mode |= SPI_3WIRE;
			break;
		case 'N':
			mode |= SPI_NO_CS;
			break;
		case 'R':
			mode |= SPI_READY;
			break;
		case OptGetRegister:
			reg_addr = strtoull(optarg, NULL, 0);
			break;
		case OptSetRegister:
			reg_addr = strtoull(optarg, NULL, 0);
			if (optind >= argc)
				print_usage(argv[0]);

			reg_addr = strtoull(optarg, NULL, 0);
			reg_val = strtoull(argv[optind], NULL, 0);
			break;
		default:
			print_usage(argv[0]);
			break;
		}
	}
}

int main(int argc, char *argv[])
{
	int ret = 0;
	int fd;

	parse_opts(argc, argv);

	fd = open(device, O_RDWR);
	if (fd < 0)
		pabort("can't open device");

	/*
	 * spi mode
	 */
	ret = ioctl(fd, SPI_IOC_WR_MODE, &mode);
	if (ret == -1)
		pabort("can't set spi mode");

	ret = ioctl(fd, SPI_IOC_RD_MODE, &mode);
	if (ret == -1)
		pabort("can't get spi mode");

	/*
	 * bits per word
	 */
	ret = ioctl(fd, SPI_IOC_WR_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't set bits per word");

	ret = ioctl(fd, SPI_IOC_RD_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't get bits per word");

	/*
	 * max speed hz
	 */
	ret = ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't set max speed hz");

	ret = ioctl(fd, SPI_IOC_RD_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't get max speed hz");

#if 0
	printf("spi mode: %d\n", mode);
	printf("bits per word: %d\n", bits);
	printf("max speed: %d Hz (%d KHz)\n", speed, speed/1000);
#endif

	if (options[OptGetRegister]) {
		reg_val = spidev_read(fd, reg_addr);
		printf("Register 0x%08llx", reg_addr);
		printf(" = %llxh (%lldd  %sb)\n", reg_val, reg_val, binary(reg_val));
	}

	if (options[OptSetRegister]) {
		spidev_write(fd, reg_addr, reg_val);
		printf("Register 0x%08llx", reg_addr);
		printf(" set to 0x%llx\n", reg_val);
	}

	close(fd);

	return ret;
}
