/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * qmicli -- Command line interface to control QMI devices
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (C) 2023 Dylan Van Assche <me@dylanvanassche.be>
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <locale.h>
#include <string.h>
#include <assert.h>

#include <glib.h>
#include <gio/gio.h>

#include <libqmi-glib.h>

#include "qmicli.h"
#include "qmicli-helpers.h"

#if defined HAVE_QMI_SERVICE_IMSP

/* Context */
typedef struct {
    QmiDevice *device;
    QmiClientImsp *client;
    GCancellable *cancellable;
} Context;
static Context *ctx;

/* Options */
static gboolean get_enabler_state_flag;
static gboolean noop_flag;

static GOptionEntry entries[] = {
#if defined HAVE_QMI_MESSAGE_IMSP_GET_ENABLER_STATE
    { "imsp-get-enabler-state", 0, 0, G_OPTION_ARG_NONE, &get_enabler_state_flag,
      "Get IMSP enabler state",
      NULL
    },
#endif
    { "imsp-noop", 0, 0, G_OPTION_ARG_NONE, &noop_flag,
      "Just allocate or release a IMSP client. Use with `--client-no-release-cid' and/or `--client-cid'",
      NULL
    },
    { NULL }
};

GOptionGroup *
qmicli_imsp_get_option_group (void)
{
    GOptionGroup *group;

    group = g_option_group_new ("imsp",
                                "IMSP options:",
                                "Show IP Multimedia Subsystem Presence Service options",
                                NULL,
                                NULL);
    g_option_group_add_entries (group, entries);

    return group;
}

gboolean
qmicli_imsp_options_enabled (void)
{
    static guint n_actions = 0;
    static gboolean checked = FALSE;

    if (checked)
        return !!n_actions;

    n_actions = (get_enabler_state_flag +
                 noop_flag);

    if (n_actions > 1) {
        g_printerr ("error: too many IMSP actions requested\n");
        exit (EXIT_FAILURE);
    }

    checked = TRUE;
    return !!n_actions;
}

static void
context_free (Context *context)
{
    if (!context)
        return;

    if (context->cancellable)
        g_object_unref (context->cancellable);
    if (context->device)
        g_object_unref (context->device);
    if (context->client)
        g_object_unref (context->client);
    g_slice_free (Context, context);
}

static void
operation_shutdown (gboolean operation_status)
{
    /* Cleanup context and finish async operation */
    context_free (ctx);
    qmicli_async_operation_done (operation_status, FALSE);
}

#if defined HAVE_QMI_MESSAGE_IMSP_GET_ENABLER_STATE

static void
get_enabler_state_ready (QmiClientImsp *client,
                         GAsyncResult *res)
{
    QmiMessageImspGetEnablerStateOutput *output;
    QmiImspEnablerState enabler_state;
    GError *error = NULL;

    output = qmi_client_imsp_get_enabler_state_finish (client, res, &error);
    if (!output) {
        g_printerr ("error: operation failed: %s\n", error->message);
        g_error_free (error);
        operation_shutdown (FALSE);
        return;
    }

    if (!qmi_message_imsp_get_enabler_state_output_get_result (output, &error)) {
        g_printerr ("error: couldn't get enabler state: %s\n", error->message);
        g_error_free (error);
        qmi_message_imsp_get_enabler_state_output_unref (output);
        operation_shutdown (FALSE);
        return;
    }

    g_print ("[%s] IMSP enabler state retrieved:\n", qmi_device_get_path_display (ctx->device));

    if (qmi_message_imsp_get_enabler_state_output_get_enabler_state (output, &enabler_state, NULL)) {
        g_print ("\tRegistration status: '%s'\n",
                 qmi_imsp_enabler_state_get_string (enabler_state));
    }

    qmi_message_imsp_get_enabler_state_output_unref (output);
    operation_shutdown (TRUE);
}

#endif /* HAVE_QMI_MESSAGE_IMSP_GET_ENABLER_STATE */

static gboolean
noop_cb (gpointer unused)
{
    operation_shutdown (TRUE);
    return FALSE;
}

void
qmicli_imsp_run (QmiDevice *device,
                 QmiClientImsp *client,
                 GCancellable *cancellable)
{
    /* Initialize context */
    ctx = g_slice_new (Context);
    ctx->device = g_object_ref (device);
    ctx->client = g_object_ref (client);
    ctx->cancellable = g_object_ref (cancellable);

#if defined HAVE_QMI_MESSAGE_IMSP_GET_ENABLER_STATE
    if (get_enabler_state_flag) {
        g_debug ("Asynchronously getting enabler state...");

        qmi_client_imsp_get_enabler_state (ctx->client,
                                           NULL,
                                           10,
                                           ctx->cancellable,
                                           (GAsyncReadyCallback)get_enabler_state_ready,
                                           NULL);
        return;
    }
#endif /* HAVE_QMI_MESSAGE_IMSP_GET_ENABLER_STATE */

    /* Just client allocate/release? */
    if (noop_flag) {
        g_idle_add (noop_cb, NULL);
        return;
    }

    g_warn_if_reached ();
}

#endif /* HAVE_QMI_SERVICE_IMSP */

