/*
 * Copyright 2020-2024 Toyota Connected North America
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "shape_system.h"

#include <core/components/derived/collidable.h>

#include <core/entity/derived/shapes/baseshape.h>
#include <core/entity/derived/shapes/cube.h>
#include <core/entity/derived/shapes/plane.h>
#include <core/entity/derived/shapes/sphere.h>
#include <core/systems/ecs.h>
#include <plugins/common/common.h>

#include "collision_system.h"

namespace plugin_filament_view {

using shapes::BaseShape;

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::vToggleAllShapesInScene(const bool enable) const {
  if (enable) {
    for (const auto& guid : _shapes) {
      const auto shape = getShape(guid);
      shape->vAddEntityToScene();
    }
  } else {
    for (const auto& guid : _shapes) {
      const auto shape = getShape(guid);
      shape->vRemoveEntityFromScene();
    }
  }
}

bool ShapeSystem::hasShape(const EntityGUID guid) const {
  if (std::find(_shapes.begin(), _shapes.end(), guid) != _shapes.end()) {
    return true;
  } else {
    return false;
  }
}

BaseShape* ShapeSystem::getShape(const EntityGUID guid) const {
  if (hasShape(guid)) {
    return dynamic_cast<BaseShape*>(ecs->getEntity(guid).get());
  }
  return nullptr;
}

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::vToggleSingleShapeInScene(const EntityGUID guid, const bool enable) const {
  if (!hasShape(guid)) {
    return;
  }

  BaseShape* shape = getShape(guid);
  if (enable) {
    shape->vAddEntityToScene();
  } else {
    shape->vRemoveEntityFromScene();
  }
}

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::vRemoveAllShapesInScene() {
  vToggleAllShapesInScene(false);

  for (const auto& guid : _shapes) {
    ecs->removeEntity(guid);
  }

  _shapes.clear();
}

////////////////////////////////////////////////////////////////////////////////////
std::unique_ptr<BaseShape> ShapeSystem::poDeserializeShapeFromData(
  const flutter::EncodableMap& mapData
) {
  ShapeType type;

  // Find the "shapeType" key in the mapData
  if (const auto it = mapData.find(flutter::EncodableValue("shapeType"));
      it != mapData.end() && std::holds_alternative<int32_t>(it->second)) {
    // Check if the value is within the valid range of the ShapeType enum
    if (int32_t typeValue = std::get<int32_t>(it->second);
        typeValue > static_cast<int32_t>(ShapeType::Unset)
        && typeValue < static_cast<int32_t>(ShapeType::Max)) {
      type = static_cast<ShapeType>(typeValue);
    } else {
      spdlog::error("Invalid shape type value: {}", typeValue);
      return nullptr;
    }
  } else {
    spdlog::error("shapeType not found or is of incorrect type");
    return nullptr;
  }

  // Based on the type_, create the corresponding shape
  switch (type) {
    case ShapeType::Plane: {
      auto toReturn = std::make_unique<shapes::Plane>();
      toReturn->deserializeFrom(mapData);
      return toReturn;
    }
    case ShapeType::Cube: {
      auto toReturn = std::make_unique<shapes::Cube>();
      toReturn->deserializeFrom(mapData);
      return toReturn;
    }
    case ShapeType::Sphere: {
      auto toReturn = std::make_unique<shapes::Sphere>();
      toReturn->deserializeFrom(mapData);
      return toReturn;
    }
    default:
      // Handle unknown shape type
      spdlog::error("Unknown shape type: {}", static_cast<int32_t>(type));
      return nullptr;
  }
}

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::addShapesToScene(std::vector<std::shared_ptr<BaseShape>>* shapes) {
  SPDLOG_TRACE("++{}", __FUNCTION__);

  // TODO remove this, just debug info print for now;
  /*for (auto& shape : *shapes) {
    shape->DebugPrint("Add shapes to scene");
  }*/

  for (auto& shape : *shapes) {
    addShapeToScene(shape);
  }

  SPDLOG_TRACE("--{}", __FUNCTION__);
}

void ShapeSystem::addShapeToScene(const std::shared_ptr<shapes::BaseShape>& shape) {
  runtime_assert(shape != nullptr, "ShapeSystem::addShapeToScene: shape cannot be null");

  filament::Scene* filamentScene = _filament->getFilamentScene();

  spdlog::trace("addShapesToScene: {}", shape->GetGuid());
  FilamentEntity oEntity = _em->create();
  filamentScene->addEntity(oEntity);
  shape->_fEntity = oEntity;

  shape->bInitAndCreateShape(_engine, oEntity);

  spdlog::trace("Adding entity {} with filament entity {}", shape->GetGuid(), oEntity.getId());

  // To investigate a better system for implementing layer mask
  // across dart to here.
  // auto instance = _rcm.getInstance(*oEntity.get());
  // To investigate
  // _rcm.setLayerMask(instance, 0xff, 0x00);

  _shapes.emplace_back(shape->GetGuid());
}

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::vOnInitSystem() {
  // Get filament
  _filament = ecs->getSystem<FilamentSystem>(__FUNCTION__);
  runtime_assert(_filament != nullptr, "ModelSystem::vOnInitSystem: FilamentSystem not init yet");

  _engine = _filament->getFilamentEngine();
  runtime_assert(_engine != nullptr, "ModelSystem::vOnInitSystem: FilamentEngine not found");

  _rcm = _engine->getRenderableManager();
  _tm = _engine->getTransformManager();
  _em = _engine->getEntityManager();
  runtime_assert(_rcm != nullptr, "ModelSystem::vOnInitSystem: RenderableManager not found");
  runtime_assert(_tm != nullptr, "ModelSystem::vOnInitSystem: TransformManager not found");
  runtime_assert(_em != nullptr, "ModelSystem::vOnInitSystem: EntityManager not found");

  /*
   * Register message handlers
   */
  vRegisterMessageHandler(ECSMessageType::ToggleShapesInScene, [this](const ECSMessage& msg) {
    spdlog::debug("ToggleShapesInScene");

    const auto value = msg.getData<bool>(ECSMessageType::ToggleShapesInScene);

    vToggleAllShapesInScene(value);

    spdlog::trace("ToggleShapesInScene Complete");
  });

  vRegisterMessageHandler(ECSMessageType::SetShapeTransform, [this](const ECSMessage& msg) {
    SPDLOG_TRACE("SetShapeTransform");

    const auto guid = msg.getData<EntityGUID>(ECSMessageType::SetShapeTransform);

    const auto position = msg.getData<filament::math::float3>(ECSMessageType::Position);
    const auto rotation = msg.getData<filament::math::quatf>(ECSMessageType::Rotation);
    const auto scale = msg.getData<filament::math::float3>(ECSMessageType::Scale);

    // find the entity in our list:
    if (hasShape(guid)) {
      const auto entity = getShape(guid);
      const auto baseTransform = entity->getComponent<BaseTransform>();
      const auto collidable = entity->getComponent<Collidable>();

      baseTransform->SetTransform(position, scale, rotation);
    }

    SPDLOG_TRACE("SetShapeTransform Complete");
  });

  vRegisterMessageHandler(ECSMessageType::ToggleVisualForEntity, [this](const ECSMessage& msg) {
    spdlog::debug("ToggleVisualForEntity");

    const auto guid = msg.getData<EntityGUID>(ECSMessageType::ToggleVisualForEntity);
    const auto value = msg.getData<bool>(ECSMessageType::BoolValue);

    vToggleSingleShapeInScene(guid, value);

    spdlog::trace("ToggleVisualForEntity Complete");
  });
}

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::vUpdate(float /*fElapsedTime*/) {}

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::vShutdownSystem() {
  // remove all filament entities.
  vRemoveAllShapesInScene();
}

////////////////////////////////////////////////////////////////////////////////////
void ShapeSystem::DebugPrint() { SPDLOG_DEBUG("{}", __FUNCTION__); }
}  // namespace plugin_filament_view
